# Copyright 2024, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

import logging

import numpy as np

from vc_calc.vc_data import MeasureData
from vc_calc.vc_calc import VcCalc


def main():
    # ====== logger 設定 =========
    # ロガーを作成
    logger = logging.getLogger(__name__)
    # ロガーのレベルを設定
    logger.setLevel(logging.DEBUG)
    # ハンドラを作成
    console_handler = logging.StreamHandler()
    # ハンドラのレベルを設定
    console_handler.setLevel(logging.DEBUG)
    # ハンドラをロガーに追加
    logger.addHandler(console_handler)

    # 演算ライブラリのインスタンス生成
    # パラメータエラーの場合、ValueErrorを返すのでチャッチできるようにする。
    try:
        fft_size = 10000
        exec_size = 1000
        avr_size = 30
        vc_calc = VcCalc(fft_size, exec_size, avr_size)
    except ValueError as err:
        logger.error(f"vc_calc initialize error: {err}")
        exit(1)  # 処理を終了する。

    # テストデータ生成（sine wave, 10Hz, 1000sps)
    frq = 10
    amplitude = 1  # 振幅 [G]
    sampling_rate = 1000  # サンプリングレート [sps]
    duration = (
        int(fft_size / sampling_rate + (exec_size / sampling_rate) * avr_size) + 1
    )  # 生成する時間 [s]
    samples = sampling_rate * duration  # サンプル数
    time = np.linspace(0, duration, samples)  # 時間軸
    sin_wave = [np.sin(2 * np.pi * frq * t) * amplitude for t in time]  # sin波のリスト

    # テストデータでVC演算
    for data in sin_wave:
        input = MeasureData(x=data, y=data, z=data)
        result_sgl, result_avr = vc_calc.calc(input)

        # single data 演算完了
        if result_sgl.done:
            print("single data calculation done")
            # 処理

        # average data 演算完了
        if result_avr.done:
            print("average data calculation done")
            # 処理


if __name__ == "__main__":
    main()
