# Copyright 2025, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

"""
VcCalcApp の設定項目に関するカスタマイズ実装

Custom implementation for VcCalcApp settings
"""

from dataclasses import asdict, dataclass
from typing import Any

from logger.core import Config, ConfigOverrideOption, Configurator
from logger.utils.env import get_bool, get_int
from logger.utils.validator import Validator

from .constants import ConfigKeys, Constants, InfoKeys


@dataclass
class VcConfig(Config):
    """
    VcCalcApp で追加した設定項目を扱うクラス

    Class to handle additional settings for VcCalcApp
    """

    vc_fft_size: int
    vc_exe_size: int
    vc_avg_size: int
    vc_log_single: bool
    vc_log_average: bool
    vc_output_raw: bool
    vc_control_gpio: bool

    def add_info(self, info_d: dict[str, Any]) -> None:
        """
        計測情報ファイルに出力する項目を登録

        Register items to output to the measurement information file
        """

        # 上位クラスの実装を実行
        # Execute the implementation of the superclass
        super().add_info(info_d)

        # 追加出力項目を登録
        # Register additional output items
        info_d.update(
            {
                InfoKeys.VC_FFT_SIZE: self.vc_fft_size,
                InfoKeys.VC_EXE_SIZE: self.vc_exe_size,
                InfoKeys.VC_AVG_SIZE: self.vc_avg_size,
                InfoKeys.VC_LOG_SINGLE: self.vc_log_single,
                InfoKeys.VC_LOG_AVERAGE: self.vc_log_average,
                InfoKeys.VC_OUTPUT_RAW: self.vc_output_raw,
                InfoKeys.VC_CONTROL_GPIO: self.vc_control_gpio,
            }
        )


class VcConfigurator(Configurator):
    """
    VcCalcApp の設定項目を構築するクラス

    Class to construct configuration items for VcCalcApp
    """

    def get_config(self, option: ConfigOverrideOption | None = None) -> Config | None:
        """
        環境変数(設定ファイル)から設定値を読み込み設定オブジェクトを返す

        Read configuration values from environment variables (configuration file) and
        return the configuration object
        """

        # 上位クラスの実装を実行
        # Execute the implementation of the superclass
        base_config = super().get_config(option)

        # 結果に関わらず、このクラスの設定項目を validate する
        # Validate the configuration items of this class regardless of the result
        v = Validator()
        results: list[bool] = []

        # A352_SPS - VC判定における制約を追加
        # A352_SPS - Add constraints for VC determination
        a352_sps = get_int(ConfigKeys.A352_SPS)
        results.append(
            v.one_of(a352_sps, ConfigKeys.A352_SPS, Constants.A352_SPSS_LIMIT)
        )

        # VC_FFT_SIZE
        vc_fft_size = get_int(ConfigKeys.VC_FFT_SIZE)
        results.append(
            v.between(
                vc_fft_size,
                ConfigKeys.VC_FFT_SIZE,
                Constants.VC_FFT_SIZE_MIN,
                Constants.VC_FFT_SIZE_MAX,
            )
        )

        # VC_EXE_SIZE
        vc_exe_size = get_int(ConfigKeys.VC_EXE_SIZE)
        results.append(
            v.constraint(
                vc_exe_size,
                ConfigKeys.VC_EXE_SIZE,
                (
                    False
                    if vc_fft_size is None or vc_exe_size is None or vc_exe_size == 0
                    else vc_fft_size % vc_exe_size == 0
                ),
                "VC_FFT_SIZE % VC_EXE_SIZE = 0",
            )
        )

        # VC_AVG_SIZE
        vc_avg_size = get_int(ConfigKeys.VC_AVG_SIZE)
        results.append(
            v.between(
                vc_avg_size,
                ConfigKeys.VC_AVG_SIZE,
                Constants.VC_AVG_SIZE_MIN,
                Constants.VC_AVG_SIZE_MAX,
            )
        )

        # VC_LOG_SINGLE
        vc_log_single = get_bool(ConfigKeys.VC_LOG_SINGLE)
        results.append(v.is_bool(vc_log_single, ConfigKeys.VC_LOG_SINGLE))

        # VC_LOG_AVERAGE
        vc_log_average = get_bool(ConfigKeys.VC_LOG_AVERAGE)
        results.append(v.is_bool(vc_log_average, ConfigKeys.VC_LOG_AVERAGE))

        # VC_OUTPUT_RAW
        vc_output_raw = get_bool(ConfigKeys.VC_OUTPUT_RAW)
        results.append(v.is_bool(vc_output_raw, ConfigKeys.VC_OUTPUT_RAW))

        # VC_CONTROL_GPIO
        vc_control_gpio = get_bool(ConfigKeys.VC_CONTROL_GPIO)
        results.append(v.is_bool(vc_control_gpio, ConfigKeys.VC_CONTROL_GPIO))

        # 結果判定： いずれか失敗している場合は失敗
        # Result determination: Fail if any of them fail
        if base_config is None or not all(results):
            return None

        # 設定オブジェクト構築・返却
        # - 上位オブジェクトを展開して新たに構築する
        # Construct and return the configuration object
        # - Unpack the upper object and construct a new one
        return VcConfig(
            **asdict(base_config),
            vc_fft_size=vc_fft_size,  # type: ignore
            vc_exe_size=vc_exe_size,  # type: ignore
            vc_avg_size=vc_avg_size,  # type: ignore
            vc_log_single=vc_log_single,  # type: ignore
            vc_log_average=vc_log_average,  # type: ignore
            vc_output_raw=vc_output_raw,  # type: ignore
            vc_control_gpio=vc_control_gpio,  # type: ignore
        )
