# Copyright 2025, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

"""
VcCalcApp のファクトリ

Factory for VcCalcApp
"""

import logging
import typing

from logger.core import Configurator, LoggerError
from logger.core.factory import LoggerFactoryImpl
from logger.measure import Sensor

from .config import VcConfig, VcConfigurator
from .measure import VcA352


class VcLoggerFactory(LoggerFactoryImpl):
    """
    VcCalcApp 対応 LoggerFactory

    LoggerFactory for VcCalcApp
    """

    def create_configurator(self) -> Configurator:
        return VcConfigurator()

    def create_A342(self, port: str, product_id: str) -> Sensor:
        # A342 は動作対象外のため、実行時例外で落とす
        # A342 is not supported, so raise a runtime exception
        msg = "VC Calculation does not support A342"
        logging.getLogger(__name__).error(msg)
        raise LoggerError(msg)

    def create_A352(self, port: str, product_id: str) -> Sensor:
        # VcConfig の属性にアクセスが必要なため Config をダウンキャストする
        # Downcast Config to access VcConfig attributes
        vcconfig: VcConfig = typing.cast(VcConfig, self.config)

        return VcA352(
            port=port,
            baud=self.config.baud_rate,
            product_id=product_id,
            sps=self.config.A352_sps,
            logger_id=self.config.logger_id,
            file_rotate_min=self.config.file_rotate_min,
            sensor_data_diag=self.config.sensor_data_diag,
            sensor_data_diag_sec=self.config.sensor_data_diag_sec,
            vc_fft_size=vcconfig.vc_fft_size,
            vc_exe_size=vcconfig.vc_exe_size,
            vc_avg_size=vcconfig.vc_avg_size,
            vc_log_single=vcconfig.vc_log_single,
            vc_log_average=vcconfig.vc_log_average,
            vc_output_raw=vcconfig.vc_output_raw,
            vc_control_gpio=vcconfig.vc_control_gpio,
        )
