# Copyright 2025, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

"""
VcCalcApp における GPIO 制御

GPIO control in VcCalcApp
"""

import logging

from gpiozero import LED  # type: ignore  # no typed version
from vc_calc.vc_constants import VC_LEVEL_TBL


class VcGPIO:
    """
    VcCalcApp 向けに GPIO リレー制御基板を操作するクラス

    使用するリレー制御基板: [BitTradeOne 社製 リレー制御拡張基板 ADRSRU4](https://bit-trade-one.co.jp/product/module/adrsru/)

    Class to operate the GPIO relay control board for VcCalcApp

    Relay control board used: BitTradeOne ADRSRU4
    """

    _R1 = LED(4)
    _R2 = LED(17)
    _R3 = LED(27)
    _R4 = LED(22)

    def __init__(self) -> None:
        self._current: LED | None = None
        self._logger = logging.getLogger(__name__)

        # 念のためにすべての LED を消灯
        # Turn off all LEDs just in case
        (led.off() for led in [self._R1, self._R2, self._R3, self._R4])

    def set_level(self, vc_level: str) -> None:
        """
        VC 判定レベルを設定し、GPIO を制御するメソッド

        Method to set the VC determination level and control the GPIO

        Args:
            vc_level (str): VC 判定レベル
                    VC determination level
                - 正しくないレベルを指定した場合、ログに ERROR を出力する
                    If an incorrect level is specified, log an ERROR
        """

        # レベルが正しいか
        # Check if the level is correct
        if vc_level not in VC_LEVEL_TBL:
            self._logger.error(f"Invalid VC Level: {vc_level}")
            return

        # VC 判定レベル GPIO 設定
        # Set GPIO for VC determination level
        next_led: LED | None = None
        if vc_level == VC_LEVEL_TBL[0]:
            next_led = self._R1
        elif vc_level == VC_LEVEL_TBL[1]:
            next_led = self._R2
        elif vc_level == VC_LEVEL_TBL[2]:
            next_led = self._R3
        elif vc_level in VC_LEVEL_TBL[3:]:
            next_led = self._R4

        # - 同じレベルの場合はスキップする
        # - Skip if the level is the same
        if self._current == next_led:
            self._logger.debug("Same level, skip")
            return

        # - 現在の LED を消灯
        # - Turn off the current LED
        if self._current is not None:
            self._off()

        # - 次の LED を点灯
        # - Turn on the next LED
        if next_led is not None:
            self._on(next_led)

    def turn_off(self) -> None:
        """
        すべての表示を OFF にする
        Turn off all displays
        """
        self._off()

    def _on(self, next_led: LED) -> None:
        self._logger.debug(f"Turn on  LED: {next_led}")
        next_led.on()
        self._current = next_led

    def _off(self) -> None:
        if self._current is None:
            return
        self._logger.debug(f"Turn off LED: {self._current}")
        self._current.off()
        self._current = None


# VcGPIO テスト用 main
# Main for testing VcGPIO
if __name__ == "__main__":

    # DEBUG output
    from logger.core import StdoutLoggingService

    StdoutLoggingService().create_client().setup(level="DEBUG")

    # VcGPIO 生成
    # Create VcGPIO
    gpio = VcGPIO()

    # 入力ループ - Ctrl+D で終了
    # Input loop - End with Ctrl+D
    while True:
        try:
            level = input("Input VC Level > ")
            gpio.set_level(level)
        except Exception:
            print("Quit.")
            # 実際には以下を呼ばなくても、ライブラリ側で OFF にして終了する
            # In practice, the library will turn off and exit without calling the following
            gpio.turn_off()
            break
