# Copyright 2025, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

"""
VcCalcApp のカスタマイズ計測実装

Custom measurement implementation for VcCalcApp
"""

import csv
import datetime
import logging
from dataclasses import dataclass
from typing import TYPE_CHECKING

import numpy as np
import numpy.typing as npt
from logger.core import MessageService
from logger.measure import A352, MeasureData, WriterArgs
from vc_calc.vc_calc import DataType, OctBand, VcCalc, VcCalcResult
from vc_calc.vc_data import MeasureData as VcData

from .topic import VcTopic

# ** GPIOモジュールの初期化タイミングに影響するため type check ガードする
# ** Type check guard to affect the initialization timing of the GPIO module
if TYPE_CHECKING:
    from .gpio import VcGPIO


@dataclass
class VcWriterArgs(WriterArgs):
    """
    VcCalcApp 用 WriterArgs

    CSV の出力処理に加えて、以下を実施する：
    - VC 判定計算
    - VC 判定ログ出力
    - VC 判定 CSV 出力
    - VC 判定 GPIO 出力

    WriterArgs for VcCalcApp

    In addition to CSV output processing, the following are performed:
    - VC determination calculation
    - VC determination log output
    - VC determination CSV output
    - VC determination GPIO output
    """

    vc_fft_size: int
    vc_exe_size: int
    vc_avg_size: int
    vc_log_single: bool
    vc_log_average: bool
    vc_output_raw: bool
    vc_control_gpio: bool

    def __post_init__(self) -> None:
        # 上位クラス実装を実行
        # Execute the implementation of the superclass
        super().__post_init__()

        # 制御用属性を定義
        # Define control attributes
        self._vc_calc: VcCalc
        self._vc_file_sgl: str
        self._vc_file_avg: str
        self._vc_count_sgl: int = 0
        self._vc_count_avg: int = 0
        self._vc_gpio: VcGPIO | None = None
        self._logger = logging.getLogger(
            f"{__name__}.{self.model}.{self.serial}.vcwriter"
        )

    def write_start(self, output_dir: str) -> None:
        # 上位クラス実装を実行（CSV出力） - RAW データ出力時のみ
        # Execute the implementation of the superclass (CSV output) - Only for RAW data output
        if self.vc_output_raw:
            super().write_start(output_dir)

        # VC判定用出力準備
        # - VcCalc を初期化
        # Prepare output for VC determination
        # - Initialize VcCalc
        self._vc_calc = VcCalc(
            fft_size=self.vc_fft_size,
            exec_size=self.vc_exe_size,
            avr_size=self.vc_avg_size,
            oct_band=OctBand.Oct_1_3,
            data_type=DataType.Velocity,
        )

        # - ファイル名を設定
        # - Set file names
        self._vc_file_sgl = f"{output_dir}/{self.model}_{self.serial}_vccalc_sgl.csv"
        self._vc_file_avg = f"{output_dir}/{self.model}_{self.serial}_vccalc_avg.csv"
        self._logger.info(f"VcCalc Single  output to {self._vc_file_sgl}")
        self._logger.info(f"VcCalc Average output to {self._vc_file_avg}")

        # - ヘッダーを出力
        # ** 改行コードに \r が重ならないように個別指定が必要
        # - Output headers
        # ** Need to specify individually to avoid overlapping with newline code \r
        headers = self._csv_header(self._vc_calc.octave_band_tbl)
        with open(self._vc_file_sgl, mode="w", encoding="UTF-8", newline="\r\n") as f:
            csv.writer(f, lineterminator="\n").writerows(headers)
        with open(self._vc_file_avg, mode="w", encoding="UTF-8", newline="\r\n") as f:
            csv.writer(f, lineterminator="\n").writerows(headers)

        # GPIO 制御
        # ** モジュールの初期化タイミングにより問題が発生するため、ここで import する
        # GPIO control
        # ** Import here due to potential issues with module initialization timing
        if self.vc_control_gpio:
            try:
                from .gpio import VcGPIO

                self._vc_gpio = VcGPIO()

            except Exception as e:
                # ** 複数プロセスで初期化しようとすると競合するため、スキップする
                # ** Skip due to conflicts when initializing with multiple processes
                self._logger.warning(f"Failed to initialize GPIO with this sensor: {e}")

    def write_data(self, records: list[MeasureData]) -> None:
        # 上位クラス実装を実行（CSV出力） - RAW データ出力時のみ
        # Execute the implementation of the superclass (CSV output) - Only for RAW data output
        if self.vc_output_raw:
            super().write_data(records)

        # VC判定計算
        # VC determination calculation
        for data in records:
            res_sng, res_avg = self._vc_calc.calc(self._conv(data))

            # Single data 演算判定
            # Single data calculation determination
            if res_sng.done:
                if self.vc_log_single:
                    self._logger.info(f"Single  VC Level: {res_sng.vc_lvl}")

                # - VC CSV出力
                # - VC CSV output
                self._vc_count_sgl += 1
                with open(
                    self._vc_file_sgl, mode="a", encoding="UTF-8", newline="\r\n"
                ) as f:
                    csv.writer(f, lineterminator="\n").writerow(
                        self._csv_line(
                            self._vc_calc.octave_band_tbl, self._vc_count_sgl, res_sng
                        )
                    )

                # - VC GPIO 出力
                # - VC GPIO output
                if self._vc_gpio:
                    self._vc_gpio.set_level(res_sng.vc_lvl)

                # - VC メッセージ出力
                # - VC message output
                MessageService.send(
                    VcTopic.vc(self.model, self.serial), {"level": res_sng.vc_lvl}
                )
                MessageService.send(
                    VcTopic.fft(self.model, self.serial), {"value": res_sng.c}
                )

            # Average data 演算判定
            # Average data calculation determination
            if res_avg.done:
                if self.vc_log_average:
                    self._logger.info(f"Average VC Level: {res_avg.vc_lvl}")

                # - VC CSV出力
                # - VC CSV output
                self._vc_count_avg += 1
                with open(
                    self._vc_file_avg, mode="a", encoding="UTF-8", newline="\r\n"
                ) as f:
                    csv.writer(f, lineterminator="\n").writerow(
                        self._csv_line(
                            self._vc_calc.octave_band_tbl, self._vc_count_avg, res_avg
                        )
                    )

    def write_end(self) -> None:
        # 上位クラス実装を実行（CSV出力） - RAW データ出力時のみ
        # Execute the implementation of the superclass (CSV output) - Only for RAW data output
        if self.vc_output_raw:
            super().write_end()

        # VC GPIO 消灯
        # Turn off VC GPIO
        if self._vc_gpio:
            self._vc_gpio.turn_off()

    @staticmethod
    def _conv(data: MeasureData) -> VcData:
        return VcData(
            index=data.index,
            count=data.count,
            temperature=data.temperature,
            x=data.x,
            y=data.y,
            z=data.z,
            flag=data.flag,
        )

    @staticmethod
    def _csv_header(octave_band_tbl: npt.NDArray[np.float64]) -> list[list[str]]:
        """
        VC 判定用 CSV ヘッダーを生成します

        Generate CSV headers for VC determination

        Returns
            list[List[str]]: ヘッダー文字列、0:1行目、1:2行目
                    Header strings, 0: 1st row, 1: 2nd row
        """
        assert octave_band_tbl.shape[1] == 3

        # No, Time, VC-Level, {Peak V, Peak F, VC Freq*} * {C, X, Y, Z}
        # 1行目が大見出し、2行目が小見出しの2行構成
        # 1st row is the main heading, 2nd row is the subheading
        h1_left: list[str] = ["No.", "Time", "VC-Level"]
        h2_left: list[str] = [""] * 3
        h1_freq: list[str] = [""] * (len(octave_band_tbl) + 2 - 1)
        h2_freq: list[str] = ["Peak Velocity (mm/s)", "Peak Frequency (Hz)"] + [
            f"{v[0]:.2f} (Hz)" for v in octave_band_tbl
        ]

        h1 = (
            h1_left
            + (["Composite"] + h1_freq)
            + (["X Axis"] + h1_freq)
            + (["Y Axis"] + h1_freq)
            + (["Z Axis"] + h1_freq)
        )
        h2 = h2_left + h2_freq * 4

        return [h1, h2]

    @staticmethod
    def _csv_line(
        octave_band_tbl: npt.NDArray[np.float64], count: int, result: VcCalcResult
    ) -> list[str]:
        line = [
            str(count),
            format(datetime.datetime.now(), "%H:%M:%S"),
            result.vc_lvl,
        ]
        for dlst in [result.c, result.x, result.y, result.z]:
            argmax = np.argmax(dlst)
            line += [f"{dlst[argmax]:.6f}", f"{octave_band_tbl[argmax][0]:.2f}"]
            line += [f"{data:.6f}" for data in dlst]
        return line


class VcA352(A352):
    """
    VcCalcApp 対応 A352

    A352 for VcCalcApp
    """

    def __init__(
        self,
        port: str,
        baud: int,
        product_id: str,
        sps: int,
        logger_id: str,
        file_rotate_min: int,
        sensor_data_diag: bool,
        sensor_data_diag_sec: float,
        vc_fft_size: int,
        vc_exe_size: int,
        vc_avg_size: int,
        vc_log_single: bool,
        vc_log_average: bool,
        vc_output_raw: bool,
        vc_control_gpio: bool,
    ) -> None:
        super().__init__(
            port,
            baud,
            product_id,
            sps,
            logger_id,
            file_rotate_min,
            sensor_data_diag,
            sensor_data_diag_sec,
        )
        # VcWriterArgs に渡すために保持する
        # Keep to pass to VcWriterArgs
        self.vc_fft_size = vc_fft_size
        self.vc_exe_size = vc_exe_size
        self.vc_avg_size = vc_avg_size
        self.vc_log_single = vc_log_single
        self.vc_log_average = vc_log_average
        self.vc_output_raw = vc_output_raw
        self.vc_control_gpio = vc_control_gpio

    def to_writer_args(self) -> WriterArgs:
        return VcWriterArgs(
            model=self.model,
            serial=self.serial,
            logger_id=self.logger_id,
            port=self.port,
            record_per_file=self._get_record_per_file(),
            vc_fft_size=self.vc_fft_size,
            vc_exe_size=self.vc_exe_size,
            vc_avg_size=self.vc_avg_size,
            vc_log_single=self.vc_log_single,
            vc_log_average=self.vc_log_average,
            vc_output_raw=self.vc_output_raw,
            vc_control_gpio=self.vc_control_gpio,
        )
