# Copyright 2024, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

from dataclasses import dataclass
from typing import Any, TypedDict

from logger import ConfigKeys, Constants, InfoKeys
from logger.utils.env import get_bool, get_float, get_int, get_str
from logger.utils.format import app_format
from logger.utils.validator import Validator


@dataclass
class Config:
    """
    設定ファイルで指定された設定値を保持するクラス

    Class that holds configuration values specified in the configuration file
    """

    logger_id: str
    output_path: str
    initial_wait: int
    measure_time_sec: int
    file_rotate_min: int
    baud_rate: int
    A342_physical: str
    A342_mode: str
    A342_rmspp_output_interval: int
    A352_sps: int
    A370_sps: int
    sensor_data_diag: bool
    sensor_data_diag_sec: float
    message_send: bool
    message_host: str
    message_port: int

    def add_info(self, info_d: dict[str, Any]) -> None:
        """
        計測情報ファイルへの出力用に設定項目を登録する

        このクラスでは以下の情報項目を登録する：
        - FILE_ROTATE_MIN, BAUD_RATE, SENSOR_DATA_DIAG, SENSOR_DATA_DIAG_SEC
        - ※ Controller, Sensor と項目を分けているため、すべての設定値を登録するわけではない

        Registers configuration items for output to the measurement information file

        This class registers the following information items:
        - FILE_ROTATE_MIN, BAUD_RATE, SENSOR_DATA_DIAG, SENSOR_DATA_DIAG_SEC
        - *Note: Not all configuration values are registered as items are separated by Controller and Sensor

        Args:
            info_d (dict): 情報登録用辞書
                Dictionary for information registration
                - 登録する値は最終的に文字列化されるため、数値型のまま登録も可能
                  The registered values can remain in numeric form as they will eventually be converted to strings
        """
        info_d.update(
            {
                InfoKeys.FILE_ROTATE_MIN: self.file_rotate_min,
                InfoKeys.BAUD_RATE: self.baud_rate,
                InfoKeys.SENSOR_DATA_DIAG: self.sensor_data_diag,
                InfoKeys.SENSOR_DATA_DIAG_SEC: f"{self.sensor_data_diag_sec:f}",
                InfoKeys.MESSAGE_SEND: self.message_send,
                InfoKeys.MESSAGE_HOST: app_format(self.message_host),
                InfoKeys.MESSAGE_PORT: app_format(self.message_port),
            }
        )


class ConfigOverrideOption(TypedDict, total=False):
    """
    設定ファイルの設定値を上書きするためのクラス

    Class for overriding configuration values in the configuration file
    """

    measure_time_sec: int | None


class Configurator:
    """
    設定ファイルから設定値を読み込み、設定オブジェクトを返すクラス

    Class that reads configuration values from a configuration file and returns a configuration object
    """

    def get_config(
        self,
        option: ConfigOverrideOption | None = None,
    ) -> Config | None:
        """
        環境変数(設定ファイル)から設定値を読み込み設定オブジェクトを返す

        - 設定値の読み込みにあたり、設定値が想定された形式であるかバリデーションを行う
            - すべてのバリデーションに成功した場合、設定オブジェクトが返る
        - option による設定値の上書きが可能

        Reads configuration values from environment variables (configuration file) and returns a configuration object

        - Validates whether the configuration values are in the expected format during the reading process
            - If all validations are successful, a configuration object is returned
        - Allows overriding configuration values with the option

        Args:
            option (ConfigOverrideOption): 環境変数の設定値を上書きする値を保持するオブジェクト
                Object holding values to override configuration values from environment variables
                - 現在は計測秒数のみ対応
                  Currently only supports measurement time
        Returns:
            Union[Config,None]: 設定オブジェクト
                Configuration object
                - バリデーションに失敗した場合はNoneが返る
                  Returns None if validation fails
        """
        v = Validator()
        results: list[bool] = []

        # LOGGER_ID
        logger_id = get_str(ConfigKeys.LOGGER_ID)
        results.append(
            v.match(logger_id, ConfigKeys.LOGGER_ID, Constants.LOGGER_ID_PATTERN)
        )

        # OUTPUT_PATH
        output_path = get_str(ConfigKeys.OUTPUT_PATH)
        results.append(v.exist_dir(output_path, ConfigKeys.OUTPUT_PATH))

        # INITIAL_WAIT
        initial_wait = get_int(ConfigKeys.INITIAL_WAIT)
        results.append(
            v.between(
                initial_wait,
                ConfigKeys.INITIAL_WAIT,
                Constants.INITIAL_WAIT_RANGE[0],
                Constants.INITIAL_WAIT_RANGE[1],
            )
        )

        # MEASURE_TIME_SEC
        # - option による上書きを処理する
        # - Process overriding with option
        measure_time_sec = (
            get_int(ConfigKeys.MEASURE_TIME_SEC)
            if option is None or option.get("measure_time_sec") is None
            else option.get("measure_time_sec")
        )
        results.append(
            v.between(
                measure_time_sec,
                ConfigKeys.MEASURE_TIME_SEC,
                Constants.MEASURE_TIME_SEC_RANGE[0],
                Constants.MEASURE_TIME_SEC_RANGE[1],
            )
        )

        # FILE_ROTATE_MIN
        file_rotate_min = get_int(ConfigKeys.FILE_ROTATE_MIN)
        results.append(
            v.between(
                file_rotate_min,
                ConfigKeys.FILE_ROTATE_MIN,
                Constants.FILE_ROTATE_MIN_RANGE[0],
                Constants.FILE_ROTATE_MIN_RANGE[1],
            )
        )

        # BAUD_RATE
        baud_rate = get_int(ConfigKeys.BAUD_RATE)
        results.append(v.one_of(baud_rate, ConfigKeys.BAUD_RATE, Constants.BAUD_RATES))

        # A342_PHYSICAL
        A342_physical = get_str(ConfigKeys.A342_PHYSICAL)
        results.append(
            v.one_of(A342_physical, ConfigKeys.A342_PHYSICAL, Constants.A342_PHYSICALS)
        )

        # A342_MODE
        A342_mode = get_str(ConfigKeys.A342_MODE)
        results.append(v.one_of(A342_mode, ConfigKeys.A342_MODE, Constants.A342_MODES))

        # A342_RMSPP_OUTPUT_INTERVAL
        A342_rmspp_output_interval = get_int(ConfigKeys.A342_RMSPP_OUTPUT_INTERVAL)
        results.append(
            v.between(
                A342_rmspp_output_interval,
                ConfigKeys.A342_RMSPP_OUTPUT_INTERVAL,
                Constants.A342_RMS_PP_OUTPUT_INTERVAL_RANGE[0],
                Constants.A342_RMS_PP_OUTPUT_INTERVAL_RANGE[1],
            )
        )

        # A352_SPS
        A352_sps = get_int(ConfigKeys.A352_SPS)
        results.append(v.one_of(A352_sps, ConfigKeys.A352_SPS, Constants.A352_SPSS))

        # A370_SPS
        A370_sps = get_int(ConfigKeys.A370_SPS)
        results.append(v.one_of(A370_sps, ConfigKeys.A370_SPS, Constants.A370_SPSS))

        # SENSOR_DATA_DIAG
        sensor_data_diag = get_bool(ConfigKeys.SENSOR_DATA_DIAG)
        results.append(v.is_bool(sensor_data_diag, ConfigKeys.SENSOR_DATA_DIAG))

        # SENSOR_DATA_DIAG_SEC
        sensor_data_diag_sec = get_float(ConfigKeys.SENSOR_DATA_DIAG_SEC)
        results.append(
            v.greater_than(sensor_data_diag_sec, ConfigKeys.SENSOR_DATA_DIAG_SEC, 0)
        )
        results.append(
            v.less_than_or_equal_to(
                sensor_data_diag_sec, ConfigKeys.SENSOR_DATA_DIAG_SEC, 10
            )
        )

        # MESSAGE_SEND
        message_send = get_bool(ConfigKeys.MESSAGE_SEND)
        results.append(v.is_bool(message_send, ConfigKeys.MESSAGE_SEND))

        # MESSAGE_HOST
        message_host = get_str(ConfigKeys.MESSAGE_HOST)
        results.append(v.defined(message_host, ConfigKeys.MESSAGE_HOST))

        # MESSAGE_PORT
        message_port = get_int(ConfigKeys.MESSAGE_PORT)
        results.append(
            v.between(
                message_port,
                ConfigKeys.MESSAGE_PORT,
                Constants.MESSAGE_PORT_RANGE[0],
                Constants.MESSAGE_PORT_RANGE[1],
            )
        )

        # 結果判定： いずれか失敗している場合は失敗
        # Result judgment: If any fail, it is considered a failure
        if not all(results):
            return None

        # 設定オブジェクト構築・返却
        # - ここまでの処理で適正な値を保持していることが確認できているため、型チェックを無効化
        # Construct and return the configuration object
        # - Disable type checking as it has been confirmed that valid values are held up to this point
        return Config(
            logger_id=logger_id,  # type: ignore
            output_path=output_path,  # type: ignore
            initial_wait=initial_wait,  # type: ignore
            measure_time_sec=measure_time_sec,  # type: ignore
            file_rotate_min=file_rotate_min,  # type: ignore
            baud_rate=baud_rate,  # type: ignore
            A342_physical=A342_physical,  # type: ignore
            A342_mode=A342_mode,  # type: ignore
            A342_rmspp_output_interval=A342_rmspp_output_interval,  # type: ignore
            A352_sps=A352_sps,  # type: ignore
            A370_sps=A370_sps,  # type: ignore
            sensor_data_diag=sensor_data_diag,  # type: ignore
            sensor_data_diag_sec=sensor_data_diag_sec,  # type: ignore
            message_send=message_send,  # type: ignore
            message_host=message_host,  # type: ignore
            message_port=message_port,  # type: ignore
        )
