# Copyright 2024, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

from multiprocessing import Queue
from multiprocessing.sharedctypes import Synchronized

from logger import LoggerFactory
from logger.measure import (
    ReaderArgs,
    Sensor,
    WriterArgs,
    reader_job,
    writer_job,
)

from .config import Config, Configurator
from .process import LoggerProcess


class LoggerFactoryImpl(LoggerFactory):
    """
    ロガープログラム内で使われる主要な要素に対するファクトリクラス

    様々な適用先への対応を容易にする目的で、以下の要素の生成を拡張可能にする：
    - 具象 Sensor
    - Sensor Reader Process
    - Sensor Writer Process
    - 設定情報読み込み Configurator

    このクラスは LoggerFactory インタフェースを継承した実体クラス。
    拡張を行うにはこのクラスをベースにする。

    以下の手順で拡張対応を行う：
    1. このクラスのサブクラスを作成する
    2. サブクラスで拡張バージョンを返すファクトリメソッドを定義する
    3. main() 関数の引数に作成したファクトリオブジェクトを渡す

    Factory class for major elements used in the logger program

    To facilitate adaptation to various applications, the following elements can be extended:
    - Concrete Sensor
    - Sensor Reader Process
    - Sensor Writer Process
    - Configurator for reading configuration information

    This class is an implementation that inherits the LoggerFactory interface.
    To extend, use this class as a base.

    Follow these steps to extend:
    1. Create a subclass of this class
    2. Define factory methods in the subclass that return the extended versions
    3. Pass the created factory object as an argument to the main() function
    """

    def __init__(self) -> None:
        self.config: Config

    def set_config(self, config: Config) -> None:
        """
        Config オブジェクトの設定

        Setting the Config object

        Args:
            config (Config):  設定情報を保持したオブジェクト
                    Object holding configuration information
        """
        self.config = config

    def create_configurator(self) -> Configurator:
        """
        設定項目を読み込み Config オブジェクトを生成する Configurator のファクトリメソッド

        Factory method for the Configurator that reads configuration items and generates a Config object

        Returns:
            Configurator object
        """
        return Configurator()

    def create_A342(self, port: str, product_id: str) -> Sensor:
        """
        A342 センサーのファクトリメソッド

        Factory method for the A342 sensor

        Args:
            port (str): センサーが接続されたポート
                    Port to which the sensor is connected
            product_id (str): センサーから取得した PRODUCT_ID
                    PRODUCT_ID obtained from the sensor

        Returns:
            A342 sensor object
        """
        # - 循環 import を回避するためこのタイミングで import
        # - Import at this point to avoid circular import
        from logger.measure import A342

        return A342(
            port=port,
            baud=self.config.baud_rate,
            product_id=product_id,
            logger_id=self.config.logger_id,
            file_rotate_min=self.config.file_rotate_min,
            mode=self.config.A342_mode,
            physical=self.config.A342_physical,
            rms_pp_interval=self.config.A342_rmspp_output_interval,
            sensor_data_diag=self.config.sensor_data_diag,
            sensor_data_diag_sec=self.config.sensor_data_diag_sec,
        )

    def create_A352(self, port: str, product_id: str) -> Sensor:
        """
        A352 センサーのファクトリメソッド

        Factory method for the A352 sensor

        Args:
            port (str): センサーが接続されたポート
                    Port to which the sensor is connected
            product_id (str): センサーから取得した PRODUCT_ID
                    PRODUCT_ID obtained from the sensor

        Returns:
            A352 sensor object
        """
        # - 循環 import を回避するためこのタイミングで import
        # - Import at this point to avoid circular import
        from logger.measure import A352

        return A352(
            port=port,
            baud=self.config.baud_rate,
            product_id=product_id,
            sps=self.config.A352_sps,
            logger_id=self.config.logger_id,
            file_rotate_min=self.config.file_rotate_min,
            sensor_data_diag=self.config.sensor_data_diag,
            sensor_data_diag_sec=self.config.sensor_data_diag_sec,
        )

    def create_A370(self, port: str, product_id: str) -> Sensor:
        """
        A370 センサーのファクトリメソッド

        Factory method for the A370 sensor

        Args:
            port (str): センサーが接続されたポート
                    Port to which the sensor is connected
            product_id (str): センサーから取得した PRODUCT_ID
                    PRODUCT_ID obtained from the sensor

        Returns:
            A370 sensor object
        """
        # - 循環 import を回避するためこのタイミングで import
        # - Import at this point to avoid circular import
        from logger.measure import A370

        return A370(
            port=port,
            baud=self.config.baud_rate,
            product_id=product_id,
            sps=self.config.A370_sps,
            logger_id=self.config.logger_id,
            file_rotate_min=self.config.file_rotate_min,
            sensor_data_diag=self.config.sensor_data_diag,
            sensor_data_diag_sec=self.config.sensor_data_diag_sec,
        )

    def create_reader_process(
        self,
        queue: Queue,
        error_queue: Queue,
        reader_args: ReaderArgs,
        measuring: Synchronized,
    ) -> LoggerProcess:
        """
        Sensor Reader プロセスのファクトリメソッド

        引数で受け取っているのはデフォルトの reader_job で使うもの。
        reader_job を置き換える場合には、必要に応じてこのメソッド内で変更してよい。

        Factory method for the Sensor Reader process.

        The arguments received are used for the default reader_job.
        If replacing reader_job, it can be modified within this method as needed.

        Args:
            queue (Queue): 計測データを Writer に送信するキュー
                    Queue to send measurement data to the Writer
            error_queue (Queue): エラー発生時にコントローラーに通知するキュー
                    Queue to notify the controller in case of errors
            reader_args (ReaderArgs): 計測設定情報を保持するオブジェクト
                    Object holding measurement configuration information
            measuring (Synchronized): 計測中フラグ
                    Flag indicating measurement in progress

        Returns:
            Reader 機能を持った LoggerProcess オブジェクト
            LoggerProcess object with Reader functionality
        """
        return LoggerProcess(
            target=reader_job, args=(queue, error_queue, reader_args, measuring)
        )

    def create_writer_process(
        self, queue: Queue, error_queue: Queue, writer_args: WriterArgs, output_dir: str
    ) -> LoggerProcess:
        """
        Sensor Writer プロセスのファクトリメソッド

        引数で受け取っているのはデフォルトの writer_job で使うもの。
        writer_job を置き換える場合には、必要に応じてこのメソッド内で変更してよい。

        Factory method for the Sensor Writer process

        The arguments received are used for the default writer_job.
        If replacing writer_job, it can be modified within this method as needed.

        Args:
            queue (Queue): 計測データを Reader から受信するキュー
                    Queue to receive measurement data from the Reader
            error_queue (Queue): エラー発生時にコントローラーに通知するキュー
                    Queue to notify the controller in case of errors
            writer_args (WriterArgs): 計測設定情報を保持するオブジェクト
                    Object holding measurement configuration information
            output_dir (str): 計測情報の保存ディレクトリ
                    Directory to save measurement information

        Returns:
            Writer 機能を持った LoggerProcess オブジェクト
            LoggerProcess object with Writer functionality
        """
        return LoggerProcess(
            target=writer_job, args=(queue, error_queue, writer_args, output_dir)
        )
