# Copyright 2024, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

import logging
import multiprocessing
import signal
from traceback import format_exc
from typing import Any, Callable, Iterable, Mapping, Type

from .service import ServiceBase, ServiceFrame


class MultiProcessServiceFrame(ServiceFrame):
    """
    マルチプロセス環境で「サービス」を動作させる基盤を提供する Singleton クラス

    Singleton 機構としては ServiceFrame の仕組みを共用する

    Singleton class providing the foundation to run "services" in a multi-process environment

    Shares the mechanism of ServiceFrame as a Singleton
    """

    # シングルトンを成立させるため、通常のインスタンス生成を抑止
    # Prevent normal instance creation to enforce Singleton pattern
    def __new__(cls):
        raise NotImplementedError("Cannot instantiate through constructor")

    @classmethod
    def get_instance(cls) -> "MultiProcessServiceFrame":
        """
        シングルトンの ServiceFrame インスタンスを取得するメソッド

        プロセス内でこのクラスのメソッドが最初に呼び出された場合、内部では
        MultiProcessServiceFrame インスタンスを生成し、Singleton に登録する

        Method to get the Singleton instance of ServiceFrame

        If this class's method is called first within the process,
        it creates and registers a MultiProcessServiceFrame instance as a Singleton

        Returns:
            MultiProcessServiceFrame
        """
        if not cls._instance:
            # このクラス経由で __internal_new__ を呼び出しているため、このクラスのインスタンスになる
            # Calls __internal_new__ through this class, making it an instance of this class
            cls._instance = cls.__internal_new__()

        assert isinstance(cls._instance, MultiProcessServiceFrame)
        return cls._instance

    def __init__(self):
        # 上位を初期化
        # Initialize the superclass
        super().__init__()

        # サブプロセスの生成方法をロガープログラムに適したものに変更する（先頭で実行）
        # Change the subprocess creation method to suit the logger program (execute first)
        multiprocessing.set_start_method("spawn")

        # プロセス関連オブジェクト生成
        # Create process-related objects
        self._server_process: multiprocessing.Process | None = None
        self._server_queue: multiprocessing.Queue = multiprocessing.Queue(-1)

    def start(self) -> None:
        """
        サービス基盤を開始するメソッド

        このクラスの実装では、サーバープロセスを起動し、登録されているサーバーを送り込む。

        Method to start the service foundation

        In this class implementation, it starts the server process and sends the registered servers.
        """
        # 初期化済み抑止
        # Prevent re-initialization
        if self._server_process:
            return

        # サーバープロセス起動
        # Start server process
        self._server_process = multiprocessing.Process(
            target=MultiProcessServiceFrame._listen,
            args=(self._servers, self._server_queue),
        )
        self._server_process.start()

        # メインプロセス用にクライアントセットアップ
        # - サーバーとは Queue で接続されるため、Queue を引き渡す
        # Set up clients for the main process
        # - Connect to the server via Queue, so pass the Queue
        for service in self._services:
            service.create_client().setup(queue=self._server_queue)

    def _post_add(self, service: ServiceBase, server: ServiceBase.Server) -> None:
        """
        サービス登録後の処理を実行するテンプレートメソッド

        このクラスでは、サーバープロセスが起動されている場合、Queue に対してサーバーを送信する。

        Template method to execute post-service registration processing

        In this class, if the server process is running, send the server to the Queue.
        """
        if self._server_process:
            # サーバー自体を転送
            # Transfer the server itself
            self._server_queue.put(server)
            # クライアントセットアップ
            # Set up client
            service.create_client().setup(queue=self._server_queue)

    def close(self, **kwargs) -> None:
        """
        サービス基盤を終了するメソッド

        このクラスの実装では、サーバープロセスを停止させるため、Queue にマーカーを送信する。

        Method to stop the service foundation

        In this class implementation, send a marker to the Queue to stop the server process.

        Args:
            kwargs (dict): 名前付き可変引数を格納した dict
                Dict containing named variable arguments
                - ここで渡された名前付き引数は、そのままサーバーの停止に引き渡される
                  Named arguments passed here are passed directly to stop the server
                - 名前を合わせておくことで、個別のサーバーに引数を渡すことができる
                  By matching names, arguments can be passed to individual servers
        """
        try:
            # サーバープロセス終了
            # End server process
            if self._server_process is not None:
                # - 終了時情報として kwargs を送る
                # - Send kwargs as termination information
                self._server_queue.put(kwargs)

                # - None を送信してプロセスを終了させる
                # - Send None to terminate the process
                self._server_queue.put_nowait(None)
                self._server_process.join()
                self._server_process = None

        except Exception as e:
            # この処理は main の終了時に呼ばれる
            # すでにサーバーが停止してログ出力が終了している可能性があるため、Error 出力のみ
            # This process is called at the end of main
            # Since the server may have already stopped and logging has ended, only output Error
            print(f"Closing multi-process service failed with an Error: {str(e)}")

    def get_clients(self) -> list[ServiceBase.Client]:
        """
        登録されているサービスのクライアントリストを返すメソッド

        Method to return the list of clients for registered services

        Returns:
            list[ServiceBase.Client]: 登録されているサービスの Client リスト
                    List of Clients for registered services
        """
        return [service.create_client() for service in self._services]

    def get_queue(self) -> multiprocessing.Queue:
        """
        サーバープロセスとの転送用に使用している Queue を返す

        Returns the Queue used for transfer with the server process

        Returns:
            multiprocessing.Queue: サーバープロセスとの転送用に使用している Queue
                    Queue used for transfer with the server process
        """
        return self._server_queue

    @staticmethod
    def _listen(  # noqa: C901
        servers: list[ServiceBase.Server], queue: multiprocessing.Queue
    ) -> None:
        """
        マルチプロセス環境で、サーバープロセスとして、転送用の Queue を監視し、
        サーバー処理をディスパッチする関数

        multiprocessing.Process に実行関数として指定するため staticmethod

        Function to monitor the transfer Queue and dispatch server processing as a server
        process in a multi-process environment

        Defined as a staticmethod to be specified as the execution function in multiprocessing.Process

        Args:
            servers (list[ServiceBase.Server]): サーバーのリスト
                    List of servers
            queue (multiprocessing.Queue): 転送用 Queue
                    Queue for transfer
        """

        # シグナル処理： SIGINT は無視する
        # Signal handling: Ignore SIGINT
        signal.signal(signal.SIGINT, signal.SIG_IGN)

        # サーバー開始処理
        # Server start process
        logger: logging.Logger | None = None

        # - サーバーディスパッチ用 dict
        # - dict for server dispatch
        server_dict: dict[Type, ServiceBase.Server] = {}

        def _start_server(server: ServiceBase.Server) -> None:
            # サーバーが扱える型に対してサーバーを登録する
            # Register the server for the types it can handle
            for dtype in server.declare_type():
                server_dict[dtype] = server
            # サーバー開始
            # Start server
            if logger:
                logger.debug(f"Starting server: {repr(server)}")
            server.start()

        # - 登録済みサーバーの開始
        # - Start registered servers
        for server in servers:
            _start_server(server)

        # ロギング確保
        # Ensure logging
        logger = logging.getLogger(__name__)

        # Queue ループ
        # Queue loop
        close_data = {}
        while True:
            try:
                # Queue からデータを取得
                # Get data from Queue
                data = queue.get()

                # - None: 終了マーカー
                # - None: Termination marker
                if data is None:
                    break

                # - データ型に応じてサーバーで処理
                # - Process with server according to data type
                dtype = type(data)
                dserv = server_dict.get(dtype)
                if dserv:
                    dserv.handle_data(data)

                # - Server: 新たなサーバーの登録
                # - Server: Register a new server
                elif issubclass(dtype, ServiceBase.Server):
                    _start_server(data)
                    servers.append(data)

                # - dict: 終了時付加情報
                # - dict: Additional information at termination
                elif dtype is dict:
                    close_data = data

                else:
                    logger.debug(f"Unknown queued data type: {dtype}")

            except Exception:
                # ** 例外が出力されない場合があるため、手動で出力
                # ** Manually output as exceptions may not be logged

                # logger.exception("An error occurred within listner process")
                logger.error(
                    "An error occurred within listner process\n" + format_exc()
                )

        # サーバー終了処理（逆順）
        # Server shutdown process (reverse order)
        for server in servers[::-1]:
            server.close(**close_data)


class LoggerProcess(multiprocessing.Process):
    """
    アプリケーションに必要な処理を追加した multiprocessing.Process のラッパークラス

    以下の機能を持つ
    - プロセスセーフなファイル出力ロギング
    - キーボードインタラプトでプロセスを終了させない
    - Terminateでプロセスを終了させない
      - process.terminate() でプロセスが終了しなくなるため、必ず終了する関数を渡すこと

    Wrapper class for multiprocessing.Process with additional functionalities
    required for the application

    Provides the following features:
    - Process-safe file output logging
    - Prevents process termination by keyboard interrupt
    - Prevents process termination by terminate
      - Ensure to pass a function that guarantees termination
        as process.terminate() will not terminate the process

    Args:
        - multiprocessing.Process に同じ
        - Same as multiprocessing.Process
    """

    def __init__(
        self,
        group: None = None,
        target: Callable | None = None,
        name: str | None = None,
        args: Iterable[Any] = (),
        kwargs: Mapping[str, Any] = {},
        *,
        daemon: bool | None = None,
    ) -> None:
        # マルチプロセス環境サービス基盤取得
        # Obtain multi-process environment service foundation
        mp_frame = MultiProcessServiceFrame.get_instance()

        # サービスクライアント取得
        # Obtain service clients
        clients = mp_frame.get_clients()
        queue = mp_frame.get_queue()

        # Process を初期化
        # Initialize Process
        super().__init__(
            target=LoggerProcess._run_target,
            args=(clients, queue, target, args, kwargs),
            group=group,
            name=name,
            kwargs=kwargs,
            daemon=daemon,
        )

    @staticmethod
    def _run_target(
        clients: list[ServiceBase.Client],
        queue: multiprocessing.Queue,
        target: Callable | None = None,
        args: Iterable[Any] = (),
        kwargs: Mapping[str, Any] = {},
    ):
        """
        指定された関数をサブプロセス上で実行させる関数

        Function to execute the specified function on the subprocess

        Args:
            clients (list[ServiceBase.Client]): サービスクライアントのリスト
                    List of service clients
            queue (multiprocessing.Queue): 転送用 Queue
                    Queue for transfer
            target (Callable | None): 実行する関数
                    Function to execute
            args (Iterable[Any]): target の引数
                    Arguments for target
            kwargs (Mapping[str, Any]): target の名前付き引数
                    Named arguments for target
        """

        # シグナル処理
        # - サブプロセスでは SIGTERM, SIGINT を無視する
        # Signal handling
        # - Ignore SIGTERM, SIGINT in subprocess
        signal.signal(signal.SIGTERM, signal.SIG_IGN)
        signal.signal(signal.SIGINT, signal.SIG_IGN)

        # サービスクライアントセットアップ
        # Set up service clients
        for client in clients:
            # マルチプロセス環境のため、queue を指定してセットアップする
            # Specify queue for setup in a multi-process environment
            client.setup(queue=queue)

        # 指定関数の実行
        # Execute the specified function
        if target:
            target(*args, **kwargs)
