# Copyright 2024, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

import logging
import time
from dataclasses import dataclass
from typing import TypeAlias

from serial import Serial


@dataclass
class WaitCommand:
    """
    Serialにメッセージを送る際一定時間停止することを表すコマンド

    Command representing a pause for a certain period when sending a message to Serial
    """

    sec: float


# コマンド型 / Command Type
Command: TypeAlias = list[int]

# コマンドリスト型 / Command List Type
CommandList: TypeAlias = list[WaitCommand | Command]


class Comm:
    """
    シリアルとの通信クラス

    Class for communication with Serial
    """

    DEFAULT_TIMEOUT = 3
    DEFAULT_READ_BYTES = 4096

    def __init__(self):
        self._connection: Serial | None = None
        self._logger = logging.getLogger(__name__)

    def open(
        self,
        port: str,
        baud: int,
        timeout: float | None = DEFAULT_TIMEOUT,
    ) -> None:
        """
        シリアルとのコネクションを開くメソッド

        Method to open a connection with Serial

        Args:
            port (str): 通信対象のポート
                    Port to communicate with
            baud (int): 通信速度
                    Communication speed
            timeout (Optional[float], optional): 接続のタイムアウト時間
                    Connection timeout duration.
                - Noneを渡した場合、タイムアウトしなくなる
                    If None is passed, it will not timeout.
                - デフォルトは Comm.DEFAULT_TIMEOUT (3秒)
                    Default is Comm.DEFAULT_TIMEOUT (3 seconds).
        """
        self._logger.debug(f"Open: {port} {baud} {timeout}")
        self._connection = Serial(port, baud, timeout=timeout)

    def send(self, commands: CommandList) -> list[int]:
        """
        コマンドを送って結果を受け取るメソッド

        Method to send commands and receive results

        Args:
            commands (CommandList): デバイスに送信するコマンド
                    Commands to send to the device.
                - WaitCommandを送った場合、その時間だけ待機する
                    If a WaitCommand is sent, it will wait for that duration.

        Raises:
            TimeoutError: タイムアウト内にデータが帰ってこなかった場合
                    If data is not returned within the timeout period.
            AssertionError: openメソッド実行前に呼び出した場合
                    If called before the open method is executed.

        Returns:
            list[int]: 受け取った結果
                    The received results.
        """
        assert self._connection

        result = []

        for command in commands:
            self._logger.debug(f"Send command: {command}")
            if isinstance(command, WaitCommand):
                time.sleep(command.sec)
            else:
                self._connection.write(bytes(command[1:]))
                self._connection.flush()

                # データ受信
                # Receive data
                if command[0] > 0:
                    read_result = self.read(command[0])
                    result.extend(read_result)

        return result

    def read(self, length: int) -> list[int]:
        """
        指定した長さのデータを読み取るメソッド

        Method to read data of the specified length

        Args:
            length (Optional[int]): 読み取りたいデータ長
                    Length of data to read.
                - 未指定の場合は一回に読み取れたデータをそのまま返す
                    If not specified, returns the data read in one go.
        Raises:
            TimeoutError: タイムアウト内にデータが帰ってこなかった場合
                    If data is not returned within the timeout period.
            AssertionError: openメソッド実行前に呼び出した場合
                    If called before the open method is executed.

        Returns:
            list[int]: intの配列に変換した受け取ったデータ
                    The received data converted to an array of integers.
        """
        return list(self.read_bytes(length))

    def read_bytes(self, length: int) -> bytes:
        """
        指定した長さのデータをバイトで読み取るメソッド

        Method to read data of the specified length in bytes

        Args:
            length (int | None): 読み取りたいデータ長
                    Length of data to read.
                - 未指定の場合は一回に読み取れたデータをそのまま返す
                    If not specified, returns the data read in one go.

        Raises:
            TimeoutError: タイムアウト内にデータが帰ってこなかった場合
                    If data is not returned within the timeout period.
            AssertionError: openメソッド実行前に呼び出した場合
                    If called before the open method is executed.

        Returns:
            bytes: 読み取ったバイト列
                    The read byte sequence.
        """
        assert self._connection

        result = bytes()

        while length > 0:
            read_length = (
                length if length < Comm.DEFAULT_READ_BYTES else Comm.DEFAULT_READ_BYTES
            )
            if read_length > 0:
                received = self._connection.read(read_length)
                read_length = len(received)

                # self._logger.debug(f"Receive data: {read_length} bytes, {str(received[:30])}...")

                # Timeout 判定
                # Timeout check
                if read_length == 0:
                    raise TimeoutError("Read timeout occurred")

                # データが分割されることもあるため list を拡張・保持する
                # - datalen を超えるデータは保持しない
                # Since data may be split, extend and retain the list
                # - Do not retain data exceeding datalen
                result += received[0:length]
                length -= read_length

        return result

    def close(self) -> None:
        """
        デバイスとの通信を終了するメソッド

        Method to terminate communication with the device
        """
        if self._connection is None:
            return
        self._logger.debug("Close")
        self._connection.reset_input_buffer()
        self._connection.close()
        self._connection = None


def send(
    commands: CommandList,
    port: str,
    baud: int,
    timeout: float | None = Comm.DEFAULT_TIMEOUT,
) -> list[int]:
    """
    通信の確立から終了まで含め、デバイスにコマンドを送り、メッセージを受け取る関数

    Function to send commands to the device and receive messages,
    including establishing and terminating communication

    Args:
        commands (CommandList): デバイスに送信するコマンド
                Commands to send to the device.
            - WaitCommandを送った場合その時間だけ待機する
                If a WaitCommand is sent, it will wait for that duration.
        port (str): 通信対象のポート
                Port to communicate with
        baud (int): 通信速度
                Communication speed
        timeout (Optional[float], optional): 接続のタイムアウト時間
                Connection timeout duration.
            - Noneを渡した場合、タイムアウトしなくなる
                If None is passed, it will not timeout.
            - デフォルトは Comm.DEFAULT_TIMEOUT(3秒)
                Default is Comm.DEFAULT_TIMEOUT (3 seconds).

    Returns:
        list[int]: 読み取った結果
                The read results.
    """
    try:
        com = Comm()
        com.open(port, baud, timeout)
        result = com.send(commands)
    finally:
        try:
            com.close()
        except Exception:
            pass
    return result
