# Copyright 2024, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

import datetime
import logging
import multiprocessing
from dataclasses import dataclass
from multiprocessing.sharedctypes import Synchronized

from logger.core import MessageService, Topic

from .comm import Comm, CommandList
from .data import MeasureData


@dataclass
class ReaderArgs:
    """Reader プロセスのジョブに渡す設定情報のデータクラス

    Sensor 継承クラスでは、このクラスを継承したクラスを実装し、to_reader_args() で生成する

    Data class for configuration information passed to the Reader process job

    In Sensor derived classes, implement a class that inherits this class and generate it with to_reader_args()
    """

    model: str
    serial: str
    port: str
    baud: int
    start_command: CommandList
    end_command: CommandList
    record_size: int
    record_begin: int
    record_end: int
    record_per_sec: int
    count_diff: int
    count_max: int
    count_start: int
    sensor_data_diag: bool
    diag_broken_count: int
    read_length: int
    timeout: float

    def __post_init__(self) -> None:
        self._diag_broken_counts = {"x": 0, "y": 0, "z": 0}

    def format_packet(self, index: int, packet: list[int]) -> MeasureData:
        """
        1レコード分のデータを受け取って計測データに変換するメソッド

        ReaderArgsを継承するクラスで具体的な処理を記載する

        Method to receive one record of data and convert it to measurement data

        Specific processing is described in classes that inherit ReaderArgs

        Args:
            index (int): 何番目のデータか
                    Index of the data
            packet (list[int]): 変換対象のデータ配列
                    Data array to be converted

        Raises:
            NotImplementedError: サブクラスで実装しなかった場合
                    If not implemented in a subclass

        Returns:
            MeasureData: 変換後のpythonデータクラス
                    Converted Python data class
        """
        raise NotImplementedError

    def complement_data(
        self, prev: MeasureData | None, next: MeasureData
    ) -> list[MeasureData]:
        """
        カウント値をチェックしてデータを補完するメソッド

        Method to check count values and complement data

        Args:
            prev (Optional[MeasureData]): 補完済みの最後のデータ
                    Previous data that has been complemented
            next (MeasureData): 補完前の新たに配列に追加するデータ
                    New data to be added to the array before complementing

        Returns:
            list[MeasureData]: データ欠損を補完した計測データ配列
                    Measurement data array with complemented missing data
        """
        logger = logging.getLogger(
            f"{__name__}.{self.model}.{self.serial}.complement_data"
        )

        # 初回のみ： index=-1 のデータを prev とする
        # Only for the first time: Use data with index=-1 as prev
        if prev is None:
            prev = MeasureData(
                index=-1,
                count=self.count_start - self.count_diff,
                temperature=0,
                x=0,
                y=0,
                z=0,
                flag=0,
            )

        # 次のカウント値に一致するまで一回ごとの差分を足し続けて差分を計算する
        # Continue adding the difference each time until it matches the next count value
        diff = self.count_diff
        while (diff + prev.count) % self.count_max != next.count:
            diff += self.count_diff

        # カウントの差分と想定される差分が同一の場合、データ損失がないためそのまま返す
        # If the difference in counts matches the expected difference, return as is since there is no data loss
        if diff == self.count_diff:
            next.index = prev.index + 1
            return [next]

        # 欠損データの補完
        # Complement missing data
        m = "Missing {} data from index: {}. Complement them.".format(
            ((diff // self.count_diff) - 1), (prev.index + 1)
        )
        logger.warning(m)
        MessageService.send(Topic.sensor_lost(self.model, self.serial), m)

        result: list[MeasureData] = []
        for _ in range(0, diff, self.count_diff):
            index = prev.index + 1
            count = (prev.count + self.count_diff) % self.count_max

            # - カウント が next と一致： ループの終端のため、元のデータを入れる
            # - If count matches next: Insert the original data as it is the end of the loop
            if count == next.count:
                next.index = index
                result.append(next)
            # - データ補完
            # - Complement data
            else:
                complement = self._gen_invalid_data(index=index, count=count)
                result.append(complement)
                prev = complement

        return result

    def _gen_invalid_data(self, index: int, count: int) -> MeasureData:
        """データ抜けがあった際に不正な値であることを示すデータクラスオブジェクトを生成するメソッド

        サブクラスで実装すること

        Method to generate a data class object indicating invalid values when data is missing

        To be implemented in subclasses

        Args:
            index (int): 不正な計測データのindex
                    Index of the invalid measurement data
            count (int): 不正な計測データのカウント値、complement_dataの中で前のデータから算出する
                    Count value of the invalid measurement data, calculated from the previous
                    data in complement_data

        Raises:
            NotImplementedError: サブクラスで実装しなかった場合
                    If not implemented in a subclass

        Returns:
            MeasureData: 補完したindexとcountを持つ不正なことを表すデータクラスオブジェクト
                    Data class object indicating invalidity with complemented index and count
        """
        raise NotImplementedError

    def diag_sensor_data(self, prev: MeasureData | None, next: MeasureData) -> None:
        """
        センサーデータのエラー判定

        Error detection in sensor data
        """

        logger = logging.getLogger(
            f"{__name__}.{self.model}.{self.serial}.diag_sensor_data"
        )
        if prev is None:
            return

        for axis in ["x", "y", "z"]:
            p = getattr(prev, axis)
            n = getattr(next, axis)
            if p == n:
                self._diag_broken_counts[axis] += 1
                if self._diag_broken_counts[axis] == self.diag_broken_count:
                    m = f"Sensor on axis: {axis} is possibly broken"
                    logger.warning(m)
                    MessageService.send(
                        Topic.sensor_abnormal(self.model, self.serial), m
                    )
            else:
                if self._diag_broken_counts[axis] >= self.diag_broken_count:
                    m = f"Sensor on axis: {axis} is fixed"
                    logger.info(m)
                    MessageService.send(
                        Topic.sensor_abnormal(self.model, self.serial), m
                    )
                self._diag_broken_counts[axis] = 0


def reader_job(
    queue: multiprocessing.Queue,
    error_queue: multiprocessing.Queue,
    args: ReaderArgs,
    measuring: Synchronized,
) -> None:
    """
    Reader プロセス上でセンサーからデータを受け取り、整形してキューに渡すジョブ関数

    Job function to receive data from sensors on the Reader process, format it, and pass it to the queue

    Args:
        queue (multiprocessing.Queue): 計測データを Writer に送信するキュー
                Queue to send measurement data to the Writer
        error_queue (multiprocessing.Queue): エラー発生時にコントローラーに通知するキュー
                Queue to notify the controller in case of errors
        args (ReaderArgs): 計測設定情報を保持するオブジェクト
                Object holding measurement configuration information
        measuring (Synchronized): 計測中フラグ
                Flag indicating if measurement is in progress
    """
    logger = logging.getLogger(f"{__name__}.{args.model}.{args.serial}.reader")
    comm = Comm()
    try:
        # 計測開始
        # Start measurement
        comm.open(args.port, args.baud, args.timeout)
        comm.send(args.start_command)

        process_count = 0
        data: list[int] = []  # 未処理 / Unprocessed
        conv: list[MeasureData] = []  # 変換済 / Converted
        send: list[MeasureData] = []  # 送信対象 / Send list
        last: MeasureData | None = None  # 補完済み最新 / Latest complemented

        # 計測中ループ
        # Measurement loop
        while measuring.value:
            result = comm.read(args.read_length)
            data.extend(result)

            # パケットから計測データに変換する
            # Convert packets to measurement data
            conv, rest = _packets_to_data(data, args)
            # - 未処理を更新
            # - Update unprocessed
            data = rest

            # 変換データを補完しながら送信対象に詰めていく
            # Complement converted data and add to send list
            for cd in conv:
                if args.sensor_data_diag:
                    args.diag_sensor_data(last, cd)
                send.extend(args.complement_data(last, cd))
                last = send[-1]

            # 1秒以上のデータが溜まっていたら Writer プロセスに送出
            # Send data to Writer process if more than 1 second of data is accumulated
            if len(send) >= args.record_per_sec:
                queue.put(send[0 : args.record_per_sec])
                # 送信対象を更新
                # Update send list
                send = send[args.record_per_sec :]
                process_count += args.record_per_sec

        # 計測中フラグがオフになったら残りを送信する
        # Send remaining data when measuring flag is off
        queue.put(send)

    except Exception as e:
        # サブプロセスでの例外発生は、ログ出力をしてエラーキューでコントローラーに送信する
        # Log exceptions in subprocess and send to controller via error queue
        m = "An error occurred during reading data"
        logger.exception(m)
        error_queue.put(JobError(model=args.model, serial=args.serial, error=e))

        # - Error, NG メッセージ送信
        # - Send error and NG messages
        MessageService.send(Topic.sensor_error(args.model, args.serial), m, exc=e)
        MessageService.send(Topic.sensor(args.model, args.serial), "NG")

    finally:
        # Reader 終了シーケンス
        # - ここでは例外をキャッチしてログ出力はするが次へ進める
        # Reader termination sequence
        # - Catch exceptions here, log them, but proceed
        try:
            # 終了コマンド
            # Send end command
            comm.send(args.end_command)
        except Exception:
            logger.exception("Failed to send stop command")
        try:
            # 通信クローズ
            # Close communication
            comm.close()
        except Exception:
            logger.exception("Failed to disconnect device connection")
        try:
            # Writer に終了を通知
            # Notify Writer to terminate
            queue.put(None)
        except Exception:
            # - 終了通知に失敗すると Writer が残存する恐れがあるため FATAL
            # - If termination notification fails, Writer may remain, hence FATAL
            logger.fatal("Failed to notify finishing writer process")


def _packets_to_data(
    packets: list[int], args: ReaderArgs
) -> tuple[list[MeasureData], list[int]]:
    """
    センサーから読み込んだデータから、計測データオブジェクトの配列に変換する関数

    Function to convert data read from sensors into an array of measurement data objects

    Args:
        packets (list[int]): センサーから読み込んだデータ
                Data read from sensors
        args (ReaderArgs): 計測設定情報を保持するオブジェクト
                Object holding measurement configuration information

    Returns:
        tuple[list[MeasureData],list[int]]: 変換済み計測データの配列, 1レコードに足りず変換できなかったデータ
                Array of converted measurement data,
                data that could not be converted due to insufficient record size
    """
    logger = logging.getLogger(f"{__name__}.{args.model}.{args.serial}.packets_to_data")

    index = 0
    data: list[MeasureData] = []
    rest: list[int] = []
    while True:
        first_err = True
        packet = packets[index : index + args.record_size]

        # packetサイズが0になっている恐れもあるため、while文の条件として長さを判定する
        # Check length as a condition in the while loop since packet size might be 0
        while len(packet) == args.record_size and (
            packet[0] != args.record_begin or packet[-1] != args.record_end
        ):
            if first_err:
                logger.warning(
                    f"Invalid packet boundary: beg={packet[0]:#X}, end={packet[-1]:#X}. Fix packet."
                )
                # 連続してログが出力されるのを抑止
                # Prevent continuous log output
                first_err = False

            # indexを1つずらして再調査
            # Shift index by one and recheck
            index += 1
            packet = packets[index : index + args.record_size]

        # 切り取ったpacketが1レコード分に満たない時はデータの終端に来ているのでループを抜ける
        # Exit loop if the cut packet is less than one record, indicating end of data
        if len(packet) < args.record_size:
            rest = packet
            break

        data.append(args.format_packet(0, packet))

        # indexをレコード分だけずらして次のループへ
        # Shift index by record size for the next loop
        index += args.record_size

    return data, rest


@dataclass
class WriterArgs:
    """
    Writer プロセスのジョブに渡す設定情報のデータクラス

    このクラスでは CSV ファイルを出力する機能を提供する

    Sensor 継承クラスでは、必要に応じてこのクラスを継承したクラスを実装し、to_writer_args() で生成する

    Data class for configuration information passed to the Writer process job

    This class provides functionality to output CSV files

    In Sensor derived classes, implement a class that inherits this class as needed and
    generate it with to_writer_args()
    """
    model: str
    serial: str
    logger_id: str
    port: str
    record_per_file: int

    def __post_init__(self) -> None:
        # ファイル出力制御用属性を追加
        # Add attributes for file output control
        self._output_dir: str
        self._csv_file: str
        self._csv_mode: str
        self._csv_count: int

    def write_start(self, output_dir: str) -> None:
        """
        出力処理を初期化する

        ここでは CSV 出力のための初期化を行う。
        サブクラスで異なる出力処理を行う場合は、オーバーライドして実装する。

        Initialize the output process

        Initialize for CSV output here.
        Override and implement if different output processing is required in subclasses.

        Args:
            output_dir (str): 出力先フォルダ
                    Output folder
        """
        self._output_dir = output_dir
        self._csv_file = f"{self._output_dir}/{self._get_filename()}"
        self._csv_count = 0
        # 最初は上書きモード
        # Initially in overwrite mode
        self._csv_mode = "w"

        _writer_log(self).info(f"Initial output to {self._csv_file}")

    def write_data(self, records: list[MeasureData]) -> None:
        """
        データ出力処理を行う

        ここでは CSV ファイルへの出力処理と、ファイルのローテートを行う。
        サブクラスで異なる出力処理を行う場合は、オーバーライドして実装する。

        Perform data output processing

        Perform output processing to CSV files and file rotation here.
        Override and implement if different output processing is required in subclasses.

        Args:
            records (List[MeasureData]): 計測データのリスト
                    List of measurement data
        """
        logger = _writer_log(self)

        # データ件数がある場合： 計測モードにより 0 レコード送信があるため回避
        # If there are data records: Avoid sending 0 records due to measurement mode
        if len(records) > 0:
            lines = "".join((d.to_csv_line() for d in records))
            with open(
                self._csv_file, mode=self._csv_mode, encoding="UTF-8", newline="\r\n"
            ) as f:
                f.write(lines)
                logger.debug(f"write {len(records)} lines")

            self._csv_count += len(records)
            # 以降は追記モード
            # Subsequent writes are in append mode
            self._csv_mode = "a"

        # ファイルローテート： ファイル名、モードを変更
        # File rotation: Change file name and mode
        if self._csv_count >= self.record_per_file:
            self._csv_file = f"{self._output_dir}/{self._get_filename()}"
            self._csv_count = 0
            self._csv_mode = "w"
            logger.info(f"Output file is rotated to {self._csv_file}")

    def write_end(self) -> None:
        """
        出力処理を終了する

        CSV 出力では何もしない。
        サブクラスで異なる出力処理を行う場合は、オーバーライドして実装する。

        End the output process

        Do nothing for CSV output.
        Override and implement if different output processing is required in subclasses.
        """
        pass

    def _get_filename(self) -> str:
        """
        CSVファイル名を生成するメソッド

        Method to generate the CSV file name

        Returns:
            str: 呼び出された日時を含むCSVファイル名, ディレクトリは含まない
                    CSV file name including the call date and time, excluding the directory
        """
        return "{}_{}_{}_{}_{}.csv".format(
            self.model,
            self.serial,
            self.logger_id,
            self.port[-1],
            datetime.datetime.now().strftime("%y%m%d_%H%M%S"),
        )


def writer_job(
    queue: multiprocessing.Queue,
    error_queue: multiprocessing.Queue,
    args: WriterArgs,
    output_dir: str,
) -> None:
    """
    Writer プロセス上で Reader プロセスから受け取ったデータをファイルに書き込むジョブ関数

    Job function to write data received from the Reader process to a file on the Writer process

    Args:
        queue (multiprocessing.Queue): 計測データを Reader から受信するキュー
                Queue to receive measurement data from the Reader
        error_queue (multiprocessing.Queue): エラー発生時にコントローラーに通知するキュー
                Queue to notify the controller in case of errors
        args (WriterArgs): 計測設定情報を保持するオブジェクト
                Object holding measurement configuration information
        output_dir (str): 計測情報の保存ディレクトリ
                Directory to save measurement information
    """
    logger = _writer_log(args)

    # WriterArgs に出力開始を指示
    # Instruct WriterArgs to start output
    args.write_start(output_dir=output_dir)

    try:
        # 出力ループ
        # Output loop
        finish = False
        while True:
            # キューから取得
            # - Writer の処理間隔を広げるため、Reader 側で 1秒分のレコードをまとめて送出している
            # Retrieve from queue
            # - To widen the processing interval of the Writer, the Reader sends records for 1 second at a time
            records: list[MeasureData] | None = queue.get()

            # - 終了判定： None の場合は計測終了
            # - Termination check: Measurement ends if None
            if records is None:
                finish = True
                break

            # WriterArgs にデータ出力を指示
            # Instruct WriterArgs to output data
            args.write_data(records)

    except Exception as e:
        # サブプロセスでの例外発生は、ログ出力してエラーキューでコントローラーに送信する
        # Log exceptions in subprocess and send to controller via error queue
        m = "An error occurred during writing data"
        logger.exception(m)
        error_queue.put(JobError(model=args.model, serial=args.serial, error=e))

        # - Error, NG メッセージ送信
        # - Send error and NG messages
        MessageService.send(Topic.sensor_error(args.model, args.serial), m, exc=e)
        MessageService.send(Topic.sensor(args.model, args.serial), "NG")

    finally:
        # WriterArgs に出力終了を指示
        # Instruct WriterArgs to end output
        args.write_end()

        # Writer 終了シーケンス
        # Writer termination sequence
        if not finish:
            # 正常終了でない場合、Reader からの終了通知を受けるまでキューから取得して空にする
            # If not normally terminated, retrieve from queue until termination notification
            # from Reader is received
            try:
                while True:
                    # この段階では出力に失敗している可能性が高いため、終了通知以外は無視する
                    # At this stage, it is likely that output has failed, so ignore anything
                    # other than termination notification
                    message: list[MeasureData] | None = queue.get()
                    if message is None:
                        break
            except Exception:
                logger.fatal("Failed to get reader process finish message")


def _writer_log(args: WriterArgs) -> logging.Logger:
    return logging.getLogger(f"{__name__}.{args.model}.{args.serial}.writer")


class JobError:
    """
    サブプロセスで発生したエラーについて、コントローラーでセンサーの特定が可能なように属性を付加したクラス

    Class to add attributes to errors occurring in subprocesses to allow the controller to identify the sensor
    """

    def __init__(self, model: str, serial: str, error: Exception) -> None:
        self.model = model
        self.serial = serial
        self.error = error
