# Copyright 2024, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.


import logging
import multiprocessing
from multiprocessing.sharedctypes import Synchronized
from typing import Any

from logger import Constants, InfoKeys, ProcessFactory
from logger.core import LoggerProcess, MessageService, Topic
from logger.utils.convert import to_uint16

from .comm import CommandList, WaitCommand, send
from .data import SelfTestResult
from .job import ReaderArgs, WriterArgs


class Sensor:
    """
    センサー抽象クラス

    以下を規定する：
    - センサーに求められる基本的な振る舞い
    - 個々のセンサーモデルに依らず共通な処理
    - Reader/Writer プロセスを構成し計測を行うメカニズムの提供

    Abstract class for sensors

    Defines the following:
    - Basic behavior required of sensors
    - Common processing regardless of individual sensor models
    - Mechanism to configure Reader/Writer processes and perform measurements
    """

    DEFAULT_START_COMMAND: CommandList = [
        [0, 0xFF, 0xFF, 0x0D],  # 復帰おまじない3回 / Recovery magic 3
        [0, 0xFF, 0xFF, 0x0D],
        [0, 0xFF, 0xFF, 0x0D],
        [0, 0xFE, 0x00, 0x0D],  # WINDOW_ID(L) write command.(WINDOW=0)
        [0, 0x83, 0x01, 0x0D],  # Automatic sampling mode
    ]
    DEFAULT_END_COMMAND: CommandList = [
        [0, 0xFE, 0x00, 0x0D],  # Window 0
        [0, 0x83, 0x02, 0x0D],  # MODE_CTRL: 1-0:10=ToConfig
        WaitCommand(0.01),  # Wait > 1ms
    ]
    DEFAULT_RECORD_BEGIN = 0x80
    DEFAULT_RECORD_END = 0x0D

    def __init__(
        self,
        model: str,
        port: str,
        baud: int,
        product_id: str,
        sps: int,
        logger_id: str,
        file_rotate_min: int,
        mode: str,
        physical: str,
        sensor_data_diag: bool,
    ) -> None:
        self._logger = logging.getLogger(__name__)
        self.model = model
        self.port = port
        self.baud = baud
        self.product_id = product_id
        self.serial = ""
        self.firm_ver = 0
        self.sps = sps
        self.logger_id = logger_id
        self.file_rotate_min = file_rotate_min
        self.mode = mode
        self.physical = physical
        self.sensor_data_diag = sensor_data_diag

        # センサーからデータを読み取る Reader プロセス
        # Reader process to read data from the sensor
        self._reader_process: LoggerProcess | None = None
        # 読み取ったデータを出力する Writer プロセス
        # Writer process to output the read data
        self._writer_process: LoggerProcess | None = None
        # Reader -> Writer にデータを転送するキュー
        # Queue to transfer data from Reader to Writer
        self._queue: multiprocessing.Queue | None = None
        # 計測状態を示すフラグ
        # Flag indicating measurement status
        self._measuring: Synchronized = multiprocessing.Value("b", False)  # type: ignore

    def __str__(self) -> str:
        return f"{self.model} #{self.serial} @{self.port}"

    def ready(self) -> bool:
        """
        センサーに異常がなく計測を開始できるか確認するメソッド

        Method to check if the sensor is free of abnormalities and ready to start measurement

        Returns:
            bool: センサーの異常有無
                    Whether the sensor is abnormal
        """
        commands: CommandList = [
            # W: SOFT_RST
            [0, 0xFE, 0x01, 0x0D],  # Window 1
            [0, 0x8A, 0x80, 0x0D],  # GLOB_CMD: 7:SOFT_RST=1
            WaitCommand(1.2),
            # R: GLOB_CMD
            [0, 0xFE, 0x01, 0x0D],  # Window 1
            [4, 0x0A, 0x00, 0x0D],  # R: GLOB_CMD -> 0-3byte
            # R: DIAG_STAT
            [0, 0xFE, 0x00, 0x0D],  # Window 0
            [4, 0x04, 0x00, 0x0D],  # R: DIAG_STAT -> 4-7byte
        ]

        # 送信
        # Send
        result = send(commands, self.port, self.baud)

        # 結果解析
        # Result analysis
        success = True

        # GLOB_CMD.NOT_READY[10] == 0
        # - ↑返却16bit中、10bit値を検査
        # - Check the 10th bit in the returned 16-bit value
        val = result[0:4][1]
        if (val & 0b00000100) != 0:
            success = False
            self._logger.error(f"Sensor device: NOT_READY ({bin(val)})")

        # DIAG.HARD_ERR[7-5] == 000
        # - ↑返却16bit中、7-5bit値を検査
        # - Check the 7-5th bits in the returned 16-bit value
        val = result[4:8][2]
        if (val & 0b11100000) != 0:
            success = False
            self._logger.error(f"Sensor device: HARD_ERR ({bin(val)})")

        return success

    def load_serial(self) -> None:
        """
        シリアルナンバーを読み込み、保持するメソッド

        Method to read and store the serial number
        """

        # シリアルコマンドリスト
        # Serial command list
        commands: CommandList = [
            [0, 0xFE, 0x01, 0x0D],  # Window 1
            [4, 0x74, 0x01, 0x0D],  # Serial_Number1
            [4, 0x76, 0x01, 0x0D],  # Serial_Number2
            [4, 0x78, 0x01, 0x0D],  # Serial_Number3
            [4, 0x7A, 0x01, 0x0D],  # Serial_Number4
        ]

        # コマンド送信
        # Send commands
        result = send(commands, self.port, self.baud)

        # シリアル文字列合成
        # - 返却は [Addr, Serial2, Serial1, CR] の構成
        # Serial string composition
        # - The return is structured as [Addr, Serial2, Serial1, CR]
        serial = ""
        for i in range(0, len(result), 4):
            serial += chr(result[i + 2])  # add Serial1
            serial += chr(result[i + 1])  # add Serial2

        # シリアルの保持
        # Store the serial
        self._logger.debug(f"load serial: {serial}")
        self.serial = serial

    def load_firm_ver(self) -> None:
        """
        ファームウェアバージョンを読み込み、保持するメソッド

        Method to read and store the firmware version
        """

        # ファームウェアコマンドリスト
        # Firmware command list
        commands: CommandList = [
            [0, 0xFE, 0x01, 0x0D],  # Window 1
            [4, 0x72, 0x00, 0x0D],  # FIRMVER
        ]

        # コマンド送信
        # Send command
        result = send(commands, self.port, self.baud)
        firm_ver = to_uint16(result[1], result[2])

        # ファームウェアバージョンの保持
        # Store the firmware version
        self._logger.debug(f"load firm_ver: {firm_ver}")
        self.firm_ver = firm_ver

    def init(self) -> None:
        """
        与えられた計測設定に基づきセンサーの計測情報を設定するメソッド

        サブクラスは具体的な処理を実装する

        Method to set sensor measurement information based on the given measurement settings

        Subclasses implement specific processing

        Raises:
            NotImplementedError: サブクラスで実装されていない場合
                    If not implemented in a subclass
        """
        raise NotImplementedError

    def start(
        self,
        error_queue: multiprocessing.Queue,
        output_dir: str,
        factory: ProcessFactory,
    ) -> None:
        """計測を開始するメソッド

        サブクラスはto_reader_argsと_to_writer_argsを実装すること

        Method to start measurement

        Subclasses should implement to_reader_args and to_writer_args

        Args:
            error_queue (multiprocessing.Queue): サブプロセス内で発生したエラーを伝えるQueue
                    Queue to communicate errors occurring within subprocesses
            output_dir (str): 計測情報の保存ディレクトリ
                    Directory to save measurement information
            factory (ProcessFactory): ファクトリオブジェクト、プロセスの生成に使用する
                    Factory object used to create processes
        """
        # 計測状態フラグ ON
        # Measurement status flag ON
        self._measuring.value = True

        # OK メッセージ送信
        # Send OK message
        MessageService.send(Topic.sensor(self.model, self.serial), "OK")

        # データ転送キュー生成
        # Create data transfer queue
        self._queue = multiprocessing.Queue(-1)

        # Reader/Writer プロセス生成
        # Create Reader/Writer processes
        self._reader_process = factory.create_reader_process(
            queue=self._queue,
            error_queue=error_queue,
            reader_args=self.to_reader_args(),
            measuring=self._measuring,
        )
        self._writer_process = factory.create_writer_process(
            queue=self._queue,
            error_queue=error_queue,
            writer_args=self.to_writer_args(),
            output_dir=output_dir,
        )

        # Reader/Writer プロセス開始
        # Start Reader/Writer processes
        self._reader_process.start()
        self._writer_process.start()

    def stop(self) -> None:
        """
        計測を終了するメソッド

        Method to stop measurement
        """
        # 計測状態フラグ OFF
        # Measurement status flag OFF
        self._measuring.value = False

        # Reader/Writer プロセス停止
        # - キューから全てのデータを消費する必要があるため、先に Writer を終了待ちする
        # Stop Reader/Writer processes
        # - Wait for Writer to finish first as all data needs to be consumed from the queue
        if self._writer_process:
            self._writer_process.join()
            self._writer_process = None

        if self._reader_process:
            self._reader_process.join()
            self._reader_process = None

        # キューのクローズ
        # Close the queue
        if self._queue:
            self._queue.close()
            self._queue = None

    def add_info(self, info_d: dict[str, Any]) -> None:
        """
        計測情報ファイルへの出力用に設定項目を登録する

        このクラスでは以下の情報項目を登録する：
        - PORT, SENSOR, PRODUCT_ID, SERIAL_NO, FIRM_VER, OUTPUT_TYPE, SPS

        Register configuration items for output to the measurement information file

        This class registers the following information items:
        - PORT, SENSOR, PRODUCT_ID, SERIAL_NO, FIRM_VER, OUTPUT_TYPE, SPS

        Args:
            info_d (dict): 情報登録用辞書
                    Dictionary for registering information
                - 登録する値は最終的に文字列化されるため、数値型のまま登録も可能
                  Values to be registered will eventually be converted to strings,
                  so they can be registered as numeric types
        """
        info_d.update(
            {
                InfoKeys.PORT: self.port,
                InfoKeys.SENSOR: self.model,
                InfoKeys.PRODUCT_ID: self.product_id,
                InfoKeys.SERIAL_NO: self.serial,
                InfoKeys.FIRM_VER: f"{self.firm_ver:#06x}",
                InfoKeys.PHYSICAL: self.physical,
                InfoKeys.OUTPUT_TYPE: self.mode,
                InfoKeys.SPS: self.sps,
            }
        )

    def to_reader_args(self) -> ReaderArgs:
        """
        Reader プロセスに渡す設定情報に変換するメソッド

        サブクラスでオブジェクトの持つ情報から設定情報に変換する処理を実装する

        Method to convert to configuration information passed to the Reader process

        Subclasses implement the process of converting object information to configuration information

        Raises:
            NotImplementedError: サブクラスで実装されていない場合
                    If not implemented in a subclass

        Returns:
            ReaderArgs: Reader プロセスに渡す設定情報
                    Configuration information passed to the Reader process
        """
        raise NotImplementedError

    def to_writer_args(self) -> WriterArgs:
        """
        Writer プロセスに渡す設定情報に変換するメソッド

        サブクラスでオブジェクトの持つ情報から設定情報に変換する処理を実装する

        Method to convert to configuration information passed to the Writer process

        Subclasses implement the process of converting object information to configuration information

        Raises:
            NotImplementedError: サブクラスで実装されていない場合
                    If not implemented in a subclass

        Returns:
            WriterArgs: Writer プロセスに渡す設定情報
                    Configuration information passed to the Writer process
        """
        raise NotImplementedError

    def self_test(self) -> SelfTestResult:
        """
        セルフテストを実行するメソッド

        Method to execute a self-test

        Raises:
            NotImplementedError: サブクラスで実装されていない場合
                    If not implemented in a subclass
        """
        raise NotImplementedError

    def _get_record_per_file(self) -> int:
        """
        １ファイルに出力する行数を計算するメソッド

        サブクラスの属性に応じて計算処理を実装する

        Method to calculate the number of rows to output per file

        Implement the calculation process according to the attributes of the subclass

        Raises:
            NotImplementedError: サブクラスで実装されていない場合
                    If not implemented in a subclass

        Returns:
            int: １ファイルに出力する行数
                    Number of rows to output per file
        """
        raise NotImplementedError

    @staticmethod
    def load_product_id(port: str, baud: int) -> str:
        """
        指定されたポートからproduct_idを読み込むメソッド

        Method to read the product_id from the specified port

        Args:
            port (str): 通信対象のポート
                    Port to communicate with

        Raises:
            ConnectionRefusedError: 通信ボーレート全てで通信か失敗した場合
                    If communication fails at all baud rates

        Returns:
            str: 読み込んだproduct_id
                    The read product_id
        """
        logger = logging.getLogger(f"{__name__}.load_product")
        command_list: CommandList = [
            [0, 0xFF, 0xFF, 0x0D],  # 復帰おまじない3回 / Recovery magic 3
            [0, 0xFF, 0xFF, 0x0D],
            [0, 0xFF, 0xFF, 0x0D],
            [0, 0xFE, 0x01, 0x0D],  # Window 1
            [4, 0x6A, 0x00, 0x0D],  # Product ID 1
            [4, 0x6C, 0x00, 0x0D],  # Product ID 2
            [4, 0x6E, 0x00, 0x0D],  # Product ID 3
            [4, 0x70, 0x00, 0x0D],  # Product ID 4
        ]

        # コマンド送信
        # Send status
        try:
            result = send(command_list, port, baud, timeout=1)

            # 製品型番文字列合成
            # - 返却は [Addr, Prod2, Prod1, CR] の構成
            # Compose product ID string
            # - The return is structured as [Addr, Prod2, Prod1, CR]
            product_id = ""
            for i in range(0, len(result), 4):
                product_id += chr(result[i + 2])  # add Prod1
                product_id += chr(result[i + 1])  # add Prod2
            logger.debug(f"load product id: {product_id}")
            return product_id

        except TimeoutError:
            logger.debug(f"Failed: port={port} baud={baud}")

        # 通信失敗
        # Communication failure
        raise ConnectionRefusedError(f"Failed to connect: {port}")

    @staticmethod
    def _determine_test_bit(value: int, mask: int) -> str:
        """
        セルフテストの値を判定するためのメソッド

        Method to determine the value of a self-test

        Args:
            value (int): 判定を行うセルフテストの結果8bit整数
                    8-bit integer result of the self-test to be determined
            mask (int): 判定bitのみを判断するためにつけるマスク整数
                    Mask integer to determine only the test bit

        Returns:
            str: OKorNGの判定結果
                    The determination result of OK or NG
        """
        result = value & mask
        return Constants.TEST_OK if result == 0 else Constants.TEST_NG


class SensorDeviceError(Exception):
    pass
