# Copyright 2024, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

import logging
import os
import re
from typing import Any


class Validator:
    """
    バリデーション実行クラス

    Validation execution class
    """

    def __init__(self) -> None:
        self.logger = logging.getLogger(__name__)

    def defined(self, value: Any, key: str) -> bool:
        """
        検証対象が存在するか（Noneではないか）を検証するメソッド

        Method to check if the target exists (is not None)

        Args:
            value (Any): 検証対象
                    The target to be validated
            key (str): 検証対象の名前（失敗時の出力に利用）
                    The name of the target (used for output on failure)

        Returns:
            bool: 検証に成功したか
                    Whether the validation was successful
        """
        result = value is not None
        if not result:
            self.logger.error(f"{key} must be defined")
        return result

    def one_of(self, value: Any, key: str, list: list[Any]) -> bool:
        """
        検証対象が与えられた配列に含まれるかを検証するメソッド

        Method to check if the target is included in the given list

        Args:
            value (Any): 検証対象
                    The target to be validated
            key (str): 検証対象の名前（失敗時の出力に利用）
                    The name of the target (used for output on failure)
            list (list[Any]): 検証対象が含まれるべき配列
                    The list in which the target should be included

        Returns:
            bool: 検証に成功したか
                    Whether the validation was successful
        """
        result = self.defined(value=value, key=key) and value in list
        if not result:
            self.logger.error(f"{key} must be one of {','.join(map(str, list))}")
        return result

    def is_bool(self, value: Any, key: str) -> bool:
        """
        検証対象が bool 値として評価できるかを検証するメソッド

        Method to check if the target can be evaluated as a boolean value

        Args:
            value (Any): 検証対象
                    The target to be validated
            key (str): 検証対象の名前（失敗時の出力に利用）
                    The name of the target (used for output on failure)

        Returns:
            bool: 検証に成功したか
                    Whether the validation was successful
        """
        return self.one_of(value, key, [True, False])

    def between(self, value: Any, key: str, min: Any, max: Any) -> bool:
        """
        検証対象が与えられた最小値と最大値の間に含まれるかを検証するメソッド

        Method to check if the target is between the given minimum and maximum values

        Args:
            value (Any): 検証対象
                    The target to be validated
            key (str): 検証対象の名前（失敗時の出力に利用）
                    The name of the target (used for output on failure)
            min (Any): 値の許容する最小値
                    The minimum acceptable value
            max (Any): 値の許容する最大値
                    The maximum acceptable value

        Returns:
            bool: 検証に成功したか
                    Whether the validation was successful
        """
        result = self.defined(value=value, key=key) and min <= value and value <= max
        if not result:
            self.logger.error(f"{key} must be between {min} and {max}")
        return result

    def greater_than(self, value: Any, key: str, min: Any) -> bool:
        """
        検証対象が与えられた最小値より大きいか検証するメソッド

        Method to check if the target is greater than the given minimum value

        Args:
            value (Any): 検証対象
                    The target to be validated
            key (str): 検証対象の名前（失敗時の出力に利用）
                    The name of the target (used for output on failure)
            min (Any): 値が超えていなければいけない値
                    The value that the target must exceed

        Returns:
            bool: 検証に成功したか
                    Whether the validation was successful
        """
        result = self.defined(value=value, key=key) and min < value
        if not result:
            self.logger.error(f"{key} must be greater than {min}")
        return result

    def less_than_or_equal_to(self, value: Any, key: str, max: Any) -> bool:
        """
        検証対象が与えられた最大値以下か検証するメソッド

        Method to check if the target is less than or equal to the given maximum value

        Args:
            value (Any): 検証対象
                    The target to be validated
            key (str): 検証対象の名前（失敗時の出力に利用）
                    The name of the target (used for output on failure)
            max (Any): 値の最大値
                    The maximum value

        Returns:
            bool: 検証に成功したか
                    Whether the validation was successful
        """
        result = self.defined(value=value, key=key) and max >= value
        if not result:
            self.logger.error(f"{key} must be less than or equal to {max}")
        return result

    def match(self, value: Any, key: str, pattern: str) -> bool:
        """
        検証対象が与えられた正規表現にマッチするかを検証するメソッド

        Method to check if the target matches the given regular expression

        Args:
            value (Any): 検証対象
                    The target to be validated
            key (str): 検証対象の名前（失敗時の出力に利用）
                    The name of the target (used for output on failure)
            pattern (str): 検証する正規表現
                    The regular expression to validate against

        Returns:
            bool: 検証に成功したか
                    Whether the validation was successful
        """
        result = (
            self.defined(value=value, key=key) and re.match(pattern, value) is not None
        )
        if not result:
            self.logger.error(f"{key} must match pattern: {pattern}")
        return result

    def constraint(self, value: Any, key: str, test: bool, desc: str) -> bool:
        """
        検証対象の制約が成立するかを検証するメソッド

        Method to check if the target satisfies the given constraint

        Args:
            value (Any): 検証対象
                    The target to be validated
            key (str): 検証対象の名前（失敗時の出力に利用）
                    The name of the target (used for output on failure)
            test (bool): 制約として判定された結果（boolで判定できる結果を与える）
                    The result of the constraint check (a boolean result)
            desc (str): test の内容を説明する文字列（失敗時の出力に利用）
                    A description of the test (used for output on failure)

        Returns:
            bool: 検証に成功したか
                    Whether the validation was successful
        """
        result = self.defined(value=value, key=key) and test
        if not result:
            self.logger.error(f"{key} must satisfy constraint: {desc}")
        return result

    def exist_dir(self, path: Any, key: str) -> bool:
        """
        検証対象が実際に存在するディレクトリかを検証するメソッド

        Method to check if the target is an existing directory

        Args:
            path (Any): 検証対象
                    The target to be validated
            key (str): 検証対象の名前（失敗時の出力に利用）
                    The name of the target (used for output on failure)

        Returns:
            bool: 検証に成功したか
                    Whether the validation was successful
        """
        result = self.defined(value=path, key=key) and os.path.isdir(path)
        if not result:
            self.logger.error(f"{key} must be an existing directory: {path}")
        return result
