# Copyright 2025, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.


from logger import Constants
from logger.measure import A370
from logger.measure.data import SelfTestResult


class TestA370:
    class TestSelfTest:
        _void = [0x00]
        _cont = [0x02, 0xFF, 0x0D]
        _done = [0x02, 0x00, 0x0D]
        _diag_0 = [0x04, 0x00, 0x00, 0x0D]  # DIAG_STAT 0

        _a370 = A370(
            port="/dev/AAA",
            baud=100,
            product_id="A370XXX",
            sps=1000,
            logger_id="hoge",
            file_rotate_min=1,
            sensor_data_diag=True,
            sensor_data_diag_sec=0.1,
        )

        def _ok_result(self) -> SelfTestResult:
            return SelfTestResult(
                model=self._a370.model,
                serial=self._a370.serial,
                acceleration=Constants.TEST_OK,
                sensitivity=f"X:{Constants.TEST_OK},Y:{Constants.TEST_OK},Z:{Constants.TEST_OK}",
                temperature=Constants.TEST_OK,
                power_voltage=Constants.TEST_OK,
                flash_memory=Constants.TEST_OK,
                structural_resonance=Constants.TEST_NOT_AVAILABLE,
            )

        def test_self_test_ok(self, mock_comm):
            send_sidefx = [
                self._void,  # 0b00000111 : acceleration, temperature, power voltage
                self._cont,
                self._done,
                self._diag_0,
                self._void,  # 0b00001000 : non-volatile memory
                self._cont,
                self._done,
                self._diag_0,
                self._void,  # 0b00010000 : sensitivity
                self._cont,
                self._done,
                self._diag_0,
            ]
            mock_comm(send_sidefx=send_sidefx)

            # Execute
            result = self._a370.self_test()

            # Judge
            assert result == self._ok_result()

        def test_self_test_ng_ptn1(self, mock_comm):
            send_sidefx = [
                self._void,  # 0b00000111 : acceleration, temperature, power voltage
                self._done,
                [0x04, 0x00, 0b00000010, 0x0D],  # DIAG_STAT1 ACC_ERR_ALL=1
                self._void,  # 0b00001000 : non-volatile memory
                self._done,
                [0x04, 0x00, 0b00000100, 0x0D],  # DIAG_STAT1 FLASH_ERR=1
                self._void,  # 0b00010000 : sensitivity
                self._done,
                [0x04, 0x00, 0b00000001, 0x0D],  # DIAG_STAT2 X_SENS_ERR=01
            ]
            mock_comm(send_sidefx=send_sidefx)

            # Execute
            result = self._a370.self_test()

            # Judge
            expect = self._ok_result()
            expect.acceleration = Constants.TEST_NG
            expect.flash_memory = Constants.TEST_NG
            expect.sensitivity = "X:{},Y:{},Z:{}".format(
                Constants.TEST_NG,
                Constants.TEST_OK,
                Constants.TEST_OK,
            )

            assert result == expect

        def test_self_test_ng_ptn2(self, mock_comm):
            send_sidefx = [
                self._void,  # 0b00000111 : acceleration, temperature, power voltage
                self._done,
                [0x04, 0b00000010, 0x00, 0x0D],  # DIAG_STAT TEMP_ERR=1
                self._void,  # 0b00001000 : non-volatile memory
                self._done,
                self._diag_0,
                self._void,  # 0b00010000 : sensitivity
                self._done,
                [0x04, 0x00, 0b00001000, 0x0D],  # DIAG_STAT2 Y_SENS_ERR=10
            ]
            mock_comm(send_sidefx=send_sidefx)

            # Execute
            result = self._a370.self_test()

            # Judge
            expect = self._ok_result()
            expect.temperature = Constants.TEST_NG
            expect.sensitivity = "X:{},Y:{},Z:{}".format(
                Constants.TEST_OK,
                Constants.TEST_NG,
                Constants.TEST_OK,
            )

            assert result == expect

        def test_self_test_ng_ptn3(self, mock_comm):
            send_sidefx = [
                self._void,  # 0b00000111 : acceleration, temperature, power voltage
                self._done,
                [0x04, 0b00000001, 0x00, 0x0D],  # DIAG_STAT VDD_ERR=1
                self._void,  # 0b00001000 : non-volatile memory
                self._done,
                self._diag_0,
                self._void,  # 0b00010000 : sensitivity
                self._done,
                [0x04, 0x00, 0b00110000, 0x0D],  # DIAG_STAT2 Z_SENS_ERR=11
            ]
            mock_comm(send_sidefx=send_sidefx)

            # Execute
            result = self._a370.self_test()

            # Judge
            expect = self._ok_result()
            expect.power_voltage = Constants.TEST_NG
            expect.sensitivity = "X:{},Y:{},Z:{}".format(
                Constants.TEST_OK,
                Constants.TEST_OK,
                Constants.TEST_NG,
            )

            assert result == expect
