# Copyright 2024, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

import json
import logging
import time
from dataclasses import dataclass
from datetime import datetime
from enum import Enum, auto
from multiprocessing import Queue
from traceback import format_exception_only
from typing import Any, Type, cast

import paho.mqtt.client as mqtt

from .config import Config
from .service import ServiceBase


class Topic:
    """
    メッセージのトピックを表すクラス

    種々のトピックを生成するクラスメソッドを備える。

    Class representing a message topic

    Provides class methods to generate various topics
    """

    class Content(Enum):
        """
        Topic の内容種類を表す定数

        種類に応じてメッセージのデータ構成が決定される：
        - STATUS: timestamp, status(str)
        - MESSAGE: timestamp, message(str)
        - ERROR: timestamp, level(str), message(str)
        - SPECIAL: timestamp, dict に要素を格納して渡す

        Constants representing the types of content in a Topic

        The structure of message data is determined by the type:
        - STATUS: timestamp, status(str)
        - MESSAGE: timestamp, message(str)
        - ERROR: timestamp, level(str), message(str)
        - SPECIAL: timestamp, elements stored and passed in a dict
        """

        STATUS = auto()
        MESSAGE = auto()
        ERROR = auto()
        SPECIAL = auto()

    def __init__(self, topic: str, content: Content) -> None:
        self.topic = topic
        self.content = content

    def __str__(self) -> str:
        return self.topic

    def __eq__(self, value) -> bool:
        if isinstance(value, Topic):
            return self.topic == value.topic and self.content == value.content
        else:
            return False

    @classmethod
    def logger(cls) -> "Topic":
        return Topic(f"{cls._root_elem}/{cls._logger_id}", Topic.Content.STATUS)

    @classmethod
    def logger_error(cls) -> "Topic":
        return Topic(f"{cls.logger()}/error", Topic.Content.ERROR)

    @classmethod
    def sensor(cls, model: str, serial: str) -> "Topic":
        return Topic(f"{cls.logger()}/sensor/{model}/{serial}", Topic.Content.STATUS)

    @classmethod
    def sensor_error(cls, model: str, serial: str) -> "Topic":
        return Topic(f"{cls.sensor(model, serial)}/error", Topic.Content.ERROR)

    @classmethod
    def sensor_lost(cls, model: str, serial: str) -> "Topic":
        return Topic(f"{cls.sensor(model, serial)}/lost", Topic.Content.MESSAGE)

    @classmethod
    def sensor_abnormal(cls, model: str, serial: str) -> "Topic":
        return Topic(f"{cls.sensor(model, serial)}/abnormal", Topic.Content.MESSAGE)

    # 制御用項目
    # Control items
    _root_elem: str | None = None
    _logger_id: str | None = None

    @classmethod
    def _init_conf(cls, root_elem: str, logger_id: str) -> None:
        cls._root_elem = root_elem
        cls._logger_id = logger_id


class MessageService(ServiceBase):
    """
    状態メッセージの送信を提供するサービス

    マルチプロセス環境、シングルプロセス環境の両方に対応し、
    メッセージブローカーへの接続、メッセージの送信を行う。

    Service providing the transmission of status messages

    Supports both multi-process and single-process environments,
    connecting to message brokers and sending messages.
    """

    @dataclass
    class Message:
        topic: Topic
        data: dict

    class ErrorLevel(Enum):
        WARNING = auto()
        ERROR = auto()
        CRITICAL = auto()

    @classmethod
    def send(
        cls,
        topic: Topic,
        data: str | dict,
        *,
        lvl: ErrorLevel = ErrorLevel.ERROR,
        exc: BaseException | None = None,
    ) -> None:
        """
        指定トピックにメッセージを送信するメソッド

        Method to send a message to the specified topic

        Args:
            topic (Topic): 送信先のトピックを表す Topic オブジェクト
                    Topic object representing the destination topic
            data (str | dict): メッセージとして送信するデータ
                    Data to be sent as a message
                - topic.content の種類に応じて送信するデータ型が異なる
                    The data type to be sent varies depending on the type of topic.content
            lvl (ErrorLevel): topic.content が ERROR の場合、エラーレベルを示す
                    Indicates the error level if topic.content is ERROR
            exc (BaseException): topic.content が ERROR の場合、例外を受け渡す
                    Passes the exception if topic.content is ER
        """
        # timestamp はすべてに付加する
        # Add timestamp to all
        datad: dict = {"timestamp": datetime.now().isoformat(" ")}

        # content に応じてメッセージを構成
        # Construct the message according to the content
        match topic.content:
            case Topic.Content.STATUS:
                datad["status"] = str(data)
            case Topic.Content.MESSAGE:
                datad["message"] = str(data)
            case Topic.Content.ERROR:
                datad["level"] = lvl.name
                datad["message"] = f"{data}: {repr(exc)}" if exc else str(data)
            case Topic.Content.SPECIAL:
                assert isinstance(data, dict)
                datad.update(data)

        # メッセージ送信
        # Send message
        cls._send_core(MessageService.Message(topic=topic, data=datad))

    class Server(ServiceBase.Server):
        """
        MessageService のサーバークラス

        Server class for MessageService
        """

        _QOS: int = 1

        def __init__(self, config: Config, root_elem: str) -> None:
            self._message_host: str | None = None
            self._message_port: int | None = None
            self._logger_id: str | None = None
            self._root_elem: str | None = None
            self._mqttc: mqtt.Client | None = None

            # 設定で有効な場合のみ保持する
            # Retain only if enabled in the configuration
            if config.message_send:
                self._message_host = config.message_host
                self._message_port = config.message_port
                self._logger_id = config.logger_id
                self._root_elem = root_elem

        def start(self) -> None:
            """
            サービスを開始するメソッド

            メッセージブローカーへの接続と過去メッセージの削除を行う

            Method to start the service

            Connects to the message broker and deletes past messages
            """
            logger = MessageService._logger
            # * この logger のレベルを DEBUG に設定すると paho のログも出力される
            # * Setting this logger's level to DEBUG will also output paho logs
            # logger.setLevel(logging.DEBUG)

            # 設定で有効な場合のみ処理する
            # Process only if enabled in the configuration
            if not (self._message_host and self._message_port):
                logger.debug("Skipped to connect message broker")
                return

            # Topic 初期化
            # Initialize Topic
            assert self._root_elem and self._logger_id
            Topic._init_conf(self._root_elem, self._logger_id)

            # MQTT クライアント構築
            # Build MQTT client
            self._mqttc = mqtt.Client(mqtt.CallbackAPIVersion.VERSION2)
            self._mqttc.on_connect = MessageService.Server._on_connect
            self._mqttc.on_message = MessageService.Server._on_message

            # - 接続失敗時に Server 操作が必要なため、userdata 登録する
            # - Register userdata for Server operations in case of connection failure
            self._mqttc.user_data_set({"server": self})

            # - DEBUG モードの場合は MQTT のログも出力する
            # - Output MQTT logs in DEBUG mode
            if logger.getEffectiveLevel() == logging.DEBUG:
                self._mqttc.enable_logger(logger)

            # - 接続
            # - Connect
            try:
                self._mqttc.connect(self._message_host, self._message_port)

            except Exception as e:
                # - 接続失敗： 無効化
                # - Connection failure: disable
                logger.warning(
                    f"Failed to connect message broker: {format_exception_only(e)}"
                )
                self._invalidate()

            else:
                # - ループ開始
                # - Start loop
                self._mqttc.loop_start()
                # - 少し待つ
                # - Wait a bit
                time.sleep(1)

        @staticmethod
        def _on_connect(client: mqtt.Client, userdata: dict, flags, rc, props):
            """
            MQTT 接続コールバック

            MQTT connection callback
            """

            logger = MessageService._logger
            logger.debug(f"At on_connect: flags={flags}, rc={rc}")
            if rc == mqtt.MQTT_ERR_SUCCESS:
                logger.info("Connected to message broker")
            else:
                # ** on_connect は失敗時にも呼ばれうるが、このサービスは失敗時は継続しないため破棄する
                # ** on_connect can be called on failure, but this service does not continue on failure, so it is discarded
                logger.warning(
                    f"Failed to connect message broker: rc={rc}, "
                    + "messaging is disabled and measurement will continue."
                )

                # - 接続に失敗した場合は再試行させずに即切断する
                # - Disconnect immediately without retrying if connection fails
                client.disconnect()

                # - Server を無効化する
                # - Disable the server
                server = cast(MessageService.Server, userdata.get("server"))
                server._invalidate()
                return

            if not MessageService._retain_removed:
                # 初回接続時、このインスタンスに属する全 retain メッセージの subscribe
                # - 過去のメッセージを削除するために受信する
                # Subscribe to all retain messages belonging to this instance on first connection
                # - Receive to delete past messages
                assert Topic._root_elem
                topic = str(Topic.logger()) + "/#"
                client.subscribe(topic)

                # - 前の命令で全て送信されるため、すぐに unsubscribe する
                # - Unsubscribe immediately as all will be sent by the previous command
                client.unsubscribe(topic)

                # - 削除処理済みフラグ
                # - Deletion process completed flag
                MessageService._retain_removed = True

        @staticmethod
        def _on_message(client: mqtt.Client, userdata, msg: mqtt.MQTTMessage):
            """
            MQTT 受信コールバック

            MQTT receive callback
            """

            # 過去メッセージの削除
            # Delete past messages
            assert msg.retain
            client.publish(
                topic=msg.topic,
                payload=None,
                retain=True,
                qos=MessageService.Server._QOS,
            )

            MessageService._logger.debug(f"Publish to remove message: {msg.topic}")

        def close(self, **kwargs) -> None:
            """
            サービスを終了するメソッド

            メッセージブローカーへの接続を閉じる

            Method to stop the service

            Closes the connection to the message broker
            """
            if self._mqttc:
                self._mqttc.loop_stop()
                self._invalidate()

        def declare_type(self) -> list[Type]:
            """
            このサーバーが処理できるデータ型を宣言するメソッド

            Method to declare the data types this server can handle
            """
            return [MessageService.Message]

        def handle_data(self, data: Any) -> None:
            """
            データに対する処理を実行するメソッド

            Method to process the data
            """
            if not self._mqttc:
                return

            logger = MessageService._logger
            msg = cast(MessageService.Message, data)

            # メッセージの publish
            # Publish the message
            logger.debug(f"Publish message: topic={msg.topic}, data={msg.data}")
            inf = self._mqttc.publish(
                topic=str(msg.topic),
                payload=json.dumps(msg.data),
                retain=True,
                qos=self._QOS,
            )

            # ** 以下のメソッドを実行すると、ブローカーと接続が切れている場合に例外が上がる
            # -> 代わりに qos=1 に設定し、再接続時に再送信されるようにする
            # ** Executing the following method raises an exception if disconnected from the broker
            # -> Instead, set qos=1 to resend on reconnection

            # inf.wait_for_publish()

            if inf.rc != mqtt.MQTT_ERR_SUCCESS:
                logger.warning(
                    "Failed to publish message: topic={}, data={}, rc={}".format(
                        msg.topic, msg.data, inf.rc
                    )
                )

        def _invalidate(self) -> None:
            """
            構築の過程で以降の使用を無効化するメソッド

            Method to disable further use during the construction process
            """
            self._mqttc = None

    class Client(ServiceBase.Client):
        """
        MessageService のクライアントクラス

        Client class for MessageService
        """

        def __init__(self, config: Config, root_elem: str) -> None:
            self._root_elem: str | None = None
            self._logger_id: str | None = None
            self._queue: Queue | None = None
            self._server: MessageService.Server | None = None

            # 設定で有効な場合のみ保持する
            # Retain only if enabled in the configuration
            if config.message_send:
                self._root_elem = root_elem
                self._logger_id = config.logger_id

        def setup(self, **kwargs) -> None:
            """
            サービスを利用するクライアントプロセスを設定するメソッド

            使用する環境に応じて設定を行う

            Method to set up the client process using the service

            Configure according to the environment used

            Args:
                kwargs (dict): 名前付き可変引数を格納した dict
                    Dict containing named variable arguments
                    - "queue" が格納されていた場合、サーバー転送用 Queue として使用する
                      If "queue" is stored, use it as a queue for server transfer
                    - "server" が格納されていた場合、直結サーバーとして使用する
                      If "server" is stored, use it as a direct server
            """

            # 設定で有効な場合のみ処理する
            # Process only if enabled in the configuration
            if not (self._root_elem and self._logger_id):
                return

            # 自身をサービスに登録する
            # Register itself to the service
            MessageService._client = self

            # 起動環境に応じたパラメータを取得する
            # Get parameters according to the startup environment
            self._queue = kwargs.get("queue")
            self._server = kwargs.get("server")

            # - どちらかしか有効でないこと
            # - Only one should be valid
            assert (self._queue or self._server) and not (self._queue and self._server)

            # Topic 初期化
            # Initialize Topic
            Topic._init_conf(self._root_elem, self._logger_id)

    # Client オブジェクト - クラスレベルで保持し、メッセージ処理を媒介する
    # Client object - held at the class level to mediate message processing
    _client: "MessageService.Client | None" = None

    # retain 削除フラグ
    # retain deletion flag
    _retain_removed: bool = False

    _logger: logging.Logger = logging.getLogger(__name__)

    def __init__(self, config: Config, root_elem: str) -> None:
        self._config = config
        self._root_elem = root_elem

    def create_server(self) -> ServiceBase.Server:
        """
        Server オブジェクトを生成するメソッド

        Method to create a Server object
        """
        return MessageService.Server(self._config, self._root_elem)

    def create_client(self) -> ServiceBase.Client:
        """
        Client オブジェクトを生成するメソッド

        Method to create a Client object
        """
        return MessageService.Client(self._config, self._root_elem)

    @classmethod
    def _send_core(cls, message: Message) -> None:
        """
        メッセージ送信 コア

        Core for sending messages
        """
        # Client がいない場合は処理しない
        # Do not process if there is no Client
        if not cls._client:
            return

        if cls._client._queue:
            cls._client._queue.put(message)
        if cls._client._server:
            cls._client._server.handle_data(message)
