# Copyright 2024, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

from __future__ import annotations

from multiprocessing import Queue
from multiprocessing.sharedctypes import Synchronized
from typing import TYPE_CHECKING

if TYPE_CHECKING:
    from logger.core import Config, Configurator, LoggerProcess
    from logger.measure import ReaderArgs, Sensor, WriterArgs


class ProcessFactory:
    """
    ロガープログラム内で使われる主要な要素に対するファクトリクラスのうち、Process 生成を行うクラス

    様々な適用先への対応を容易にする目的で、以下の要素の生成を拡張可能にする：
    - Sensor Reader Process
    - Sensor Writer Process

    このクラスは循環 import 問題を回避する目的で作成されたインタフェースクラスであり、
    このモジュール内の LoggerFactory インタフェースクラス、
    その実体である logger.core.factory.LoggerFactoryImpl を参照。

    Class for creating processes among the factory classes for major elements used in the logger program

    To facilitate adaptation to various applications, the following elements can be extended:
    - Sensor Reader Process
    - Sensor Writer Process

    This class is an interface created to avoid circular import issues,
    referring to the LoggerFactory interface class in this module,
    and its implementation, logger.core.factory.LoggerFactoryImpl.
    """

    def create_reader_process(
        self,
        queue: Queue,
        error_queue: Queue,
        reader_args: ReaderArgs,
        measuring: Synchronized,
    ) -> LoggerProcess:
        """
        Sensor Reader プロセスのファクトリメソッド

        引数で受け取っているのはデフォルトの reader_job で使うもの。
        reader_job を置き換える場合には、必要に応じてこのメソッド内で変更してよい。

        Factory method for the Sensor Reader process.

        The arguments received are used for the default reader_job.
        If replacing reader_job, it can be modified within this method as needed.

        Args:
            queue (Queue): 計測データを Writer に送信するキュー
                    Queue to send measurement data to the Writer
            error_queue (Queue): エラー発生時にコントローラーに通知するキュー
                    Queue to notify the controller in case of errors
            reader_args (ReaderArgs): 計測設定情報を保持するオブジェクト
                    Object holding measurement configuration information
            measuring (Synchronized): 計測中フラグ
                    Flag indicating measurement in progress

        Returns:
            Reader 機能を持った LoggerProcess オブジェクト
            LoggerProcess object with Reader functionality
        """
        raise NotImplementedError

    def create_writer_process(
        self,
        queue: Queue,
        error_queue: Queue,
        writer_args: WriterArgs,
        output_dir: str,
    ) -> LoggerProcess:
        """
        Sensor Writer プロセスのファクトリメソッド

        引数で受け取っているのはデフォルトの writer_job で使うもの。
        writer_job を置き換える場合には、必要に応じてこのメソッド内で変更してよい。

        Factory method for the Sensor Writer process

        The arguments received are used for the default writer_job.
        If replacing writer_job, it can be modified within this method as needed.

        Args:
            queue (Queue): 計測データを Reader から受信するキュー
                    Queue to receive measurement data from the Reader
            error_queue (Queue): エラー発生時にコントローラーに通知するキュー
                    Queue to notify the controller in case of errors
            writer_args (WriterArgs): 計測設定情報を保持するオブジェクト
                    Object holding measurement configuration information
            output_dir (str): 計測情報の保存ディレクトリ
                    Directory to save measurement information

        Returns:
            Writer 機能を持った LoggerProcess オブジェクト
            LoggerProcess object with Writer functionality
        """
        raise NotImplementedError


class LoggerFactory(ProcessFactory):
    """
    ロガープログラム内で使われる主要な要素に対するファクトリクラス

    様々な適用先への対応を容易にする目的で、以下の要素の生成を拡張可能にする：
    - 具象 Sensor
    - Sensor Reader Process
    - Sensor Writer Process
    - 設定情報読み込み Configurator

    このクラスは循環 import 問題を回避する目的で作成されたインタフェースクラスであり、
    実体は logger.core.factory.LoggerFactoryImpl を参照。

    Factory class for major elements used in the logger program

    To facilitate adaptation to various applications, the following elements can be extended:
    - Concrete Sensor
    - Sensor Reader Process
    - Sensor Writer Process
    - Configurator for reading configuration information

    This class is an interface created to avoid circular import issues,
    and its implementation is logger.core.factory.LoggerFactoryImpl.
    """

    def set_config(self, config: Config) -> None:
        """
        Config オブジェクトの設定

        Setting the Config object

        Args:
            config (Config):  設定情報を保持したオブジェクト
                    Object holding configuration information
        """
        raise NotImplementedError

    def create_configurator(self) -> Configurator:
        """
        設定項目を読み込み Config オブジェクトを生成する Configurator のファクトリメソッド

        Factory method for the Configurator that reads configuration items and generates a Config object

        Returns:
            Configurator object
        """
        raise NotImplementedError

    def create_A342(self, port: str, product_id: str) -> Sensor:
        """
        A342 センサーのファクトリメソッド

        Factory method for the A342 sensor

        Args:
            port (str): センサーが接続されたポート
                    Port to which the sensor is connected
            product_id (str): センサーから取得した PRODUCT_ID
                    PRODUCT_ID obtained from the sensor

        Returns:
            A342 sensor object
        """
        raise NotImplementedError

    def create_A352(self, port: str, product_id: str) -> Sensor:
        """
        A352 センサーのファクトリメソッド

        Factory method for the A352 sensor

        Args:
            port (str): センサーが接続されたポート
                    Port to which the sensor is connected
            product_id (str): センサーから取得した PRODUCT_ID
                    PRODUCT_ID obtained from the sensor

        Returns:
            A352 sensor object
        """
        raise NotImplementedError

    def create_A370(self, port: str, product_id: str) -> Sensor:
        """
        A370 センサーのファクトリメソッド

        Factory method for the A370 sensor

        Args:
            port (str): センサーが接続されたポート
                    Port to which the sensor is connected
            product_id (str): センサーから取得した PRODUCT_ID
                    PRODUCT_ID obtained from the sensor

        Returns:
            A370 sensor object
        """
        raise NotImplementedError
