# Copyright 2024, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

import math
import time
from typing import Any, TypedDict

from logger import Constants, InfoKeys
from logger.utils.convert import to_dec24, to_int8, to_int16, to_uint16

from .comm import Comm, WaitCommand
from .data import MeasureData, SelfTestResult
from .job import ReaderArgs, WriterArgs
from .sensor import Sensor


class A342ReaderArgs(ReaderArgs):
    """
    A342 の Reader 情報を保持したオブジェクト

    Object holding the Reader information for A342
    """

    def format_packet(self, index: int, packet: list[int]) -> MeasureData:
        """
        1レコード分のデータを受け取って計測データに変換するメソッド

        Method to convert one record of data into measurement data

        パケット構成: 13byte
        Packet structure: 13 bytes
        - 0: 0x80
        - 1-2: TEMP2
        - 3-5: XVELC
        - 6-8: YVELC
        - 9-11: ZVELC
        - 12: 0x0D

        パケット構成: 19byte
        Packet structure: 19 bytes
        - 0: 0x80
        - 1-2: FLAG
        - 3-4: TEMP1
        - 5-7: XVELC
        - 8-10: YVELC
        - 11-13: ZVELC
        - 14-15: COUNT
        - 16-17: CHECKSUM
        - 18: 0x0D

        Args:
            index (int): 何番目のデータか
                    Which data index
            packet (list[int]): 変換対象のデータ配列
                    Data array to be converted

        Returns:
            MeasureData: 変換後のデータオブジェクト
                    Converted data object
        """
        # 通信速度に応じて処理が異なる
        # Processing varies depending on the communication speed
        if self.baud == Constants.BAUD_RATE_460800:
            # 460kbps通信時、FLAG は TEMP2 の下位バイトの上位6bit
            # At 460kbps, FLAG is the upper 6 bits of the lower byte of TEMP2
            flag = packet[2] & 0b11111100
            temperature = to_int8(packet[1]) * -0.9707008 + 34.987
            x = to_dec24(packet[3], packet[4], packet[5])
            y = to_dec24(packet[6], packet[7], packet[8])
            z = to_dec24(packet[9], packet[10], packet[11])
            # 460kbps通信時、COUNT は TEMP2 の下位バイトの下位2bit
            # At 460kbps, COUNT is the lower 2 bits of the lower byte of TEMP2
            count = packet[2] & 0b11
            return MeasureData(
                index=index,
                count=count,
                temperature=temperature,
                x=x,
                y=y,
                z=z,
                flag=flag,
            )
        else:
            flag = packet[2]
            temperature = to_int16(packet[3], packet[4]) * -0.0037918 + 34.987
            x = to_dec24(packet[5], packet[6], packet[7])
            y = to_dec24(packet[8], packet[9], packet[10])
            z = to_dec24(packet[11], packet[12], packet[13])
            count = to_uint16(packet[14], packet[15])
            return MeasureData(
                index=index,
                count=count,
                temperature=temperature,
                x=x,
                y=y,
                z=z,
                flag=flag,
            )

    def _gen_invalid_data(self, index: int, count: int) -> MeasureData:
        """
        データ抜けがあった際に不正な値であることを示すデータオブジェクトを生成するメソッド

        Method to generate a data object indicating invalid values when data is missing

        Args:
            index (int): 不正な計測データのindex
                    Index of the invalid measurement data
            count (int): 不正な計測データのカウント値、complement_dataの中で前のデータから算出する
                    Count value of the invalid measurement data,
                    calculated from the previous data in complement_data

        Returns:
            MeasureData: 補完したindexとcountを持つ不正なことを表すデータオブジェクト
                    Data object indicating invalidity with the complemented index and count
        """
        invalid_value = to_dec24(0x7F, 0xFF, 0xFF)
        if self.baud == Constants.BAUD_RATE_460800:
            temperature = to_int8(0x7F) * -0.9707008 + 34.987
        else:
            temperature = to_int16(0x7F, 0xFF) * -0.0037918 + 34.987

        return MeasureData(
            index=index,
            count=count,
            temperature=temperature,
            x=invalid_value,
            y=invalid_value,
            z=invalid_value,
            flag=0,
        )


class A342(Sensor):
    """
    A342センサークラス

    A342 sensor class
    """

    class _BaudRateConfig(TypedDict):
        record_size: int
        tmp_command: int
        burst_ctl_command: tuple[int, int]  # 上位、下位 / upper, lower
        count_max: int
        count_diff: int

    class _PhysicalConfig(TypedDict):
        sps: int
        command: int
        wait_sec: float

    class ModeConfig(TypedDict):
        command: int

    BAUD_RATE_CONFIGS: dict[int, _BaudRateConfig] = {
        Constants.BAUD_RATE_460800: {
            "record_size": 13,
            "tmp_command": 0b00,  # TEMP2
            "burst_ctl_command": (
                0x47,
                0x00,
            ),  # FLAG:off, TEMP:on(2), XYZ:on, COUNT:off, CHKSUM:off
            "count_max": 4,
            "count_diff": 1,
        },
        Constants.BAUD_RATE_921600: {
            "record_size": 19,
            "tmp_command": 0b10,  # TEMP1
            "burst_ctl_command": (
                0xC7,
                0x03,
            ),  # FLAG:on, TEMP:on(2), XYZ:on, COUNT:on, CHKSUM:on
            "count_max": 2**16,
            "count_diff": 2,
        },
    }
    PHYSICAL_CONFIGS: dict[str, _PhysicalConfig] = {
        Constants.PHYSICAL_VELOCITY: {
            "sps": 3000,
            "command": 0b0000000,
            "wait_sec": 0.18,
        },
        Constants.PHYSICAL_DISPLACEMENT: {
            "sps": 300,
            "command": 0b1000000,
            "wait_sec": 1.74,
        },
    }
    MODE_CONFIGS: dict[str, ModeConfig] = {
        Constants.MODE_RAW: {"command": 0b000000},
        Constants.MODE_RMS: {"command": 0b010000},
        Constants.MODE_PP: {
            "command": 0b100000,
        },
    }

    def __init__(
        self,
        port: str,
        baud: int,
        product_id: str,
        logger_id: str,
        file_rotate_min: int,
        mode: str,
        physical: str,
        rms_pp_interval: int,
        sensor_data_diag: bool,
        sensor_data_diag_sec: float,
    ) -> None:
        super().__init__(
            model="A342",
            port=port,
            baud=baud,
            product_id=product_id,
            sps=A342.PHYSICAL_CONFIGS[physical]["sps"],
            logger_id=logger_id,
            file_rotate_min=file_rotate_min,
            physical=physical,
            mode=mode,
            sensor_data_diag=sensor_data_diag,
        )
        self.rms_pp_interval = rms_pp_interval
        self.record_size = A342.BAUD_RATE_CONFIGS[baud]["record_size"]
        self.count_diff = A342.BAUD_RATE_CONFIGS[baud]["count_diff"]
        self.count_max = A342.BAUD_RATE_CONFIGS[baud]["count_max"]
        self.wait_command = WaitCommand(A342.PHYSICAL_CONFIGS[physical]["wait_sec"])

        # 生データの時はSPSをそのまま使って故障判定の間隔を取得する
        # For raw data, use SPS directly to get the failure detection interval
        if self.mode == Constants.MODE_RAW:
            self.diag_broken_count = math.ceil(sensor_data_diag_sec * self.sps)

        # RMSとP-Pでは出力間隔からSPSを算出する
        # For RMS and P-P, calculate SPS from the output interval
        else:
            if self.physical == Constants.PHYSICAL_VELOCITY:
                sps = 10 / self.rms_pp_interval
            else:
                sps = 1 / self.rms_pp_interval
            self.diag_broken_count = math.ceil(sensor_data_diag_sec * sps)

        # SIG_CONTROLのコマンドを合成する
        # Combine the SIG_CONTROL command
        self.sig_ctl_command = (
            A342.PHYSICAL_CONFIGS[physical]["command"]
            | A342.MODE_CONFIGS[mode]["command"]
            | A342.BAUD_RATE_CONFIGS[baud]["tmp_command"]
        )
        self.burst_ctl_command = A342.BAUD_RATE_CONFIGS[baud]["burst_ctl_command"]
        self.smpl_ctl_command = self._get_smpl_ctl_command()

    def init(self) -> None:
        """
        与えられた計測設定に基づきセンサーの計測情報を設定するメソッド

        Method to set the sensor measurement information based on the given measurement settings
        """
        comm = Comm()
        comm.open(self.port, self.baud)
        comm.send(
            [
                [0, 0xFF, 0xFF, 0x0D],  # おまじない 3 回 （センサー復帰） / Magic 3
                [0, 0xFF, 0xFF, 0x0D],
                [0, 0xFF, 0xFF, 0x0D],
                [0, 0xFE, 0x00, 0x0D],  # Window 0
                [0, 0x83, 0x02, 0x0D],  # MODE_CTRL: 1-0:10=ToConfig
                # W: 出力タイプ / Output type
                [0, 0xFE, 0x01, 0x0D],  # Window 1
                [0, 0x80, self.sig_ctl_command, 0x0D],  # SIG_CTRL:
                [0, 0x81, 0x8E, 0x0D],  # SIG_CTRL: 15:TEMP 11-9:VELC
                WaitCommand(0.3),  # Wait > 118ms
                [0, 0x84, self.smpl_ctl_command[1], 0x0D],  # SMPL_CTRL: UPDT_RATE_RMSPP
                [0, 0x85, self.smpl_ctl_command[0], 0x0D],  # SMPL_CTRL: DOUT_RATE_RMSPP
                # W: UART auto sampling
                [0, 0xFE, 0x01, 0x0D],  # Window 1
                [0, 0x88, 0x01, 0x0D],  # UART_CTRL: 1:AUTO_START=0 0:UART_AUTO=1
                [0, 0x8C, self.burst_ctl_command[1], 0x0D],  # BURST_CTRL: lower
                [0, 0x8D, self.burst_ctl_command[0], 0x0D],  # BURST_CTRL: upper
                # W: XYZ_ALARM -- 用途に応じ調整必要なため設定しない
                # W: XYZ_ALARM -- Not set as adjustments are needed depending on the application
                WaitCommand(0.1),  # Wait > 5ms
            ]
        )

    def to_reader_args(self) -> ReaderArgs:
        """
        Reader プロセスに渡す設定情報に変換するメソッド

        Method to convert to configuration information to pass to the Reader process

        Returns:
            ReaderArgs: Reader プロセスに渡す設定情報
                    Configuration information to pass to the Reader process
        """
        assert self.rms_pp_interval

        # RMSとP-Pでは、デフォルトタイムアウトより長い期間readが帰らないため、計算する
        # For RMS and P-P, read does not return for a longer period than the default timeout, so calculate it
        timeout: float = Comm.DEFAULT_TIMEOUT
        if self.mode == Constants.MODE_RMS or self.mode == Constants.MODE_PP:
            if self.physical == Constants.PHYSICAL_VELOCITY:
                timeout = self.rms_pp_interval / 10
            else:
                timeout = self.rms_pp_interval
        timeout = max(Comm.DEFAULT_TIMEOUT, timeout)

        count_start, count_diff = self._get_complement_data_params()
        record_per_sec = self._get_record_per_sec()

        # 一回の取得データ数が小さいとデータとびが発生するため、最大4096byteとして1秒間分のデータをまとめて読み込む
        # To prevent data loss due to small data retrieval, read up to 4096 bytes of data for one second
        read_length = min(record_per_sec * self.record_size, Comm.DEFAULT_READ_BYTES)

        return A342ReaderArgs(
            model=self.model,
            serial=self.serial,
            port=self.port,
            baud=self.baud,
            record_size=self.record_size,
            record_per_sec=record_per_sec,
            # 過渡応答期間があり、RMSまたはP-Pがタイムアウトする恐れがあるため、スタート後にwaitを入れる
            # Include wait after start due to transient response period and potential RMS or P-P timeout
            start_command=[*Sensor.DEFAULT_START_COMMAND, self.wait_command],
            end_command=Sensor.DEFAULT_END_COMMAND,
            record_begin=Sensor.DEFAULT_RECORD_BEGIN,
            record_end=Sensor.DEFAULT_RECORD_END,
            read_length=read_length,
            timeout=timeout,
            count_diff=count_diff,
            count_max=self.count_max,
            count_start=count_start,
            sensor_data_diag=self.sensor_data_diag,
            diag_broken_count=self.diag_broken_count,
        )

    def to_writer_args(self) -> WriterArgs:
        """
        Writer プロセスに渡す設定情報に変換するメソッド

        Method to convert to configuration information to pass to the Writer process

        Returns:
            WriterArgs: Writer プロセスに渡す設定情報
                    Configuration information to pass to the Writer process
        """
        return WriterArgs(
            model=self.model,
            serial=self.serial,
            logger_id=self.logger_id,
            port=self.port,
            record_per_file=self._get_record_per_file(),
        )

    def add_info(self, info_d: dict[str, Any]) -> None:
        """
        計測情報ファイルへの出力用に設定項目を登録する

        このクラスでは以下の情報項目を登録する：
        - RMS_PP_INTERVAL

        Register configuration items for output to the measurement information file

        This class registers the following information item:
        - RMS_PP_INTERVAL

        Args:
            info_d (dict) 情報登録用辞書
                Dictionary for registering information
                - 登録する値は最終的に文字列化されるため、数値型のまま登録も可能
                    The values to be registered will eventually be converted to strings,
                    so it is possible to register them as numeric types
        """
        # 上位実装を実行
        # Execute the superclass implementation
        super().add_info(info_d)

        # 項目を追加
        # Add item
        info_d[InfoKeys.RMS_PP_INTERVAL] = self.rms_pp_interval

    def self_test(self) -> SelfTestResult:
        """
        センサーのセルフテストを実行する

        Execute self-test of the sensor

        Returns:
            SelfTestResult: セルフテスト結果
                    Result of the self-test
        """
        self._logger.info(f"Start self test for {self}")
        comm = Comm()
        comm.open(self.port, self.baud)

        # セルフテスト実行順序
        # - 加速度、温度、電源電圧
        # - 不揮発性メモリ
        # - 構造共振
        # Self-test execution order
        # - Acceleration, temperature, power voltage
        # - Non-volatile memory
        # - Structural resonance
        result: list[list[int]] = []
        for command in [0b00000111, 0b00001000, 0b10000000]:
            comm.send(
                [
                    [0, 0xFE, 0x01, 0x0D],  # Window 1
                    [0, 0x83, command, 0x0D],  # Execute self-test
                ]
            )
            while True:
                packet = comm.send([[4, 0x02, 0x00, 0x0D]])
                # セルフテスト完了判定
                # Self-test completion check
                if packet[1] & command == 0:
                    break
                else:
                    time.sleep(0.1)

            result.append(
                comm.send(
                    [
                        [0, 0xFE, 0x00, 0x0D],  # Window 0
                        [4, 0x04, 0x00, 0x0D],  # R: DIAG_STAT
                    ]
                )
            )

        # 加速度: bit1
        # 温度: bit9
        # 電源電圧: bit8
        # Acceleration: bit1
        # Temperature: bit9
        # Power voltage: bit8
        acc_result = self._determine_test_bit(result[0][2], 0b00000010)
        temp_result = self._determine_test_bit(result[0][1], 0b00000010)
        vdd_result = self._determine_test_bit(result[0][1], 0b00000001)

        # 不揮発性メモリ: bit2
        # Non-volatile memory: bit2
        mem_result = self._determine_test_bit(result[1][2], 0b00000100)

        # 構造共振のみDIAG_STAT1ではないため、ループの結果を無視して読み込む
        # Structural resonance is not in DIAG_STAT1, so read it separately
        diag_stat2 = comm.send(
            [
                [0, 0xFE, 0x00, 0x0D],  # Window 0
                [4, 0x0C, 0x00, 0x0D],  # R: DIAG_STAT2
            ]
        )

        x_exi_result = self._determine_test_bit(diag_stat2[1], 0b00000011)
        y_exi_result = self._determine_test_bit(diag_stat2[1], 0b00001100)
        z_exi_result = self._determine_test_bit(diag_stat2[1], 0b00110000)

        test_result = SelfTestResult(
            model=self.model,
            serial=self.serial,
            acceleration=acc_result,
            sensitivity=Constants.TEST_NOT_AVAILABLE,
            temperature=temp_result,
            power_voltage=vdd_result,
            flash_memory=mem_result,
            structural_resonance=f"X:{x_exi_result},Y:{y_exi_result},Z:{z_exi_result}",
        )
        test_result.output()

        return test_result

    def _get_smpl_ctl_command(self) -> tuple[int, int]:  # noqa: C901
        """
        RMSとP-Pの計算、出力間隔を取得するメソッド

        Method to obtain the calculation and output interval for RMS and P-P

        Returns:
            Tuple[int, int]: (出力レートのコマンド, 更新レートのコマンド)
                    (Output rate command, update rate command)
        """
        # A342では設定済み
        # Already set in A342
        assert self.rms_pp_interval

        # config取得時に検証済みだが一応チェック
        # Verified during config retrieval, but check again
        if self.rms_pp_interval < 1 or self.rms_pp_interval > 255:
            raise ValueError

        # 生データ出力時は使用しないためデフォルト値を返す
        # Return default values as they are not used for raw data output
        if self.mode == Constants.MODE_RAW:
            return (0x0A, 0x07)

        if self.rms_pp_interval <= 1:
            update_rate = 4
        elif self.rms_pp_interval <= 3:
            update_rate = 5
        elif self.rms_pp_interval <= 6:
            update_rate = 6
        elif self.rms_pp_interval <= 13:
            update_rate = 7
        elif self.rms_pp_interval <= 27:
            update_rate = 8
        elif self.rms_pp_interval <= 54:
            update_rate = 9
        elif self.rms_pp_interval <= 109:
            update_rate = 10
        elif self.rms_pp_interval <= 218:
            update_rate = 11
        else:
            update_rate = 12

        return (self.rms_pp_interval, update_rate)

    def _get_sps(self) -> float:
        """
        SPS を返すメソッド

        RMS や P-P では設定により変わるため計算して返す

        Method to return SPS

        For RMS and P-P, calculate and return based on the settings

        Returns:
            float: 現在の設定における SPS 値
                    Current SPS value based on the settings
        """
        if self.mode == Constants.MODE_RAW:
            return self.sps

        assert self.rms_pp_interval
        # 出力間隔(秒)
        # Output interval (seconds)
        interval = (
            self.rms_pp_interval
            if self.physical == Constants.PHYSICAL_DISPLACEMENT
            else self.rms_pp_interval / 10
        )
        # サンプリングレートは出力間隔の逆数
        # Sampling rate is the reciprocal of the output interval
        sps = 1 / interval

        return sps

    def _get_record_per_sec(self) -> int:
        """
        一秒間のレコード数を返す

        Returns the number of records per second

        Returns:
            int: 一秒間のレコード数
                    Number of records per second
        """
        return math.ceil(self._get_sps())

    def _get_record_per_file(self) -> int:
        """
        ファイルごとのレコード数を返す

        Returns the number of records per file

        Returns:
            int: ファイルごとのレコード数
                    Number of records per file
        """
        return math.ceil(self._get_sps() * self.file_rotate_min * Constants.SEC_IN_MIN)

    def _get_complement_data_params(self) -> tuple[int, int]:
        """
        A342ReaderArgsでのデータ欠損補完に用いるパラメータを取得するメソッド

        Method to obtain parameters used for data loss compensation in A342ReaderArgs

        Returns:
            Tuple[int, int]: 1番目: index 0の時のcount値、2番目: カウントの増分
                    1st: count value when index is 0, 2nd: count increment
        """
        # 生データの時とRMS, P-Pで2bitカウンタの時にはコンストラクタで取得した値をそのまま使える
        # Use the values obtained in the constructor for raw data and 2-bit counter in RMS, P-P
        if self.mode == Constants.MODE_RAW or self.baud == Constants.BAUD_RATE_460800:
            return (self.count_diff, self.count_diff)

        # RMS と P-P で921.6kbpsの時は、カウントの増分がカウントの最大値を超えうるため、仮想的な最大値を計算しておく
        # For RMS and P-P at 921.6kbps, calculate a virtual maximum value
        # as the count increment can exceed the maximum count
        else:
            assert self.rms_pp_interval
            count_diff = self.rms_pp_interval * 600
            count_start = (count_diff + 2) % self.count_max
            return (count_start, count_diff)
