# Copyright 2024, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

import datetime
import logging
import math
import multiprocessing
import os
import platform
import threading
import time

import serial.tools.list_ports

from logger import Constants, InfoKeys, LoggerFactory
from logger.core import Config, LoggerError
from logger.utils.format import app_format
from logger.utils.timezone import get_time_zone

from .job import JobError
from .sensor import Sensor, SensorDeviceError


class Controller:
    """
    センサーによる計測をコントロールするクラス

    以下の処理を行う：
    - USB 接続されたセンサーを検索し、初期化する
    - タイミングに従い、センサーの計測を開始する
    - タイマーや外部イベントに従い、センサーの計測を停止する
    - センサーのセルフテストを実行する

    Class to control measurements by sensors

    Performs the following operations:
    - Searches for and initializes USB-connected sensors
    - Starts sensor measurements according to timing
    - Stops sensor measurements according to timers or external events
    - Executes self-tests of the sensors
    """

    def __init__(self, config: Config, start_type: str, factory: LoggerFactory) -> None:
        """
        Controller のコンストラクタ

        Constructor for the Controller

        Args:
            config (Config): 設定情報を保持した Config オブジェクト
                    Config object holding configuration information
            start_type (str): main 開始時に指定された起動タイプ
                    Startup type specified at the start of main
            factory (LoggerFactory): Logger の主要な要素を生成するファクトリ
                    Factory to generate major elements of Logger
        """
        self.config = config
        self._factory: LoggerFactory = factory
        self._factory.set_config(config)
        self._start_type = start_type
        self._sensors: list[Sensor] = []
        self._stop_timer: threading.Timer | None = None
        self._measuring: bool = False
        self._error_queue: multiprocessing.Queue | None = None
        self._start_time: datetime.datetime | None = None
        self._end_time: datetime.datetime | None = None
        self._output_dir: str | None = None
        self._logger = logging.getLogger(__name__)

    def init(self) -> None:
        """
        計測の初期化を行う

        Initialize the measurement
        """
        self._logger.info("Start initialization")

        # センサー検索： USBデバイスをサーチしてセンサを探す
        # Search for sensors: Scan USB devices to find sensors
        for port in Controller._list_comports():
            self._logger.debug(f"COM port found: {port}")

            # - センサーとの接続確立試行
            # - Attempt to establish connection with the sensor
            sensor = self._init_sensor(port)
            if sensor:
                self._sensors.append(sensor)

        # - センサーが見つからなければ終了
        # - If no sensors are found, terminate
        if len(self._sensors) == 0:
            self._logger.error("No sensor found")
            raise LoggerError("No Ready Sensor found")

        # 接続できたセンサーに初期化を指示する
        # Instruct the connected sensors to initialize
        for sensor in self._sensors:
            sensor.init()

    def start(self) -> None:  # noqa: C901
        """
        センサーによる計測を開始する

        Start measurement by sensors

        Raises:
            LoggerError: すべてのセンサーが計測に失敗した場合
                    If all sensors fail to measure
        """
        # 初回待機： 自動計測時
        # Initial wait: for auto measurements
        if self._start_type == Constants.START_TYPE_AUTO:
            self._logger.info(f"Initial wait for {self.config.initial_wait} sec")
            time.sleep(self.config.initial_wait)

        # 計測開始
        # - 計測開始日時
        # Start measurement
        # - Measurement start time
        self._start_time = datetime.datetime.now()
        self._output_dir = None

        # - 出力先フォルダ
        # - Output folder
        out_dir: str = str(self.get_output_dir())
        os.makedirs(out_dir, exist_ok=False)

        # - 計測時間設定: 計測時間が設定されている場合
        # - Set measurement time: If measurement time is set
        if self.config.measure_time_sec > 0:
            # タイマー設定： タイマー発火時は stop() が呼ばれる
            # Set timer: stop() is called when the timer fires
            self._stop_timer = threading.Timer(self.config.measure_time_sec, self.stop)
            self._stop_timer.start()

        # - 内部計測フラグ ON
        # - Internal measurement flag ON
        self._measuring = True

        # - 計測エラー通知用キュー
        # - Queue for measurement error notifications
        self._error_queue = multiprocessing.Queue()

        # - センサーに計測開始指示
        # - Instruct sensors to start measurement
        self._logger.info("Start measurement")
        failed = set()
        for sensor in self._sensors:
            try:
                sensor.start(
                    error_queue=self._error_queue,
                    output_dir=out_dir,
                    factory=self._factory,
                )
            except Exception:
                self._logger.exception(f"Sensor: {sensor} failed to start measurement")
                failed.add(f"{sensor.model}_{sensor.serial}")

        # - すべて計測開始に失敗: start() 処理をエラーで終了
        # - If all sensors fail to start measurement: Terminate start() process with error
        if len(failed) == len(self._sensors):
            self._logger.error("All sensors failed to start measurement")
            raise LoggerError("All sensors failed to start measurement")

        # 計測情報ファイル 開始時出力: 計測終了時間を除き出力、終了時に上書き
        # Output measurement information file at start: Output excluding end time, overwrite at end
        self._output_info()

        # 計測監視ループ: センサーサブプロセスからのエラーを定期的にチェック
        # Measurement monitoring loop: Periodically check for errors from sensor subprocesses
        while self._measuring:
            time.sleep(1)

            # エラーキューに登録あり
            # If there are entries in the error queue
            if self._error_queue and not self._error_queue.empty():
                error: JobError = self._error_queue.get()
                for sensor in self._sensors:
                    # エラー発生源センサー特定
                    # Identify the sensor that caused the error
                    if sensor.model == error.model and sensor.serial == error.serial:
                        failed.add(f"{error.model}_{error.serial}")
                        self._logger.error(
                            f"Sensor: {sensor} got error during measurement"
                        )
                        # センサー停止試行
                        # Attempt to stop the sensor
                        try:
                            sensor.stop()
                        except Exception:
                            pass
                        break

                # 全てのセンサーの計測が失敗したらエラーで終了
                # If all sensors fail to measure, terminate with error
                if len(failed) == len(self._sensors):
                    self._logger.error("All sensors failed to measure")
                    raise LoggerError("All sensors failed to measure")

        # センサー計測終了待ち： タイマーが設定されている場合、サブプロセスの終了を待つ
        # Wait for sensor measurement to end: If a timer is set, wait for subprocesses to end
        if self._stop_timer:
            self._stop_timer.join()

    def stop(self, terminate: bool = False) -> None:
        """
        センサーによる計測を終了する

        このメソッドは、計測時間タイマーの発火か、main の終了シーケンスから呼ばれる

        Stop measurement by sensors

        This method is called either by the measurement time timer firing or from the main termination sequence

        Args:
            terminate (bool, optional): SIGTERM などの中断により呼ばれたか
                    Whether it was called due to an interruption like SIGTERM
                - Default: False
        """
        # 計測中判定： main の終了シーケンスでは未計測時に呼ばれる場合がある
        # Measurement in progress check: May be called during the main termination sequence when not measuring
        if not self._measuring:
            self._logger.debug("stop method called, but not measure")
            return

        try:
            # 終了処理
            # - 内部計測フラグ OFF: start() での計測監視ループを停止する
            # Termination process
            # - Internal measurement flag OFF: Stops the measurement monitoring loop in start()
            self._measuring = False

            # - 中断による終了時: 計測時間タイマーを停止する
            # - On termination due to interruption: Stop the measurement time timer
            if terminate and self._stop_timer:
                self._stop_timer.cancel()
                self._stop_timer = None

            # - センサー停止指示
            # - Instruct sensors to stop
            for sensor in self._sensors:
                try:
                    sensor.stop()
                except Exception:
                    self._logger.exception(
                        f"Sensor: {sensor} failed to stop measurement"
                    )

            # - 終了日時記録
            # - Record end time
            self._end_time = datetime.datetime.now()

            # - 終了ログ
            # - End log
            if terminate:
                self._logger.info("Terminate measurement")
            else:
                self._logger.info("Finish measurement")

            # - 計測情報ファイル 終了時出力（上書き）
            # - Output measurement information file at end (overwrite)
            self._output_info()

            # - エラーキューのクローズ
            # - Close the error queue
            if self._error_queue:
                self._error_queue.close()
                self._error_queue = None

        except Exception:
            # main の終了シーケンスで呼ばれるため、Error はログ出力して無視する
            # Called during the main termination sequence, so log and ignore errors
            self._logger.exception("An error occurred during stop measurement")

    def get_output_dir(self) -> str | None:
        """
        計測データの出力先フォルダを返す

        - 計測が開始されている場合、開始年月日を基準にパスを返す
        - システム時刻が不正な場合は開始年月日のフォルダが存在する可能性があるため、追い番を追加する

        Returns the output folder for measurement data

        - If measurement has started, returns a path based on the start date and time
        - If the system time is incorrect, a folder with the same start date might already exist,
          so a sequential number is appended to avoid conflicts

        Returns:
            str | None : 計測が開始されている場合、開始年月日基準のパスを返す
                    Returns a path based on the start date if measurement has started
                - 計測が開始されていない場合
                    If measurement has not started: None
        """
        # 出力先フォルダが決定済み
        # Output folder is already determined
        if self._output_dir:
            return self._output_dir

        # 出力先フォルダが未決定
        # Output folder is not yet determined
        if self._start_time:
            # - 既存のフォルダを回避する
            # - Avoid using an existing folder
            dir_ptn = f"{self.config.output_path}/{self._start_time:%Y%m%d_%H%M%S}"
            dir_cnt = 0
            out_dir = dir_ptn
            while os.path.exists(out_dir):
                dir_cnt += 1
                out_dir = f"{dir_ptn}_{dir_cnt}"
            if dir_cnt:
                self._logger.warning(
                    f"Output folder {dir_ptn} already exists, so append a sequential number"
                )
            self._output_dir = out_dir
            return self._output_dir
        else:
            return None

    @classmethod
    def _list_comports(cls) -> list[str]:
        """
        通信可能なポートの一覧を取得するメソッド

        Method to get a list of available communication ports

        Returns:
            list[str]: 通信できるポートの一覧
                    List of available communication ports
        """
        result = []
        for com in serial.tools.list_ports.comports():
            if com.device.startswith("/dev/ttyAMA"):
                # ラズパイの GPIO ポートのため使用しない
                # Do not use Raspberry Pi GPIO ports
                continue
            result.append(com.device)
        return sorted(result)

    def _init_sensor(self, port: str) -> Sensor | None:
        """
        センサーの初期化をするメソッド

        Method to initialize the sensor

        Args:
            port (str): センサーと通信を試みるポート
                    Port to attempt communication with the sensor

        Raises:
            SensorDeviceError: センサーが通信可能な状態でない場合
                    If the sensor is not in a communicable state

        Returns:
            Sensor | None : 初期化に成功したセンサー
                    The sensor that was successfully initialized
        """
        # センサーとの初期通信
        # Initial communication with the sensor
        try:
            sensor = self._create_sensor(port)
        except ConnectionRefusedError:
            self._logger.error(f"Failed to connect: {port}")
            return None

        # センサーの状態チェック
        # Check the status of the sensor
        try:
            # 状態チェック
            # Status check
            if not sensor.ready():
                raise SensorDeviceError()

            # シリアル読み込み
            # Load serial
            sensor.load_serial()

            # ファームウェアバージョン読み込み
            # Load firmware version
            sensor.load_firm_ver()

        except Exception:
            self._logger.exception(f"Failed to initialize: {port}")
            return None

        return sensor

    def _create_sensor(self, port: str) -> Sensor:
        """
        具象クラスのセンサーを生成するメソッド

        Method to create a concrete class sensor

        Args:
            port (str): センサーとの通信を試みるポート
                    Port to attempt communication with the sensor

        Raises:
            ValueError: 不正なセンサーのproduct_idが取得された場合
                    If an invalid sensor product_id is obtained

        Returns:
            Sensor: 作成されたセンサー(具象クラス)
                    The created sensor (concrete class)
        """
        # センサークラスに PRODUCT_ID の取得を指示
        # Instruct the sensor class to obtain the PRODUCT_ID
        product_id = Sensor.load_product_id(port, self.config.baud_rate)
        self._logger.info(
            f"Found product: {product_id} port={port} baud={self.config.baud_rate}"
        )

        # A352 生成
        # Create A352
        if product_id.startswith("A352"):
            return self._factory.create_A352(port, product_id)

        # A342 生成
        # Create A342
        if product_id.startswith("A342"):
            return self._factory.create_A342(port, product_id)

        # A370 生成
        # Create A370
        if product_id.startswith("A370"):
            return self._factory.create_A370(port, product_id)

        else:
            raise ValueError(f"Unknown PRODUCT_ID: {product_id}")

    def _output_info(self) -> None:
        """
        計測情報ファイルを出力する関数

        Function to output the measurement information file

        Raises:
            AssertionError: 計測開始前に呼ばれた場合（保存先ディレクトリが存在しない）
                    If called before measurement starts (output directory does not exist)
        """
        # 出力先フォルダ
        # Output folder
        output_dir = self.get_output_dir()
        assert output_dir

        # 出力項目補正
        # - 初回待機秒数
        # Adjust output items
        # - Initial wait seconds
        init_wait: int = self.config.initial_wait
        if self._start_type == Constants.START_TYPE_MANUAL:
            init_wait = 0

        # - 計測時間
        # - Measurement time
        meas_time: int | None = None
        if self._start_time and self._end_time:
            meas_time = math.floor((self._end_time - self._start_time).total_seconds())

        # Controller レベル出力項目
        # - 環境情報
        # Controller level output items
        # - Environment information
        info_com: dict = {
            InfoKeys.OS: platform.platform(),
            InfoKeys.LOGGER_VERSION: Constants.LOGGER_VERSION,
            InfoKeys.LOGGER_ID: self.config.logger_id,
        }

        # - 計測情報
        # - Measurement information
        info_com.update(
            {
                InfoKeys.DATE: app_format(self._start_time, "%Y/%m/%d %H:%M:%S"),
                InfoKeys.TIME_ZONE: get_time_zone(),
                InfoKeys.START_TYPE: self._start_type,
                InfoKeys.MEASURE_TIME_SEC: app_format(meas_time),
                InfoKeys.INITIAL_WAIT: init_wait,
            }
        )

        # Config レベル出力項目
        # Config level output items
        self.config.add_info(info_com)

        # センサーごとに出力
        # Output for each sensor
        for sensor in self._sensors:
            info_d: dict = info_com.copy()

            # Sensor レベル出力項目
            # Sensor level output items
            sensor.add_info(info_d)

            # 計測ディレクトリのinfo.csvに書き込み
            # - ファイルシステムに異常がある場合例外が発生するが、ファイル出力が失敗するため最初のセンサーで例外にする
            # Write to info.csv in the measurement directory
            # - If there is an abnormality in the file system, an exception will occur,
            #   but the file output will fail, so an exception will be made for the first sensor
            with open(
                f"{output_dir}/{sensor.model}_{sensor.serial}_info.csv",
                mode="w",
                encoding="UTF-8",
                newline="\r\n",
            ) as f:
                for key, value in info_d.items():
                    f.write(f"{key},{value}\n")

    def self_test(self) -> None:
        """
        センサーのセルフテストを実行する

        Execute self-test of the sensors
        """

        self._logger.info("Start connection")

        # USBデバイスサーチ
        # Search for USB devices
        for port in Controller._list_comports():
            self._logger.debug(f"COM port found: {port}")

            # センサー初期化
            # Initialize sensor
            sensor = self._create_sensor(port)
            sensor.load_serial()
            self._sensors.append(sensor)

        if len(self._sensors) == 0:
            self._logger.error("No sensor found")
            raise LoggerError("No Ready Sensor found")

        # セルフテスト実行
        # Execute self-test
        for sensor in self._sensors:
            sensor.self_test()
