# Copyright 2024, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

import logging
import signal
import sys
import time
from dataclasses import asdict
from traceback import format_exc

import schedule

from logger import ExitCode
from logger.core import (
    MessageService,
    ServiceFrame,
    StdoutLoggingService,
    TerminateException,
    raise_terminate_exception,
)
from logger.core.factory import LoggerFactoryImpl
from logger.tool.hwmonitor.manager import MonitorManager

from .topic import HWTopic


def main():
    """
    メイン関数 for hwmonitor

    Main function for hwmonitor
    """

    # プロセス初期化
    # - シグナルハンドラー登録 : SIGTERM, SIGINT 発生時は例外処理にて対処する
    # Process initialization
    # - Register signal handlers: Handle exceptions for SIGTERM, SIGINT
    signal.signal(signal.SIGTERM, raise_terminate_exception)
    signal.signal(signal.SIGINT, raise_terminate_exception)

    # - ロギング初期化
    # - Initialize logging
    svc_frame = ServiceFrame.get_instance()
    svc_frame.add_service(StdoutLoggingService())
    svc_frame.start()

    logger = logging.getLogger(__name__)

    # 監視開始
    # Start monitoring
    logger.info("Start monitor")
    try:
        # 設定ファイルの読み込み
        # Load configuration file
        factory = LoggerFactoryImpl()
        config = factory.create_configurator().get_config()

        if config is None:
            logger.error("Config file has wrong value, cannot start hwmonitor")
            return ExitCode.EX_NG

        # メッセージサービス登録
        # Register message service
        svc_frame.add_service(MessageService(config, root_elem="hwmonitor"))

        # - START メッセージ送信
        # - Send START message
        MessageService.send(HWTopic.hwmonitor(), "START")

        # 監視初期化
        # Initialize monitoring
        manager = MonitorManager(output_path=config.output_path)
        manager.init()

        # - 監視処理関数
        # - Monitoring process function
        def job() -> None:
            result = manager.monitor()
            logger.info(result)
            manager.output(result)

            # メッセージ送信
            # Send message
            resd = asdict(result)
            resd.pop("time")  # 不要な要素を削除 / Remove unnecessary element
            MessageService.send(HWTopic.hwmonitor_data(), resd)

        # - １分間隔で処理を実行
        # - Execute process at 1-minute intervals
        schedule.every(1).minutes.do(job)

        # 監視実行
        # Execute monitoring
        while True:
            schedule.run_pending()
            time.sleep(1)

    except TerminateException:
        logger.info("Finish monitor")
        return ExitCode.EX_OK

    except Exception as e:
        # ** 例外が出力されない場合があるため、手動で出力
        # ** Manually output exceptions as they may not be logged
        # logger.exception("Unknown error occurred")
        m = "Unknown error occurred"
        logger.error(f"{m}\n{format_exc()}")

        # - Error メッセージ送信
        # - Send error message
        MessageService.send(HWTopic.hwmonitor_error(), m, exc=e)

        return ExitCode.EX_NG

    finally:
        # STOP メッセージ送信
        # Send STOP message
        MessageService.send(HWTopic.hwmonitor(), "STOP")

        # ServiceFrame 終了
        # Terminate ServiceFrame
        svc_frame.close()


if __name__ == "__main__":
    code = main()
    sys.exit(code)
