# Copyright 2024, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

import datetime
import os

import psutil

from logger.tool.hwmonitor.data import MonitorResult


class MonitorManager:
    """
    ハードウェア状態モニターの管理クラス

    以下の機能を提供する：
    - 監視機能の初期化
    - 使用状況の計測
    - 計測結果のファイル出力

    Management class for hardware status monitoring

    Provides the following functions:
    - Initialization of monitoring functions
    - Measurement of usage
    - Output of measurement results to a file
    """

    CSV_HEADER = (
        "Time,CPU Temperature(deg C),CPU Usage(%),Memory Usage(%),Disk Usage(%)\n"
    )

    def __init__(self, output_path: str) -> None:
        self.output_path = output_path
        self.filename = ""

    def init(self) -> None:
        """
        ハードウェア監視の初期化を行うメソッド

        - 出力ファイルの作成
        - 監視パッケージの初期化

        Method to initialize hardware monitoring

        - Create output file
        - Initialize monitoring package
        """
        dir_name = f"{self.output_path}/hardware"
        os.makedirs(name=dir_name, exist_ok=True)
        self.filename = f"{dir_name}/{format(datetime.datetime.now(), '%Y%m%d_%H%M%S')}_hwmonitor.csv"
        with open(self.filename, mode="w", encoding="UTF-8", newline="\r\n") as f:
            f.write(MonitorManager.CSV_HEADER)

        # 初回実行時は0.0%しか表示されないのでinitのタイミングでcpu_percentを取得する
        # Since only 0.0% is displayed at the first execution, obtain cpu_percent at the timing of init
        psutil.cpu_percent(interval=1)

    def monitor(self) -> MonitorResult:
        """
        ハードウェアの使用状況を計測するメソッド

        Method to measure hardware usage

        Returns:
            MonitorResult: 計測結果オブジェクト
                    Measurement result object
        """
        time = datetime.datetime.now()
        cpu_temperature = psutil.sensors_temperatures()["cpu_thermal"][0].current  # type: ignore
        cpu_usage = psutil.cpu_percent()
        memory_usage = psutil.virtual_memory().percent
        disk_usage = psutil.disk_usage(self.output_path).percent

        return MonitorResult(
            time=time,
            cpu_temperature=cpu_temperature,
            cpu_usage=cpu_usage,
            memory_usage=memory_usage,
            disk_usage=disk_usage,
        )

    def output(self, data: MonitorResult) -> None:
        """
        CSVに監視結果を書き出すメソッド

        Method to write monitoring results to CSV

        Args:
            data (MonitorResult): 監視結果のクラスインスタンス
                    Instance of the monitoring result class
        """
        with open(self.filename, mode="a", encoding="UTF-8", newline="\r\n") as f:
            f.write(data.to_csv_line())
