# Copyright 2024, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.


def to_int8(b1: int) -> int:
    """
    1 byte -> 8bit signed int 変換

    1 byte -> 8bit signed int conversion

    Args:
        b1 (int): センサーから取得した１バイト数値
                1-byte value obtained from the sensor

    Returns:
        int: １バイト数値を符号付き8bitに変換した数値
                1-byte value converted to signed 8-bit integer
    """
    return (0x80 & b1) * -1 + (0x7F & b1)


def to_uint16(b1: int, b2: int) -> int:
    """
    2 byte -> 16bit unsigned int 変換

    2 byte -> 16bit unsigned int conversion

    Args:
        b1 (int): センサーから取得した２バイト数値の１バイト目
                First byte of the 2-byte value obtained from the sensor
        b2 (int): センサーから取得した２バイト数値の２バイト目
                Second byte of the 2-byte value obtained from the sensor

    Returns:
        int: ２バイト数値を符号なし16bitに変換した数値
                2-byte value converted to unsigned 16-bit integer
    """
    return b1 * 2**8 + b2


def to_int16(b1: int, b2: int) -> int:
    """
    2 byte -> 16bit signed int 変換

    2 byte -> 16bit signed int conversion

    Args:
        b1 (int): センサーから取得した２バイト数値の１バイト目
                First byte of the 2-byte value obtained from the sensor
        b2 (int): センサーから取得した２バイト数値の２バイト目
                Second byte of the 2-byte value obtained from the sensor

    Returns:
        int: ２バイト数値を符号付き16bitに変換した数値
                2-byte value converted to signed 16-bit integer
    """
    num = to_uint16(b1, b2)
    return (0x8000 & num) * -1 + (0x7FFF & num)


def to_uint32(ba4: list[int]) -> int:
    """
    4 byte -> 32bit unsigned int 変換

    4 byte -> 32bit unsigned int conversion

    Args:
        ba4 (list[int]): センサーから取得した４バイト数値を格納したリスト
                List containing the 4-byte value obtained from the sensor
            - ba4[0] に１バイト目、ba4[3] に４バイト目を格納する
            - Store the first byte in ba4[0] and the fourth byte in ba4[3]

    Returns:
        int: ４バイト数値を符号なし32bit変換した数値
                4-byte value converted to unsigned 32-bit integer
    """
    return ba4[0] * 2**24 + ba4[1] * 2**16 + ba4[2] * 2**8 + ba4[3]


def to_int32(ba4: list[int]) -> int:
    """
    4 byte -> 32bit signed int 変換

    4 byte -> 32bit signed int conversion

    Args:
        ba4 (list[int]): センサーから取得した４バイト数値を格納したリスト
                List containing the 4-byte value obtained from the sensor
            - ba4[0] に１バイト目、ba4[3] に４バイト目を格納する
            - Store the first byte in ba4[0] and the fourth byte in ba4[3]

    Returns:
        int: ４バイト数値を符号あり32bit変換した数値
                4-byte value converted to signed 32-bit integer
    """
    num = to_uint32(ba4)
    return (0x80000000 & num) * -1 + (0x7FFFFFFF & num)


def to_dec24(b1: int, b2: int, b3: int) -> float:
    """
    3 bytes -> 24bit signed decimal 変換

    仕様：
    - 24bit, 2の補数
        - bit23 : 符号部 -- 整数部も兼ねる
        - bit22 : 整数部
        - bit21-0 : 小数部

    3 bytes -> 24bit signed decimal conversion

    Specification:
    - 24bit, 2's complement
        - bit23 : Sign bit -- also serves as the integer part
        - bit22 : Integer part
        - bit21-0 : Fractional part

    Args:
        b1 (int): センサーから取得した３バイト小数表現の１バイト目
                First byte of the 3-byte decimal representation obtained from the sensor
        b2 (int): センサーから取得した３バイト小数表現の２バイト目
                Second byte of the 3-byte decimal representation obtained from the sensor
        b3 (int): センサーから取得した３バイト小数表現の３バイト目
                Third byte of the 3-byte decimal representation obtained from the sensor

    Returns:
        float: ３バイト小数表現を仕様に基づき変換した符号付き24bit浮動小数
                3-byte decimal representation converted to signed 24-bit floating point based on the specification
    """
    # 最上位2bit取り出し
    msb2 = (b1 & 0x00C0) >> 6
    # 小数部合成
    dec = ((b1 & 0x3F) << 16) + (b2 << 8) + b3
    # 合成
    return (msb2 & 0b10) * -1 + (msb2 & 0b01) + dec / 2**22


def to_dec32(ba4: list[int]) -> float:
    """
    4 bytes -> 32bit signed decimal 変換

    仕様：
    - 32bit, 2の補数
        - bit31 : 符号部 -- 整数部も兼ねる
        - bit30-24 : 整数部
        - bit23-0  : 小数部

    4 bytes -> 32bit signed decimal conversion

    Specification:
    - 32bit, 2's complement
        - bit31 : Sign bit -- also serves as the integer part
        - bit30-24 : Integer part
        - bit23-0  : Fractional part

    Args:
        ba4 (list[int]): センサーから取得した４バイト小数表現を格納したリスト
                List containing the 4-byte decimal representation obtained from the sensor
            - ba4[0] に１バイト目、ba4[3] に４バイト目を格納する
            - Store the first byte in ba4[0] and the fourth byte in ba4[3]

    Returns:
        float: ４バイト小数表現を仕様に基づき変換した符号付き32bit浮動小数
                4-byte decimal representation converted to signed 32-bit floating point based on the specification
    """
    return to_int32(ba4) / 2**24
