# Copyright 2024, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

import tempfile
from typing import Callable, Generator

import pytest


@pytest.fixture()
def tempdir() -> Generator:
    with tempfile.TemporaryDirectory() as dirname:
        yield dirname


@pytest.fixture()
def mock_comm(
    mocker,
) -> Callable[[list[int] | None, list[int] | None, list[list[int]] | None], None]:
    def func(
        send_return: list[int] | None = None,
        read_return: list[int] | None = None,
        send_sidefx: list[list[int]] | None = None,
    ) -> None:
        mocker.patch("logger.measure.comm.Comm.open")
        mocker.patch("logger.measure.comm.Comm.close")
        mocker.patch("logger.measure.comm.Comm.read_bytes", return_value=[])
        mocker.patch("logger.measure.comm.Comm.read", return_value=read_return)
        if send_sidefx is None:
            mocker.patch("logger.measure.comm.Comm.send", return_value=send_return)
        else:
            mocker.patch("logger.measure.comm.Comm.send", side_effect=send_sidefx)

    return func


@pytest.fixture()
def mock_controller_list_comports(mocker) -> None:
    mocker.patch(
        "logger.measure.controller.Controller._list_comports", return_value=["/dev/AAA"]
    )


@pytest.fixture()
def mock_sensor(mocker) -> Callable[[str], None]:
    def func(load_product_id_return: str) -> None:
        mocker.patch(
            "logger.measure.sensor.Sensor.load_product_id",
            return_value=load_product_id_return,
        )

    return func


@pytest.fixture()
def mock_A352(mocker) -> Callable[[bool], None]:
    def func(ready_return: bool) -> None:
        mocker.patch("logger.measure.A352.A352.init")
        mocker.patch("logger.measure.A352.A352.start")
        mocker.patch("logger.measure.A352.A352.stop")
        mocker.patch("logger.measure.A352.A352.load_serial")
        mocker.patch("logger.measure.A352.A352.load_firm_ver")
        mocker.patch("logger.measure.A352.A352.ready", return_value=ready_return)

    return func
