# Copyright 2024, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.


from logger import Constants
from logger.measure import A352, A352ReaderArgs, WriterArgs
from logger.measure.comm import Comm, WaitCommand
from logger.measure.data import SelfTestResult


class TestA352:
    class TestToReaderArgs:
        def test_to_reader_args(self):
            sensor = A352(
                port="/dev/AAA",
                baud=100,
                product_id="A352XXX",
                sps=1000,
                logger_id="hoge",
                file_rotate_min=1,
                sensor_data_diag=True,
                sensor_data_diag_sec=0.1,
            )
            sensor.serial = "test_serial"
            expect = A352ReaderArgs(
                model="A352",
                serial="test_serial",
                port="/dev/AAA",
                baud=100,
                record_size=22,
                record_per_sec=1000,
                start_command=[
                    [0, 0xFF, 0xFF, 0x0D],  # Recovery magic 3
                    [0, 0xFF, 0xFF, 0x0D],
                    [0, 0xFF, 0xFF, 0x0D],
                    [0, 0xFE, 0x00, 0x0D],  # WINDOW_ID(L) write command.(WINDOW=0)
                    [0, 0x83, 0x01, 0x0D],  # Automatic sampling mode
                ],
                end_command=[
                    [0, 0xFE, 0x00, 0x0D],  # Window 0
                    [0, 0x83, 0x02, 0x0D],  # MODE_CTRL: 1-0:10=ToConfig
                    WaitCommand(0.01),  # Wait (undefined, follow to A342))
                ],
                record_begin=0x80,
                record_end=0x0D,
                count_diff=4,
                count_max=2**16,
                count_start=4,
                sensor_data_diag=True,
                diag_broken_count=100,
                read_length=Comm.DEFAULT_READ_BYTES,
                timeout=Comm.DEFAULT_TIMEOUT,
            )
            assert sensor.to_reader_args() == expect

    class TestToWriterArgs:
        def test_to_writer_args(self):
            sensor = A352(
                port="/dev/AAA",
                baud=100,
                product_id="A352XXX",
                sps=1000,
                logger_id="hoge",
                file_rotate_min=5,
                sensor_data_diag=True,
                sensor_data_diag_sec=0.1,
            )
            sensor.serial = "test_serial"
            expect = WriterArgs(
                model="A352",
                serial="test_serial",
                port="/dev/AAA",
                logger_id="hoge",
                record_per_file=1000 * 60 * 5,
            )
            assert sensor.to_writer_args() == expect

    class TestAddInfo:
        KEYS: list[str] = [
            "PORT",
            "SENSOR",
            "PRODUCT_ID",
            "SERIAL_NO",
            "FIRM_VER",
            "PHYSICAL",
            "OUTPUT_TYPE",
            "SPS",
            "FILTER",
        ]

        def test_add_part_of_info_keys(self):
            sensor = A352(
                port="/dev/AAA",
                baud=100,
                product_id="A352XXX",
                sps=1000,
                logger_id="hoge",
                file_rotate_min=1,
                sensor_data_diag=True,
                sensor_data_diag_sec=0.1,
            )
            sensor.serial = "test_serial"

            # add_info
            info_d: dict = {}
            sensor.add_info(info_d)

            assert len(info_d) == len(self.KEYS)
            assert list(info_d.keys()) == self.KEYS
            # Value checks are performed in the Controller tests

    class TestSelfTest:
        _void = [0x00]
        _cont = [0x02, 0xFF, 0x0D]
        _done = [0x02, 0x00, 0x0D]
        _diag_0 = [0x04, 0x00, 0x00, 0x0D]  # DIAG_STAT 0

        _a352 = A352(
            port="/dev/AAA",
            baud=100,
            product_id="A352XXX",
            sps=1000,
            logger_id="hoge",
            file_rotate_min=1,
            sensor_data_diag=True,
            sensor_data_diag_sec=0.1,
        )

        def _ok_result(self) -> SelfTestResult:
            return SelfTestResult(
                model=self._a352.model,
                serial=self._a352.serial,
                acceleration=Constants.TEST_OK,
                sensitivity=f"X:{Constants.TEST_OK},Y:{Constants.TEST_OK},Z:{Constants.TEST_OK}",
                temperature=Constants.TEST_OK,
                power_voltage=Constants.TEST_OK,
                flash_memory=Constants.TEST_OK,
                structural_resonance=Constants.TEST_NOT_AVAILABLE,
            )

        def test_self_test_ok(self, mock_comm):
            send_sidefx = [
                self._void,  # 0b00000111 : acceleration, temperature, power voltage
                self._cont,
                self._done,
                self._diag_0,
                self._void,  # 0b00001000 : non-volatile memory
                self._cont,
                self._done,
                self._diag_0,
                self._void,  # 0b00010000 : X-axis sensitivity
                self._cont,
                self._done,
                self._diag_0,
                self._void,  # 0b00100000 : Y-axis sensitivity
                self._cont,
                self._done,
                self._diag_0,
                self._void,  # 0b01000000 : Z-axis sensitivity
                self._cont,
                self._done,
                self._diag_0,
            ]
            mock_comm(send_sidefx=send_sidefx)

            # Execute
            result = self._a352.self_test()

            # Judge
            assert result == self._ok_result()

        def test_self_test_ng_ptn1(self, mock_comm):
            send_sidefx = [
                self._void,  # 0b00000111 : acceleration, temperature, power voltage
                self._done,
                [0x04, 0x00, 0b00000010, 0x0D],  # DIAG_STAT ACC_ERR_ALL=1
                self._void,  # 0b00001000 : non-volatile memory
                self._done,
                [0x04, 0x00, 0b00000100, 0x0D],  # DIAG_STAT FLASH_ERR=1
                self._void,  # 0b00010000 : X-axis sensitivity
                self._done,
                [0x04, 0b00001100, 0x00, 0x0D],  # DIAG_STAT SENS_ERR=11
                self._void,  # 0b00100000 : Y-axis sensitivity
                self._done,
                self._diag_0,
                self._void,  # 0b01000000 : Z-axis sensitivity
                self._done,
                self._diag_0,
            ]
            mock_comm(send_sidefx=send_sidefx)

            # Execute
            result = self._a352.self_test()

            # Judge
            expect = self._ok_result()
            expect.acceleration = Constants.TEST_NG
            expect.flash_memory = Constants.TEST_NG
            expect.sensitivity = "X:{},Y:{},Z:{}".format(
                Constants.TEST_NG,
                Constants.TEST_OK,
                Constants.TEST_OK,
            )

            assert result == expect

        def test_self_test_ng_ptn2(self, mock_comm):
            send_sidefx = [
                self._void,  # 0b00000111 : acceleration, temperature, power voltage
                self._done,
                [0x04, 0b00000010, 0x00, 0x0D],  # DIAG_STAT TEMP_ERR=1
                self._void,  # 0b00001000 : non-volatile memory
                self._done,
                self._diag_0,
                self._void,  # 0b00010000 : X-axis sensitivity
                self._done,
                self._diag_0,
                self._void,  # 0b00100000 : Y-axis sensitivity
                self._done,
                [0x04, 0b00001100, 0x00, 0x0D],  # DIAG_STAT SENS_ERR=11
                self._void,  # 0b01000000 : Z-axis sensitivity
                self._done,
                self._diag_0,
            ]
            mock_comm(send_sidefx=send_sidefx)

            # Execute
            result = self._a352.self_test()

            # Judge
            expect = self._ok_result()
            expect.temperature = Constants.TEST_NG
            expect.sensitivity = "X:{},Y:{},Z:{}".format(
                Constants.TEST_OK,
                Constants.TEST_NG,
                Constants.TEST_OK,
            )

            assert result == expect

        def test_self_test_ng_ptn3(self, mock_comm):
            send_sidefx = [
                self._void,  # 0b00000111 : acceleration, temperature, power voltage
                self._done,
                [0x04, 0b00000001, 0x00, 0x0D],  # DIAG_STAT VDD_ERR=1
                self._void,  # 0b00001000 : non-volatile memory
                self._done,
                self._diag_0,
                self._void,  # 0b00010000 : X-axis sensitivity
                self._done,
                self._diag_0,
                self._void,  # 0b00100000 : Y-axis sensitivity
                self._done,
                self._diag_0,
                self._void,  # 0b01000000 : Z-axis sensitivity
                self._done,
                [0x04, 0b00001100, 0x00, 0x0D],  # DIAG_STAT SENS_ERR=11
            ]
            mock_comm(send_sidefx=send_sidefx)

            # Execute
            result = self._a352.self_test()

            # Judge
            expect = self._ok_result()
            expect.power_voltage = Constants.TEST_NG
            expect.sensitivity = "X:{},Y:{},Z:{}".format(
                Constants.TEST_OK,
                Constants.TEST_OK,
                Constants.TEST_NG,
            )

            assert result == expect
