import json
import random
import time
from typing import Literal

import paho.mqtt.publish as publish
from topic_type import TopicType

from raspi_web.common import config
from raspi_web.data.vc_data import VCLevelType
from raspi_web.utils.types.error import ErrorType

HOST = config.get_str("MQTT_BROKER_HOST")
PORT = config.get_int("MQTT_BROKER_PORT")


def _publish_message(topic, payload):

    publish.single(
        topic=topic,
        payload=payload,
        hostname=HOST,
        port=PORT,
        retain=True,
    )


# ロガー
def publish_logger_status(logger_id: str, status: Literal["START", "STOP"]):
    topic = f"logger/{logger_id}"

    timestamp = time.strftime("%Y/%m/%d %H:%M:%S", time.localtime())

    payload = {"timestamp": timestamp, "status": status}

    _publish_message(topic=topic, payload=json.dumps(payload))


def publish_logger_error(logger_id: str, level: ErrorType, message: str):
    topic = f"logger/{logger_id}/error"

    timestamp = time.strftime("%Y/%m/%d %H:%M:%S", time.localtime())

    payload = {"timestamp": timestamp, "level": level, "message": message}

    _publish_message(topic=topic, payload=json.dumps(payload))


# センサー
def publish_sensor_status(
    logger_id: str, model: str, serial: str, status: Literal["OK", "NG"]
):
    topic = f"logger/{logger_id}/sensor/{model}/{serial}"

    timestamp = time.strftime("%Y/%m/%d %H:%M:%S", time.localtime())

    payload = {"timestamp": timestamp, "status": status}

    _publish_message(topic=topic, payload=json.dumps(payload))


def publish_sensor_error(
    logger_id: str, model: str, serial: str, level: ErrorType, message: str
):
    topic = f"logger/{logger_id}/sensor/{model}/{serial}/error"

    timestamp = time.strftime("%Y/%m/%d %H:%M:%S", time.localtime())

    payload = {"timestamp": timestamp, "level": level, "message": message}

    _publish_message(topic=topic, payload=json.dumps(payload))


def publish_sensor_loss(logger_id: str, model: str, serial: str, message: str):
    topic = f"logger/{logger_id}/sensor/{model}/{serial}/loss"

    timestamp = time.strftime("%Y/%m/%d %H:%M:%S", time.localtime())

    payload = {"timestamp": timestamp, "message": message}

    _publish_message(topic=topic, payload=json.dumps(payload))


def publish_sensor_abnormal(logger_id: str, model: str, serial: str, message: str):
    topic = f"logger/{logger_id}/sensor/{model}/{serial}/abnormal"

    timestamp = time.strftime("%Y/%m/%d %H:%M:%S", time.localtime())

    payload = {"timestamp": timestamp, "message": message}

    _publish_message(topic=topic, payload=json.dumps(payload))


# ハードウェアモニター
def publish_hwmonitor_status(logger_id: str, status: Literal["START", "STOP"]):
    topic = f"hwmonitor/{logger_id}"

    timestamp = time.strftime("%Y/%m/%d %H:%M:%S", time.localtime())

    payload = {"timestamp": timestamp, "status": status}

    _publish_message(topic=topic, payload=json.dumps(payload))


def publish_hwmonitor_data(
    logger_id: str,
    cpu_temperature: float,
    cpu_usage: float,
    memory_usage: float,
    disk_usage: float,
):
    topic = f"hwmonitor/{logger_id}/data"

    timestamp = time.strftime("%Y/%m/%d %H:%M:%S", time.localtime())

    payload = {
        "timestamp": timestamp,
        "cpu_temperature": cpu_temperature,
        "cpu_usage": cpu_usage,
        "memory_usage": memory_usage,
        "disk_usage": disk_usage,
    }

    _publish_message(topic=topic, payload=json.dumps(payload))


def publish_hwmonitor_error(logger_id: str, level: ErrorType, message: str):
    topic = f"hwmonitor/{logger_id}/error"

    timestamp = time.strftime("%Y/%m/%d %H:%M:%S", time.localtime())

    payload = {"timestamp": timestamp, "level": level, "message": message}

    _publish_message(topic=topic, payload=json.dumps(payload))


def publish_vc_data_level(logger_id: str, model: str, serial: str, level: VCLevelType):
    topic = f"logger/{logger_id}/sensor/{model}/{serial}/vc"

    timestamp = time.strftime("%Y/%m/%d %H:%M:%S", time.localtime())

    payload = {"timestamp": timestamp, "level": level}

    _publish_message(topic=topic, payload=json.dumps(payload))


def publish_vc_data_fft(logger_id: str, model: str, serial: str, fft: list[float]):
    topic = f"logger/{logger_id}/sensor/{model}/{serial}/fft"

    timestamp = time.strftime("%Y/%m/%d %H:%M:%S", time.localtime())

    payload = {"timestamp": timestamp, "value": fft}

    _publish_message(topic=topic, payload=json.dumps(payload))


def main(logger_id: str, model: str, serial: str, topic_type: TopicType):  # noqa: C901
    """指定したトピックにサンプルメッセージを1秒おきに10回送信する

    Args:
        logger_id (str): サンプルトピックに用いるロガーID
        model (str): サンプルトピックに用いるセンサーモデル
        serial (str): サンプルトピックに用いるセンサーシリアル
        topic_type (TopicType): トピックの種類
    """
    for i in range(10):
        match topic_type:
            # ロガー
            case "logger_status":
                status = "START" if i % 3 == 0 else "STOP"
                publish_logger_status(logger_id=logger_id, status=status)  # type: ignore

            case "logger_error":
                level = "CRITICAL" if i % 3 == 0 else "ERROR"
                message = "ErrorA" if i % 2 == 0 else "ErrorB"
                publish_logger_error(logger_id=logger_id, level=level, message=message)  # type: ignore

            # センサー
            case "sensor_status":
                status = "OK" if i % 3 == 0 else "NG"
                publish_sensor_status(logger_id=logger_id, model=model, serial=serial, status=status)  # type: ignore

            case "sensor_error":
                level = "CRITICAL" if i % 3 == 0 else "ERROR"
                message = "ErrorA" if i % 2 == 0 else "ErrorB"
                publish_sensor_error(logger_id=logger_id, model=model, serial=serial, level=level, message=message)  # type: ignore

            case "sensor_loss":
                message = "LossA" if i % 2 == 0 else "LossB"
                publish_sensor_loss(logger_id=logger_id, model=model, serial=serial, message=message)  # type: ignore

            case "sensor_abnormal":
                message = "AbnormalA" if i % 2 == 0 else "AbnormalB"
                publish_sensor_abnormal(logger_id=logger_id, model=model, serial=serial, message=message)  # type: ignore

            # ハードウェアモニター
            case "hwmonitor_status":
                status = "START" if i % 3 == 0 else "STOP"
                publish_hwmonitor_status(logger_id=logger_id, status=status)  # type: ignore

            case "hwmonitor_data":
                cpu_temperature = 10 if i % 2 == 0 else 20
                cpu_usage = 1 if i % 3 == 0 else 3
                memory_usage = 100 if i % 2 == 0 else 50
                disk_usage = 10 if i % 3 == 0 else 30

                publish_hwmonitor_data(
                    logger_id=logger_id,
                    cpu_temperature=cpu_temperature,
                    cpu_usage=cpu_usage,
                    memory_usage=memory_usage,
                    disk_usage=disk_usage,
                )  # type: ignore

            case "hwmonitor_error":
                level = "CRITICAL" if i % 3 == 0 else "ERROR"
                message = "ErrorA" if i % 2 == 0 else "ErrorB"
                publish_hwmonitor_error(logger_id=logger_id, level=level, message=message)  # type: ignore

            # VCデータ
            case "vc_data_level":
                level = "OA" if i % 3 == 0 else "G"
                publish_vc_data_level(logger_id=logger_id, model=model, serial=serial, level=level)  # type: ignore

            case "vc_data_fft":
                fft = [random.uniform(1e-9, 1e2) for _ in range(27)]
                publish_vc_data_fft(logger_id=logger_id, model=model, serial=serial, fft=fft)  # type: ignore

        time.sleep(1)


if __name__ == "__main__":
    logger_id = "0"
    model = "A352"
    serial = "0001"

    topic_type: TopicType = "logger_status"

    main(logger_id=logger_id, model=model, serial=serial, topic_type=topic_type)
