# Copyright 2025, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

from collections import deque
from typing import Literal

from raspi_web.data.logger import Logger, LoggerError, LoggerStatus
from raspi_web.utils.constants import _MAX_MESSAGE_COUNT
from raspi_web.utils.types.error import ErrorType


class TestLogger:

    def test_update_status(self):
        logger = Logger(logger_id="0")
        expect: Literal["START", "STOP"] = "START"
        update_info = LoggerStatus(timestamp="2000/11/11 11:11:11", status=expect)

        logger.update_status(status=update_info)

        assert logger.status is not None
        assert expect == logger.status.status

    def test_update_status_to_none(self):
        logger = Logger(
            logger_id="0",
            status=LoggerStatus(timestamp="2000/11/11 11:11:11", status="START"),
        )
        update_info = None

        logger.update_status(status=update_info)

        assert logger.status is None

    def test_update_error(self):
        logger = Logger(logger_id="0")
        expect_error_level: ErrorType = "CRITICAL"
        expect_error_message = "critical error"
        update_info = LoggerError(
            timestamp="2000/11/11 11:11:11",
            level=expect_error_level,
            message=expect_error_message,
        )

        logger.update_error(error=update_info)

        assert 1 == len(logger.error)
        assert expect_error_level == logger.error[0].level
        assert expect_error_message == logger.error[0].message

    def test_update_error_to_none(self):
        logger = Logger(
            logger_id="0",
            error=deque[LoggerError](
                [
                    LoggerError(
                        timestamp="2000/11/11 11:11:11",
                        level="CRITICAL",
                        message="critical error",
                    )
                ],
                maxlen=_MAX_MESSAGE_COUNT,
            ),
        )
        update_info = None

        logger.update_error(error=update_info)

        assert 0 == len(logger.error)

    def test_update_error_over_limit(self):
        error0_message = "error0_message"
        error1_message = "error1_message"
        error2_message = "error2_message"

        error0 = LoggerError(
            timestamp="2000/11/11 00:00:00", level="CRITICAL", message=error0_message
        )
        error1 = LoggerError(
            timestamp="2000/11/11 10:00:00", level="CRITICAL", message=error1_message
        )
        error2 = LoggerError(
            timestamp="2000/11/11 20:00:00", level="CRITICAL", message=error2_message
        )

        logger = Logger(
            logger_id="0",
            error=deque[LoggerError](
                [error0, error1, error2], maxlen=_MAX_MESSAGE_COUNT
            ),
        )

        over_error_message = "errorover_message"
        over_error = LoggerError(
            timestamp="2000/11/11 23:00:00",
            level="CRITICAL",
            message=over_error_message,
        )

        update_info = over_error

        logger.update_error(error=update_info)

        assert _MAX_MESSAGE_COUNT == len(logger.error)
        # 古いエラーが押し出される
        # The oldest error is pushed out
        assert error1 == logger.error[0]
        # 新しいエラーが末尾
        # The newest error is at the end
        assert over_error == logger.error[-1]

    def test_is_empty(self):
        logger = Logger(logger_id="0")
        expect = True

        actual = logger.is_empty()

        assert expect == actual

    def test_is_empty_exist_status(self):
        logger = Logger(
            logger_id="0",
            status=LoggerStatus(timestamp="2000/11/11 11:11:11", status="START"),
        )

        expect = False

        actual = logger.is_empty()

        assert expect == actual

    def test_is_empty_exist_error(self):
        logger = Logger(
            logger_id="0",
            error=deque(
                [
                    LoggerError(
                        timestamp="2000/11/11 11:11:11",
                        level="CRITICAL",
                        message="err発生",
                    )
                ],
                maxlen=_MAX_MESSAGE_COUNT,
            ),
        )

        expect = False

        actual = logger.is_empty()

        assert expect == actual
