# Copyright 2025, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

from collections import deque
from dataclasses import dataclass, field
from typing import Literal

from raspi_web.utils.constants import _MAX_MESSAGE_COUNT
from raspi_web.utils.types.error import ErrorType


@dataclass
class HardwareStatus:
    """
    ラズベリーパイのハードウェア状態

    Raspberry Pi hardware status

    Attributes:
        timestamp (str): タイムスタンプ
                Timestamp
        status (LoggerStatusType): ラズベリーパイの状態
                Raspberry Pi status
    """
    timestamp: str
    status: Literal["START", "STOP"]


@dataclass
class HardwareData:
    """
    ラズベリーパイのハードウェアデータ

    Raspberry Pi hardware data

    Attributes:
        timestamp (str): タイムスタンプ
                Timestamp
        cpu_temperature (float): CPU温度
                CPU temperature
        cpu_usage (float): CPU使用率
                CPU usage
        memory_usage (float): メモリ使用率
                Memory usage
        disk_usage (float): ディスク使用率
                Disk usage
    """
    timestamp: str
    cpu_temperature: float
    cpu_usage: float
    memory_usage: float
    disk_usage: float


@dataclass
class HardwareError:
    """
    ラズベリーパイのハードウェアエラー

    Raspberry Pi hardware error

    Attributes:
        timestamp (str): タイムスタンプ
                Timestamp
        level (ErrorType): エラーの深刻度
                Error severity
        message (str): エラーのメッセージ
                Error message
    """
    timestamp: str
    level: ErrorType
    message: str


@dataclass
class HardwareMonitor:
    """
    ラズベリーパイのハードウェアモニター

    Raspberry Pi hardware monitor

    Attributes:
        logger_id (str): ロガーID
                Logger ID
        status (HardwareStatus): ハードウェアの状態. デフォルトはNone.
                Hardware status. Defaults to None.
        hardware_data (HardwareData): ハードウェアのデータ. デフォルトはNone.
                Hardware data. Defaults to None.
        error (deque[HardwareError]): ハードウェアエラー. 最大3件まで蓄積. 古いものから削除. デフォルトは[].
                Hardware errors. Stores up to 3 errors. Removes oldest first. Defaults to [].
    """
    logger_id: str
    status: HardwareStatus | None = None
    data: HardwareData | None = None
    error: deque[HardwareError] = field(
        default_factory=lambda: deque(maxlen=_MAX_MESSAGE_COUNT)
    )

    def update_status(self, status: HardwareStatus | None):
        """
        ハードウェアモニターのステータスを更新する

        Update the hardware monitor status

        Args:
            status (HardWareStatus | None): 更新したい値
                    Value to update
        """
        self.status = status

    def update_data(self, data: HardwareData | None):
        """
        ハードウェアモニターのデータを更新する

        Update the hardware monitor data

        Args:
            data (HardWareData | None): 更新したい値
                    Value to update
        """
        self.data = data

    def update_error(self, error: HardwareError | None):
        """
        ハードウェアモニターのエラーを更新する

        Update the hardware monitor errors

        Args:
            error (HardwareError | None): 更新したい値
                    Value to update
        """
        if error is not None:
            self.error.append(error)
        else:
            self.error = deque[HardwareError](maxlen=_MAX_MESSAGE_COUNT)

    def is_empty(self) -> bool:
        """
        ハードウェアモニターの各要素が空かどうかを判断する

        Check if all elements of the hardware monitor are empty

        Returns:
            bool: 空ならTrue, 1箇所でも値があればFalse
                    True if empty, False if any element has a value
        """
        return not (self.status or self.data or self.error)
