# Copyright 2025, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

import copy
import json
import re
import threading
from dataclasses import dataclass
from typing import Callable, ParamSpec, Self, TypeVar

from raspi_web.data.data_set import DataSet
from raspi_web.data.hwmonitor import (
    HardwareData,
    HardwareError,
    HardwareMonitor,
    HardwareStatus,
)
from raspi_web.data.logger import Logger, LoggerError, LoggerStatus
from raspi_web.data.sensor import Sensor, SensorError, SensorMessage, SensorStatus
from raspi_web.data.vc_data import FFTData, VCData, VCLevel

from .topic import HARDWAREMONITOR_TOPIC_BASE, LOGGER_TOPIC_BASE

# デコレータの引数にとる関数の引数の型のパラメータ仕様変数
# Parameter specification variable for the types of arguments of the function taken by the decorator
P = ParamSpec("P")
# デコレータの引数にとる関数の戻り値の型の変数
# Type variable for the return type of the function taken by the decorator
R = TypeVar("R")

@dataclass
class TopicPattern:
    """
    トピックの文字列とトピックごとのハンドラーをセットで持つ

    Holds a set of topic strings and handlers for each topic

    Attributes:
        topic_string (str): トピックの文字列. トピックが動的に変わる場合には正規表現で記載.
                Topic string. If the topic changes dynamically, it is described using regular expressions.
        handler(Callable): トピックごとに呼び出すハンドラー
                Handler called for each topic
    """
    topic_string: str
    handler: Callable


class DataManager:
    """
    MQTTブローカーから送られてきたデータを保持する

    MQTTサブスクライバーからの通知を受け取り, トピックごとに処理を行い, データを保持する

    Dashからの要望に合わせデータを渡す

    Holds data sent from the MQTT broker

    Receives notifications from MQTT subscribers, processes them by topic, and retains the data

    Provides data according to requests from Dash
    """
    # シングルトン設定
    ## シングルトンインスタンス
    # Singleton configuration
    ## Singleton instance
    _instance: Self | None = None

    ## - シングルトンを成立させるため、通常のインスタンス生成を抑止
    ## - Prevents normal instance creation to enforce singleton pattern
    def __new__(cls):
        raise NotImplementedError("Cannot generate instance by constructor")

    @classmethod
    def __internal_new__(cls):
        instance = super().__new__(cls)
        instance.__init__()  # type: ignore
        return instance

    @classmethod
    def get_instance(cls) -> Self:
        """
        シングルトンの DataManager インスタンスを取得するメソッド

        Method to get the singleton instance of DataManager

        Returns:
            DataManager: データマネージャー
                    Data manager
        """
        if not cls._instance:
            cls._instance = cls.__internal_new__()
        return cls._instance  # type: ignore

    # 実処理
    # Main processing
    def __init__(self):
        """
        データマネージャーの初期化

        Initialization of the Data Manager
        """
        # data
        # ロガーIDがキー、そのロガーIDのデータがバリュー. デフォルトは{}.
        # Logger ID as key, data of that logger ID as value. Defaults to {}.
        self._data_sets: dict[str, DataSet] = {}
        # {ロガーID}#{モデル}#{シリアル}がキー、そのセンサーのVCデータがバリュー. デフォルトは{}.
        # {Logger ID}#{Model}#{Serial} as key, VC data of that sensor as value. Defaults to {}.
        self._vc_data: dict[str, VCData] = {}
        # データアクセスを制限するスレッドロック
        # Thread lock to restrict data access
        self._thread_lock = threading.Lock()

    @staticmethod
    def _thread_lock_decorator(func: Callable[P, R]) -> Callable[P, R]:
        """
        スレッドをロックするデコレーター

        MQTTサブスクライバーが別スレッドで動いているため, スレッドをロックしデータの更新と読み出しの整合性をとる

        Decorator to lock the thread

        Since MQTT subscribers run on separate threads, lock the thread to ensure consistency of data updates and reads

        Args:
            func (Callable[P, R]): 保持しているデータにアクセスする関数
                    Function that accesses the retained data

        Returns:
            Callable[P, R]: 保持しているデータにアクセスする関数に処理中はスレッドをロックする機能をつけた関数
                    Function that accesses the retained data with thread locking during processing
        """
        def _wrapper(*args: P.args, **kwargs: P.kwargs):
            # クラス内のデコレータを作る際にargsにselfも含まれるためそれを使う args[0] == self
            # When creating a decorator within a class, use self included in args (args[0] == self)
            with args[0]._thread_lock:  # type: ignore
                result = func(*args, **kwargs)
            return result

        return _wrapper

    # コマンド系
    ## データセット
    # Command-related
    ## Data set
    def _create_data_set_frame(self, logger_id: str) -> DataSet:
        """
        _data_setsにデータセットの枠を作成

        Create a frame for the data set in _data_sets

        Args:
            logger_id (str): ロガーID
                    Logger ID

        Returns:
            DataSet: データセット
                    Data set
        """
        if logger_id not in self._data_sets:
            self._data_sets[logger_id] = DataSet()

        return self._data_sets[logger_id]

    def _remove_data_set_frame(self, logger_id: str):
        """
        データセットが空なら_data_setsからデータセットの枠を取り除く

        Remove the data set frame from _data_sets if the data set is empty

        Args:
            logger_id (str): ロガーID
                    Logger ID
        """
        data_set = self._data_sets.get(logger_id)
        if data_set is None:
            return
        if data_set.is_empty():
            self._data_sets.pop(logger_id)

    ## VCデータ
    ## VC Data
    def _create_vc_data_key(self, logger_id: str, model: str, serial: str) -> str:
        """
        _vc_dataのキーを作成する

        Create a key for _vc_data

        Args:
            logger_id (str): ロガーID
                    Logger ID
            model (str): モデル
                    Model
            serial (str): シリアル番号
                    Serial number

        Returns:
            str: _vc_dataのキー
                    Key for _vc_data
        """
        return f"{logger_id}#{model}#{serial}"

    def _create_vc_data_frame(self, vc_data_key: str) -> VCData:
        """
        _vc_dataにvcデータの枠を作成

        Create a frame for VC data in _vc_data

        Args:
            vc_data_key (str): _vc_dataのキー
                    Key for _vc_data

        Returns:
            VCData: VCデータ
                    VC data
        """
        if vc_data_key not in self._vc_data:
            self._vc_data[vc_data_key] = VCData()

        return self._vc_data[vc_data_key]

    def _remove_vc_data_frame(self, vc_data_key: str):
        """
        vcデータが空なら_vc_dataからvcデータの枠を取り除く

        Remove the VC data frame from _vc_data if the VC data is empty

        Args:
            vc_data_key (str): _vc_dataのキー
                    Key for _vc_data
        """
        vc_data = self._vc_data.get(vc_data_key)
        if vc_data is None:
            return
        if vc_data.is_empty():
            self._vc_data.pop(vc_data_key)

    # トピックハンドラー
    ## ロガー
    # Topic handler
    ## Logger
    def _preprocess_logger_handler(
        self, logger_topic: str
    ) -> tuple[str, DataSet, Logger]:
        """
        ロガーハンドラーの前処理としてロガートピックからIDの抽出、データセット、ロガーの取得を行う

        Preprocess logger handler to extract ID from logger topic, obtain data set, and logger

        Args:
            logger_topic (str): ロガートピック
                    Logger topic

        Returns:
            tuple[str, DataSet, Logger]: ロガーID、データセット、ロガー
                    Logger ID, data set, logger
        """
        topic_parts = logger_topic.split("/")
        logger_id = topic_parts[1]

        data_set = self._create_data_set_frame(logger_id=logger_id)

        # ロガーの取得
        # Obtain logger
        logger = data_set.get_logger(logger_id=logger_id)

        return logger_id, data_set, logger

    def _postprocess_logger_handler(self, logger_id: str, data_set: DataSet):
        """
        ロガーハンドラーの後処理として必要があればロガーの削除、データセットの削除を行う

        Post-process logger handler to delete logger and data set if necessary

        Args:
            logger_id (str): ロガーID
                    Logger ID
            data_set (DataSet): データセット
                    Data set
        """
        # ロガーのid以外の各フィールドがNoneなら削除
        # Delete if all fields except logger ID are None
        data_set.clean_logger()

        # データセットが空なら削除
        # Delete if data set is empty
        self._remove_data_set_frame(logger_id=logger_id)

    @_thread_lock_decorator
    def _handle_logger_status(self, topic: str, message_dict: dict):
        """
        logger/$LOGGERトピックを処理

        Process logger/$LOGGER topic

        Args:
            topic (str): logger/$LOGGER_ID
            message_dict (dict): {
                                    "timestamp": "yyyy/mm/dd hh:mm:ss.mmmmmm",
                                    "status": "START"|"STOP"
                                }
        """
        logger_id, data_set, logger = self._preprocess_logger_handler(
            logger_topic=topic
        )

        # 更新処理
        # Update processing
        timestamp = message_dict.get("timestamp")
        logger_status = message_dict.get("status")

        update_info: LoggerStatus | None

        if logger_status is None:
            update_info = None
        else:
            update_info = LoggerStatus(timestamp=timestamp, status=logger_status)  # type: ignore

        logger.update_status(update_info)

        self._postprocess_logger_handler(logger_id=logger_id, data_set=data_set)

    @_thread_lock_decorator
    def _handle_logger_error(self, topic: str, message_dict: dict):
        """
        logger/$LOGGER/errorトピックを処理

        Process logger/$LOGGER/error topic

        Args:
            topic (str): logger/$LOGGER_ID/error
            message_dict (dict): {
                                    "timestamp": "yyyy/mm/dd hh:mm:ss.mmmmmm",
                                    "level": "CRITICAL"|"ERROR"|"WARNING",
                                    "message": ...
                                }
        """
        logger_id, data_set, logger = self._preprocess_logger_handler(
            logger_topic=topic
        )

        data_set = self._create_data_set_frame(logger_id=logger_id)

        # ロガーの取得
        # Obtain logger
        logger = data_set.get_logger(logger_id=logger_id)

        # 更新処理
        # Update processing
        timestamp = message_dict.get("timestamp")
        level = message_dict.get("level")
        message = message_dict.get("message")

        update_info: LoggerError | None

        if level is None:
            update_info = None
        else:
            update_info = LoggerError(timestamp=timestamp, level=level, message=message)  # type: ignore

        logger.update_error(error=update_info)

        self._postprocess_logger_handler(logger_id=logger_id, data_set=data_set)

    ## センサー
    ## Sensor
    def _preprocess_sensor_handler(
        self, sensor_topic: str
    ) -> tuple[str, str, str, DataSet, Sensor]:
        """
        センサーハンドラーの前処理としてセンサートピックからIDの抽出、データセット、センサーの取得を行う

        Preprocess sensor handler to extract ID from sensor topic, obtain data set, and sensor

        Args:
            sensor_topic (str): センサートピック
                    Sensor topic

        Returns:
            tuple[str, str, str, DataSet, Sensor]: ロガーID、モデル、シリアル、データセット、センサー
                    Logger ID, model, serial, data set, sensor
        """
        topic_parts = sensor_topic.split("/")
        logger_id = topic_parts[1]
        model = topic_parts[3]
        serial = topic_parts[4]

        data_set = self._create_data_set_frame(logger_id=logger_id)

        # センサーの取得
        # Obtain sensor
        sensor = data_set.get_sensor(logger_id=logger_id, model=model, serial=serial)

        return logger_id, model, serial, data_set, sensor

    def _postprocess_sensor_handler(
        self, logger_id: str, model: str, serial: str, data_set: DataSet
    ):
        """
        センサーハンドラーの後処理として必要があればセンサーの削除、データセットの削除を行う

        Post-process sensor handler to delete sensor and data set if necessary

        Args:
            logger_id (str): ロガーID
                    Logger ID
            model (str): モデル
                    Model
            serial (str): シリアル
                    Serial
            data_set (DataSet): データセット
                    Data set
        """
        # ロガーのid, model, serial以外の各フィールドがNoneなら削除
        # Delete if all fields except logger ID, model, and serial are None
        data_set.clean_sensor(model=model, serial=serial)

        # データセットが空なら削除
        # Delete if data set is empty
        self._remove_data_set_frame(logger_id=logger_id)

    @_thread_lock_decorator
    def _handle_sensor_status(self, topic: str, message_dict: dict):
        """
        logger/$LOGGER/sensor/$model/$serialトピックを処理

        Process logger/$LOGGER/sensor/$model/$serial topic

        Args:
            topic (str): logger/$LOGGER_ID/sensor/$model/$serial
            message_dict (dict): {
                                    "timestamp": "yyyy/mm/dd hh:mm:ss.mmmmmm",
                                    "status": "OK"|"NG"
                                }
        """
        logger_id, model, serial, data_set, sensor = self._preprocess_sensor_handler(
            sensor_topic=topic
        )

        # 更新処理
        # Update processing
        timestamp = message_dict.get("timestamp")
        sensor_status = message_dict.get("status")

        update_info: SensorStatus | None

        if sensor_status is None:
            update_info = None
        else:
            update_info = SensorStatus(timestamp=timestamp, status=sensor_status)  # type: ignore

        sensor.update_status(status=update_info)

        self._postprocess_sensor_handler(
            logger_id=logger_id, model=model, serial=serial, data_set=data_set
        )

    @_thread_lock_decorator
    def _handle_sensor_error(self, topic: str, message_dict: dict):
        """
        logger/$LOGGER/sensor/$model/$serial/errorトピックを処理

        Process logger/$LOGGER/sensor/$model/$serial/error topic

        Args:
            topic (str): logger/$LOGGER_ID/sensor/$model/$serial/error
            message_dict (dict): {
                                    "timestamp": "yyyy/mm/dd hh:mm:ss.mmmmmm",
                                    "level": "CRITICAL"|"ERROR"|"WARNING",
                                    "message": ...
                                }
        """
        logger_id, model, serial, data_set, sensor = self._preprocess_sensor_handler(
            sensor_topic=topic
        )

        # 更新処理
        # Update processing
        timestamp = message_dict.get("timestamp")
        level = message_dict.get("level")
        error_message = message_dict.get("message")

        update_info: SensorError | None

        if level is None:
            update_info = None
        else:
            update_info = SensorError(timestamp=timestamp, level=level, message=error_message)  # type: ignore

        sensor.update_error(error=update_info)

        self._postprocess_sensor_handler(
            logger_id=logger_id, model=model, serial=serial, data_set=data_set
        )

    @_thread_lock_decorator
    def _handle_sensor_loss(self, topic: str, message_dict: dict):
        """
        logger/$LOGGER/sensor/$model/$serial/lossトピックを処理

        Process logger/$LOGGER/sensor/$model/$serial/loss topic

        Args:
            topic (str): logger/$LOGGER_ID/sensor/$model/$serial/loss
            message_dict (dict): {
                                    "timestamp": "yyyy/mm/dd hh:mm:ss.mmmmmm",
                                    "message": ...
                                }
        """
        logger_id, model, serial, data_set, sensor = self._preprocess_sensor_handler(
            sensor_topic=topic
        )

        # 更新処理
        # Update processing
        timestamp = message_dict.get("timestamp")
        loss_message = message_dict.get("message")

        update_info: SensorMessage | None

        if loss_message is None:
            update_info = None
        else:
            update_info = SensorMessage(timestamp=timestamp, message=loss_message)  # type: ignore

        sensor.update_loss(loss=update_info)

        self._postprocess_sensor_handler(
            logger_id=logger_id, model=model, serial=serial, data_set=data_set
        )

    @_thread_lock_decorator
    def _handle_sensor_abnormal(self, topic: str, message_dict: dict):
        """
        logger/$LOGGER/sensor/$model/$serial/abnormalトピックを処理

        Process logger/$LOGGER/sensor/$model/$serial/abnormal topic

        Args:
            topic (str): logger/$LOGGER_ID/sensor/$model/$serial/abnormal
            message_dict (dict): {
                                    "timestamp": "yyyy/mm/dd hh:mm:ss.mmmmmm",
                                    "message": ...
                                }
        """
        logger_id, model, serial, data_set, sensor = self._preprocess_sensor_handler(
            sensor_topic=topic
        )

        # 更新処理
        # Update processing
        timestamp = message_dict.get("timestamp")
        abnormal_message = message_dict.get("message")

        update_info: SensorMessage | None

        if abnormal_message is None:
            update_info = None
        else:
            update_info = SensorMessage(timestamp=timestamp, message=abnormal_message)  # type: ignore

        sensor.update_abnormal(abnormal=update_info)

        self._postprocess_sensor_handler(
            logger_id=logger_id, model=model, serial=serial, data_set=data_set
        )

    ## ハードウェアモニター
    ## Hardware Monitor
    def _preprocess_hwmonitor_handler(
        self, hwmonitor_topic: str
    ) -> tuple[str, DataSet, HardwareMonitor]:
        """
        ハードウェアモニターハンドラーの前処理としてハードウェアモニタートピックからIDの抽出、データセット、ハードウェアモニターの取得を行う

        Preprocess hardware monitor handler to extract ID from hardware monitor topic, obtain data set, and hardware monitor

        Args:
            hwmonitor_topic (str): ハードウェアモニタートピック
                    Hardware monitor topic

        Returns:
            tuple[str, DataSet, HardwareMonitor]: ロガーID、データセット、ハードウェアモニター
                    Logger ID, data set, hardware monitor
        """
        topic_parts = hwmonitor_topic.split("/")
        logger_id = topic_parts[1]

        data_set = self._create_data_set_frame(logger_id=logger_id)

        # ハードウェアモニターの取得
        # Obtain hardware monitor
        hwmonitor = data_set.get_hwmonitor(logger_id=logger_id)

        return logger_id, data_set, hwmonitor

    def _postprocess_hwmonitor_handler(self, logger_id: str, data_set: DataSet):
        """
        ハードウェアモニターハンドラーの後処理として必要があればハードウェアモニターの削除、データセットの削除を行う

        Post-process hardware monitor handler to delete hardware monitor and data set if necessary

        Args:
            logger_id (str): ロガーID
                    Logger ID
            data_set (DataSet): データセット
                    Data set
        """
        # ロガーのid以外の各フィールドがNoneなら削除
        # Delete if all fields except logger ID are None
        data_set.clean_hwmonitor()

        # データセットが空なら削除
        # Delete if data set is empty
        self._remove_data_set_frame(logger_id=logger_id)

    @_thread_lock_decorator
    def _handle_hwmonitor_status(self, topic: str, message_dict: dict):
        """
        hwmonitor/$LOGGERトピックを処理

        Process hwmonitor/$LOGGER topic

        Args:
            topic (str): hwmonitor/$LOGGER_ID
            message_dict (dict): {
                                    "timestamp": "yyyy/mm/dd hh:mm:ss.mmmmmm",
                                    "status": "START"|"STOP"
                                }
        """
        logger_id, data_set, hwmonitor = self._preprocess_hwmonitor_handler(
            hwmonitor_topic=topic
        )

        # 更新処理
        # Update processing
        timestamp = message_dict.get("timestamp")
        hw_status = message_dict.get("status")

        update_info: HardwareStatus | None

        if hw_status is None:
            update_info = None
        else:
            update_info = HardwareStatus(timestamp=timestamp, status=hw_status)  # type: ignore

        hwmonitor.update_status(update_info)

        self._postprocess_hwmonitor_handler(logger_id=logger_id, data_set=data_set)

    @_thread_lock_decorator
    def _handle_hwmonitor_data(self, topic: str, message_dict: dict):
        """
        hwmonitor/$LOGGER/dataトピックを処理

        Process hwmonitor/$LOGGER/data topic

        Args:
            topic (str): hwmonitor/$LOGGER_ID/data
            message_dict (dict): {
                                    "timestamp": "yyyy/mm/dd hh:mm:ss.mmmmmm",
                                    "cpu_temperature": 99.999,
                                    "cpu_usage": 99.9,
                                    "memory_usage": 99.9,
                                    "disk_usage": 99.9
                                }
        """
        logger_id, data_set, hwmonitor = self._preprocess_hwmonitor_handler(
            hwmonitor_topic=topic
        )

        # 更新処理
        # Update processing
        timestamp = message_dict.get("timestamp")
        cpu_temperature = message_dict.get("cpu_temperature")
        cpu_usage = message_dict.get("cpu_usage")
        memory_usage = message_dict.get("memory_usage")
        disk_usage = message_dict.get("disk_usage")

        update_info: HardwareData | None

        if cpu_temperature is None:
            update_info = None
        else:
            update_info = HardwareData(
                timestamp=timestamp,  # type: ignore
                cpu_temperature=cpu_temperature,  # type: ignore
                cpu_usage=cpu_usage,  # type: ignore
                memory_usage=memory_usage,  # type: ignore
                disk_usage=disk_usage,  # type: ignore
            )

        hwmonitor.update_data(update_info)

        self._postprocess_hwmonitor_handler(logger_id=logger_id, data_set=data_set)

    @_thread_lock_decorator
    def _handle_hwmonitor_error(self, topic: str, message_dict: dict):
        """
        hwmonitor/$LOGGER/errorトピックを処理

        Process hwmonitor/$LOGGER/error topic

        Args:
            topic (str): hwmonitor/$LOGGER_ID/error
            message_dict (dict): {
                                    "timestamp": "yyyy/mm/dd hh:mm:ss.mmmmmm",
                                    "level": "CRITICAL"|"ERROR"|"WARNING",
                                    "message": ...
                                }
        """
        logger_id, data_set, hwmonitor = self._preprocess_hwmonitor_handler(
            hwmonitor_topic=topic
        )

        # 更新処理
        # Update processing
        timestamp = message_dict.get("timestamp")
        level = message_dict.get("level")
        message = message_dict.get("message")

        update_info: HardwareError | None

        if level is None:
            update_info = None
        else:
            update_info = HardwareError(timestamp=timestamp, level=level, message=message)  # type: ignore

        hwmonitor.update_error(update_info)

        self._postprocess_hwmonitor_handler(logger_id=logger_id, data_set=data_set)

    ## VC判定データ
    ## VC Evaluation Data
    def _preprocess_vc_data_handler(self, vc_data_topic: str) -> tuple[str, VCData]:
        """
        VCデータハンドラーの前処理としてVCデータトピックからIDの抽出、VCデータの取得を行う

        Preprocess VC data handler to extract ID from VC data topic, obtain VC data

        Args:
            vc_data_topic (str): VCデータトピック
                    VC data topic

        Returns:
            tuple[str, VCData]: VCデータキー、VCデータ
                    VC data key, VC data
        """
        topic_parts = vc_data_topic.split("/")
        logger_id = topic_parts[1]
        model = topic_parts[3]
        serial = topic_parts[4]
        vc_data_key = self._create_vc_data_key(
            logger_id=logger_id, model=model, serial=serial
        )

        # VCデータの枠組みを作成
        # Create a framework for VC data
        vc_data = self._create_vc_data_frame(vc_data_key=vc_data_key)

        return vc_data_key, vc_data

    @_thread_lock_decorator
    def _handle_vc_data_level(self, topic: str, message_dict: dict):
        """
        logger/$LOGGER/sensor/$model/$serial/vcトピックを処理

        Process logger/$LOGGER/sensor/$model/$serial/vc topic

        Args:
            topic (str): logger/$LOGGER/sensor/$model/$serial/vc
            message_dict (dict): {
                                    "timestamp": "yyyy/mm/dd hh:mm:ss.mmmmmm",
                                    "level": "OA"
                                }
        """
        vc_data_key, vc_data = self._preprocess_vc_data_handler(vc_data_topic=topic)

        # 更新処理
        # Update processing
        timestamp = message_dict.get("timestamp")
        level = message_dict.get("level")

        update_info: VCLevel | None

        if level is None:
            update_info = None
        else:
            update_info = VCLevel(timestamp=timestamp, level=level)  # type: ignore

        vc_data.update_vc_level(update_info)

        # ロガーのid以外の各フィールドがNoneなら削除
        # Delete if all fields except logger ID are None
        self._remove_vc_data_frame(vc_data_key=vc_data_key)

    @_thread_lock_decorator
    def _handle_vc_data_fft(self, topic: str, message_dict: dict):
        """
        logger/$LOGGER/sensor/$model/$serial/fftトピックを処理

        Process logger/$LOGGER/sensor/$model/$serial/fft topic

        Args:
            topic (str): logger/$LOGGER/sensor/$model/$serial/fft
            message_dict (dict): {
                                    "timestamp": "yyyy/mm/dd hh:mm:ss.mmmmmm",
                                    "value": [0.000188, 0.000086, 0.000148, 0.000143, ...]
                                }
        """
        vc_data_key, vc_data = self._preprocess_vc_data_handler(vc_data_topic=topic)

        # 更新処理
        # Update processing
        timestamp = message_dict.get("timestamp")
        fft_value = message_dict.get("value")

        update_info: FFTData | None

        if fft_value is None:
            update_info = None
        else:
            update_info = FFTData(timestamp=timestamp, value=fft_value)  # type: ignore

        vc_data.update_fft_data(update_info)

        # ロガーのid以外の各フィールドがNoneなら削除
        # Delete if all fields except logger ID are None
        self._remove_vc_data_frame(vc_data_key=vc_data_key)

    # トピックのパターン
    # Topic patterns
    _TOPIC_PATTERN_LIST: list[TopicPattern] = [
        # ロガー
        # Logger
        TopicPattern(
            topic_string=r"{}/(\w+)".format(LOGGER_TOPIC_BASE),
            handler=_handle_logger_status,
        ),
        TopicPattern(
            topic_string=r"{}/(\w+)/error".format(LOGGER_TOPIC_BASE),
            handler=_handle_logger_error,
        ),
        # センサー
        # Sensor
        TopicPattern(
            topic_string=r"{}/(\w+)/sensor/(\w+)/(\w+)".format(LOGGER_TOPIC_BASE),
            handler=_handle_sensor_status,
        ),
        TopicPattern(
            topic_string=r"{}/(\w+)/sensor/(\w+)/(\w+)/error".format(LOGGER_TOPIC_BASE),
            handler=_handle_sensor_error,
        ),
        TopicPattern(
            topic_string=r"{}/(\w+)/sensor/(\w+)/(\w+)/loss".format(LOGGER_TOPIC_BASE),
            handler=_handle_sensor_loss,
        ),
        TopicPattern(
            topic_string=r"{}/(\w+)/sensor/(\w+)/(\w+)/abnormal".format(
                LOGGER_TOPIC_BASE
            ),
            handler=_handle_sensor_abnormal,
        ),
        # ハードウェアモニター
        # Hardware monitor
        TopicPattern(
            topic_string=r"{}/(\w+)".format(HARDWAREMONITOR_TOPIC_BASE),
            handler=_handle_hwmonitor_status,
        ),
        TopicPattern(
            topic_string=r"{}/(\w+)/data".format(HARDWAREMONITOR_TOPIC_BASE),
            handler=_handle_hwmonitor_data,
        ),
        TopicPattern(
            topic_string=r"{}/(\w+)/error".format(HARDWAREMONITOR_TOPIC_BASE),
            handler=_handle_hwmonitor_error,
        ),
        # VCデータ
        # VC data
        TopicPattern(
            topic_string=r"{}/(\w+)/sensor/(\w+)/(\w+)/vc".format(LOGGER_TOPIC_BASE),
            handler=_handle_vc_data_level,
        ),
        TopicPattern(
            topic_string=r"{}/(\w+)/sensor/(\w+)/(\w+)/fft".format(LOGGER_TOPIC_BASE),
            handler=_handle_vc_data_fft,
        ),
    ]

    def receive_notification(self, topic: str, payload: bytes):
        """
        サブスクライバーからのトピックに合わせて処理を振り分ける

        Distribute processing according to the topic from the subscriber

        Args:
            topic (str): サブスクライバーに送られてきたトピック
                    Topic sent to the subscriber
            payload (bytes): サブスクライバーに送られてきたペイロード
                    Payload sent to the subscriber
        """
        message_json = payload.decode("utf-8")

        message_dict: dict
        if message_json == "":
            # payloadがNoneの場合
            # If payload is None
            message_dict = {}
        else:
            message_dict = json.loads(message_json)

        for topic_pattern in self._TOPIC_PATTERN_LIST:
            if re.fullmatch(pattern=topic_pattern.topic_string, string=topic):
                topic_pattern.handler(self, topic, message_dict)

    # クエリ系
    ## データセット
    # Query-related
    ## Data set
    @_thread_lock_decorator
    def get_data_sets(self) -> dict[str, DataSet]:
        """
        ロガーの一覧を取得する

        Retrieve a list of loggers

        Returns:
            list[Logger]: ロガーの一覧
                    List of loggers
        """
        return copy.deepcopy(self._data_sets)

    ## VCデータ
    ## VC data
    @_thread_lock_decorator
    def get_vc_data(self, logger_id: str, model: str, serial: str) -> VCData | None:
        """
        VCデータを取得する

        Retrieve VC data

        Args:
            logger_id (str): ロガーID
                    Logger ID
            model (str): モデル
                    Model
            serial (str): シリアル
                    Serial

        Returns:
            VCData | None: VCデータ, 存在しない場合はNone
                    VC data, or None if it does not exist
        """
        vc_data_key = self._create_vc_data_key(
            logger_id=logger_id, model=model, serial=serial
        )
        if vc_data_key in self._vc_data:
            return copy.deepcopy(self._vc_data[vc_data_key])
        else:
            return None
