# Copyright 2025, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

import dash
from dash import dcc, html
from dash.dependencies import Input, Output

from raspi_web.infra.data_manager import DataManager
from raspi_web.ui.component.data_set import data_set_component

# Dashアプリケーションのインスタンスを取得
# Get the instance of the Dash application
app = dash.get_app()

# ルートパスにページを登録
# Register the page at the root path
dash.register_page(__name__, path="/")

# ホーム画面のレイアウトを定義
# Define the layout for the home screen
layout = html.Div(
    [
        html.H1("Logger List"),
        dcc.Interval(
            id="interval_component",
            # 1秒ごとに更新（ミリ秒単位）
            # Update every second (in milliseconds)
            interval=1 * 1000,
            n_intervals=0,
        ),
        # ロガーの表示
        # Display loggers
        html.Div(id="loggers_div"),
    ]
)


# インターバルで設定した秒毎に実行されるコールバック関数を定義
# Define the callback function to be executed at intervals set in seconds
@app.callback(
    Output("loggers_div", "children"), Input("interval_component", "n_intervals")
)
def update_table(n):
    """
    インターバルで設定した秒毎にDataManagerからロガーの一覧を取得し、コンポーネント化して返す

    Retrieve the list of loggers from DataManager at intervals set in seconds, and return as components

    Returns:
        list[html.Div]: ロガー一覧
                List of loggers
    """
    data_manager = DataManager.get_instance()
    data_sets = data_manager.get_data_sets()

    data_set_components = [
        data_set_component(logger_id=logger_id, data_set=data_set)
        for logger_id, data_set in data_sets.items()
    ]

    return data_set_components
