# Copyright 2025, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

import copy
import threading
import time
from collections import deque
from typing import Callable, Self

import pytest

from raspi_web.data.hwmonitor import (
    HardwareData,
    HardwareError,
    HardwareMonitor,
    HardwareStatus,
)
from raspi_web.data.logger import (
    Logger,
    LoggerError,
    LoggerStatus,
)
from raspi_web.data.sensor import Sensor, SensorError, SensorMessage, SensorStatus
from raspi_web.data.vc_data import FFTData, VCData, VCLevel, VCLevelType
from raspi_web.infra.data_manager import DataManager, DataSet
from raspi_web.utils.constants import _MAX_MESSAGE_COUNT
from tests.testutils.data_sample import get_fft_data_sample


@pytest.fixture()
def reset_data_manager():
    # setup
    data_manager = DataManager.get_instance()
    data_manager._data_sets = {}
    data_manager._vc_data = {}
    yield
    # teardown
    data_manager._data_sets = {}
    data_manager._vc_data = {}


class TestDataManager:

    # コマンド系
    ## データセット関連のテスト
    # Commands
    ## Tests related to DataSet
    def test__create_data_set_frame(self, reset_data_manager):
        expect_logger_id = "hoge"
        data_manager = DataManager.get_instance()
        data_manager._create_data_set_frame(logger_id=expect_logger_id)

        assert 1 == len(data_manager._data_sets)
        assert expect_logger_id in data_manager._data_sets

    def test__create_data_set_frame_already_exist(self, reset_data_manager):
        expect_logger_id = "hoge"
        data_manager = DataManager.get_instance()
        data_manager._data_sets[expect_logger_id] = DataSet()
        data_manager._create_data_set_frame(logger_id=expect_logger_id)

        assert 1 == len(data_manager._data_sets)

    def test__remove_data_set_frame(self, reset_data_manager):
        logger_id = "logger_sample"
        data_manager = DataManager.get_instance()
        data_manager._data_sets[logger_id] = DataSet()

        data_manager._remove_data_set_frame(logger_id=logger_id)

        assert 0 == len(data_manager._data_sets)

    def test__remove_data_set_frame_not_empty(self, reset_data_manager):
        logger_id = "logger_sample"
        data_manager = DataManager.get_instance()
        data_manager._data_sets[logger_id] = DataSet(
            logger=Logger(
                logger_id=logger_id,
                status=LoggerStatus(timestamp="2024/12/12 00:00:00", status="START"),
            )
        )

        data_manager._remove_data_set_frame(logger_id=logger_id)

        assert 1 == len(data_manager._data_sets)

    ## VCデータ関連のテスト
    ## Tests related to VC Data
    def test__create_vc_data_frame(self, reset_data_manager):
        logger_id = "hoge"
        model = "A352"
        serial = "0000"
        data_manager = DataManager.get_instance()
        expect_vc_data_key = data_manager._create_vc_data_key(
            logger_id=logger_id, model=model, serial=serial
        )
        data_manager._create_vc_data_frame(vc_data_key=expect_vc_data_key)

        assert 1 == len(data_manager._vc_data)
        assert expect_vc_data_key in data_manager._vc_data

    def test__create_vc_data_frame_already_exist(self, reset_data_manager):
        logger_id = "hoge"
        model = "A352"
        serial = "0000"
        data_manager = DataManager.get_instance()
        vc_data_key = data_manager._create_vc_data_key(
            logger_id=logger_id, model=model, serial=serial
        )
        data_manager._vc_data[vc_data_key] = VCData()
        data_manager._create_vc_data_frame(vc_data_key=vc_data_key)

        assert 1 == len(data_manager._vc_data)

    def test__remove_vc_data_frame(self, reset_data_manager):
        logger_id = "logger_sample"
        model = "A352"
        serial = "0000"
        data_manager = DataManager.get_instance()
        vc_data_key = data_manager._create_vc_data_key(
            logger_id=logger_id, model=model, serial=serial
        )
        data_manager._vc_data[vc_data_key] = VCData()

        data_manager._remove_vc_data_frame(vc_data_key=vc_data_key)

        assert 0 == len(data_manager._vc_data)

    def test__remove_vc_data_frame_not_empty(self, reset_data_manager):
        logger_id = "logger_sample"
        model = "A352"
        serial = "0000"
        data_manager = DataManager.get_instance()
        vc_data_key = data_manager._create_vc_data_key(
            logger_id=logger_id, model=model, serial=serial
        )
        data_manager._vc_data[vc_data_key] = VCData(
            vc_level=VCLevel(timestamp="2024/12/12 00:00:00", level="OA")
        )

        data_manager._remove_vc_data_frame(vc_data_key=vc_data_key)

        assert 1 == len(data_manager._vc_data)

    # トピックハンドラー系
    ## ロガー関連のテスト
    # Topic Handlers
    ## Tests related to Logger
    def test__preprocess_logger_handler(self, reset_data_manager):
        logger_id = "0"
        topic = f"logger/{logger_id}"
        data_manager = DataManager.get_instance()
        result_logger_id, result_data_set, result_logger = (
            data_manager._preprocess_logger_handler(logger_topic=topic)
        )

        assert logger_id == result_logger_id
        assert result_data_set is not None
        assert result_logger is not None

    ### ステータス関連のテスト
    ### Tests related to Status
    def test__handle_logger_status(self, reset_data_manager):
        logger_id = "0"
        expect_status = "START"
        topic = f"logger/{logger_id}"
        message_dict = {"timestamp": "yyyy/mm/dd hh:mm:ss", "status": expect_status}
        data_manager = DataManager.get_instance()
        data_manager._handle_logger_status(topic=topic, message_dict=message_dict)

        data_set = data_manager._data_sets[logger_id]

        assert expect_status == data_set.logger.status.status  # type: ignore

    def test__handle_logger_status_none(self, reset_data_manager):
        logger_id = "0"
        topic = f"logger/{logger_id}"
        message_dict: dict = {}

        data_manager = DataManager.get_instance()

        logger = Logger(
            logger_id=logger_id,
            status=LoggerStatus(timestamp="2000/01/01 01:01:01", status="START"),
            error=deque[LoggerError](
                [
                    LoggerError(
                        timestamp="2024/11/21 16:30:00",
                        level="CRITICAL",
                        message="critical error",
                    )
                ],
                maxlen=_MAX_MESSAGE_COUNT,
            ),
        )
        data_manager._data_sets[logger_id] = DataSet(logger=logger)
        data_manager._handle_logger_status(topic=topic, message_dict=message_dict)

        data_set = data_manager._data_sets[logger_id]

        assert data_set.logger is not None
        assert data_set.logger.status is None

    def test__handle_logger_status_none_remove_data_set_frame(self, reset_data_manager):
        logger_id = "0"
        topic = f"logger/{logger_id}"
        message_dict: dict = {}

        data_manager = DataManager.get_instance()

        logger = Logger(
            logger_id=logger_id,
            status=LoggerStatus(timestamp="2000/01/01 01:01:01", status="START"),
        )
        data_manager._data_sets[logger_id] = DataSet(logger=logger)
        data_manager._handle_logger_status(topic=topic, message_dict=message_dict)

        result = data_manager._data_sets.get(logger_id)

        assert result is None

    ### エラー関連のテスト
    ### Tests related to Errors
    def test__handle_logger_error(self, reset_data_manager):
        logger_id = "0"
        expect_level = "CRITICAL"
        expect_message = "error01"
        topic = f"logger/{logger_id}/error"
        message_dict = {
            "timestamp": "yyyy/mm/dd hh:mm:ss",
            "level": expect_level,
            "message": expect_message,
        }
        data_manager = DataManager.get_instance()
        data_manager._handle_logger_error(topic=topic, message_dict=message_dict)

        data_set = data_manager._data_sets[logger_id]

        assert expect_level == data_set.logger.error[0].level  # type: ignore
        assert expect_message == data_set.logger.error[0].message  # type: ignore

    def test__handle_logger_error_none(self, reset_data_manager):
        logger_id = "0"
        topic = f"logger/{logger_id}/error"
        message_dict: dict = {}

        data_manager = DataManager.get_instance()

        logger = Logger(
            logger_id=logger_id,
            status=LoggerStatus(timestamp="2000/01/01 01:01:01", status="START"),
            error=deque[LoggerError](
                [
                    LoggerError(
                        timestamp="2024/11/21 16:30:00",
                        level="CRITICAL",
                        message="critical error",
                    )
                ],
                maxlen=_MAX_MESSAGE_COUNT,
            ),
        )
        data_manager._data_sets[logger_id] = DataSet(logger=logger)
        data_manager._handle_logger_error(topic=topic, message_dict=message_dict)

        data_set = data_manager._data_sets[logger_id]

        assert data_set.logger is not None
        assert len(data_set.logger.error) == 0

    def test__handle_logger_error_none_remove_data_set_frame(self, reset_data_manager):
        logger_id = "0"
        topic = f"logger/{logger_id}/error"
        message_dict: dict = {}

        data_manager = DataManager.get_instance()

        logger = Logger(
            logger_id=logger_id,
            error=deque[LoggerError](
                [
                    LoggerError(
                        timestamp="2024/11/21 16:30:00",
                        level="CRITICAL",
                        message="critical error",
                    )
                ],
                maxlen=_MAX_MESSAGE_COUNT,
            ),
        )
        data_manager._data_sets[logger_id] = DataSet(logger=logger)
        data_manager._handle_logger_error(topic=topic, message_dict=message_dict)

        result = data_manager._data_sets.get(logger_id)

        assert result is None

    ## センサー関連のテスト
    ## Tests related to Sensor
    def test__preprocess_sensor_handler(self, reset_data_manager):
        logger_id = "0"
        model = "A352"
        serial = "0001"

        topic = f"logger/{logger_id}/sensor/{model}/{serial}"
        data_manager = DataManager.get_instance()
        (
            result_logger_id,
            result_model,
            result_serial,
            result_data_set,
            result_sensor,
        ) = data_manager._preprocess_sensor_handler(sensor_topic=topic)

        assert logger_id == result_logger_id
        assert model == result_model
        assert serial == result_serial
        assert result_data_set is not None
        assert result_sensor is not None

    ### ステータス関連のテスト
    ### Tests related to Status
    def test_handle_sensor_status(self, reset_data_manager):
        logger_id = "0"
        model = "A352"
        serial = "0001"
        expect_status = "OK"

        topic = f"logger/{logger_id}/sensor/{model}/{serial}"
        message_dict = {"timestamp": "yyyy/mm/dd hh:mm:ss", "status": expect_status}

        data_manager = DataManager.get_instance()
        data_manager._handle_sensor_status(topic=topic, message_dict=message_dict)

        data_set = data_manager._data_sets[logger_id]

        assert len(data_set.sensors) == 1
        assert expect_status == data_set.sensors[0].status.status  # type: ignore

    def test_handle_sensor_status_none(self, reset_data_manager):
        logger_id = "0"
        model = "A352"
        serial = "0001"

        topic = f"logger/{logger_id}/sensor/{model}/{serial}"
        message_dict: dict = {}

        sensor = Sensor(
            logger_id=logger_id,
            model=model,
            serial=serial,
            status=SensorStatus(timestamp="2000/01/01 00:00:00", status="OK"),
            error=deque[SensorError](
                [
                    SensorError(
                        timestamp="2000/01/01 00:00:00",
                        level="CRITICAL",
                        message="critical error",
                    )
                ],
                maxlen=_MAX_MESSAGE_COUNT,
            ),
        )

        data_manager = DataManager.get_instance()
        data_manager._data_sets[logger_id] = DataSet(sensors=[sensor])
        data_manager._handle_sensor_status(topic=topic, message_dict=message_dict)

        data_set = data_manager._data_sets[logger_id]

        assert data_set.sensors[0].status is None

    def test_handle_sensor_status_none_remove_data_set_frame(self, reset_data_manager):
        logger_id = "0"
        model = "A352"
        serial = "0001"

        topic = f"logger/{logger_id}/sensor/{model}/{serial}"
        message_dict: dict = {}

        sensor = Sensor(
            logger_id=logger_id,
            model=model,
            serial=serial,
            status=SensorStatus(timestamp="2000/01/01 00:00:00", status="OK"),
        )

        data_manager = DataManager.get_instance()
        data_manager._data_sets[logger_id] = DataSet(sensors=[sensor])
        data_manager._handle_sensor_status(topic=topic, message_dict=message_dict)

        result = data_manager._data_sets.get(logger_id)

        assert result is None

    ### エラー関連のテスト
    ### Tests related to Errors
    def test_handle_sensor_error(self, reset_data_manager):
        logger_id = "0"
        model = "A352"
        serial = "0001"
        expect_level = "CRITICAL"
        expect_message = "error01"

        topic = f"logger/{logger_id}/sensor/{model}/{serial}/error"
        message_dict = {
            "timestamp": "yyyy/mm/dd hh:mm:ss",
            "level": expect_level,
            "message": expect_message,
        }
        data_manager = DataManager.get_instance()
        data_manager._handle_sensor_error(topic=topic, message_dict=message_dict)

        data_set = data_manager._data_sets[logger_id]

        assert len(data_set.sensors) == 1
        assert expect_level == data_set.sensors[0].error[0].level  # type: ignore
        assert expect_message == data_set.sensors[0].error[0].message  # type: ignore

    def test_handle_sensor_error_none(self, reset_data_manager):
        logger_id = "0"
        model = "A352"
        serial = "0001"

        topic = f"logger/{logger_id}/sensor/{model}/{serial}/error"
        message_dict: dict = {}

        sensor = Sensor(
            logger_id=logger_id,
            model=model,
            serial=serial,
            status=SensorStatus(timestamp="2000/01/01 00:00:00", status="OK"),
            error=deque[SensorError](
                [
                    SensorError(
                        timestamp="2000/01/01 00:00:00",
                        level="CRITICAL",
                        message="critical error",
                    )
                ],
                maxlen=_MAX_MESSAGE_COUNT,
            ),
        )

        data_manager = DataManager.get_instance()
        data_manager._data_sets[logger_id] = DataSet(sensors=[sensor])
        data_manager._handle_sensor_error(topic=topic, message_dict=message_dict)

        data_set = data_manager._data_sets[logger_id]

        assert len(data_set.sensors[0].error) == 0

    def test_handle_sensor_error_none_remove_data_set_frame(self, reset_data_manager):
        logger_id = "0"
        model = "A352"
        serial = "0001"

        topic = f"logger/{logger_id}/sensor/{model}/{serial}/error"
        message_dict: dict = {}

        sensor = Sensor(
            logger_id=logger_id,
            model=model,
            serial=serial,
            error=deque[SensorError](
                [
                    SensorError(
                        timestamp="2000/01/01 00:00:00",
                        level="CRITICAL",
                        message="critical error",
                    )
                ],
                maxlen=_MAX_MESSAGE_COUNT,
            ),
        )

        data_manager = DataManager.get_instance()
        data_manager._data_sets[logger_id] = DataSet(sensors=[sensor])
        data_manager._handle_sensor_error(topic=topic, message_dict=message_dict)

        result = data_manager._data_sets.get(logger_id)

        assert result is None

    ### データ欠落発生情報関連のテスト
    ### Tests related to Data Loss Information
    def test_handle_sensor_loss(self, reset_data_manager):
        logger_id = "0"
        model = "A352"
        serial = "0001"
        expect_message = "loss01"

        topic = f"logger/{logger_id}/sensor/{model}/{serial}/loss"
        message_dict = {
            "timestamp": "yyyy/mm/dd hh:mm:ss",
            "message": expect_message,
        }
        data_manager = DataManager.get_instance()
        data_manager._handle_sensor_loss(topic=topic, message_dict=message_dict)

        data_set = data_manager._data_sets[logger_id]

        assert len(data_set.sensors) == 1
        assert expect_message == data_set.sensors[0].loss[0].message  # type: ignore

    def test_handle_sensor_loss_none(self, reset_data_manager):
        logger_id = "0"
        model = "A352"
        serial = "0001"

        topic = f"logger/{logger_id}/sensor/{model}/{serial}/loss"
        message_dict: dict = {}

        sensor = Sensor(
            logger_id=logger_id,
            model=model,
            serial=serial,
            status=SensorStatus(timestamp="2000/01/01 00:00:00", status="OK"),
            loss=deque[SensorMessage](
                [
                    SensorMessage(
                        timestamp="2000/01/01 00:00:00",
                        message="loss01",
                    )
                ],
                maxlen=_MAX_MESSAGE_COUNT,
            ),
        )

        data_manager = DataManager.get_instance()
        data_manager._data_sets[logger_id] = DataSet(sensors=[sensor])
        data_manager._handle_sensor_loss(topic=topic, message_dict=message_dict)

        data_set = data_manager._data_sets[logger_id]

        assert len(data_set.sensors[0].loss) == 0

    def test_handle_sensor_loss_none_remove_data_set_frame(self, reset_data_manager):
        logger_id = "0"
        model = "A352"
        serial = "0001"

        topic = f"logger/{logger_id}/sensor/{model}/{serial}/loss"
        message_dict: dict = {}

        sensor = Sensor(
            logger_id=logger_id,
            model=model,
            serial=serial,
            loss=deque[SensorMessage](
                [
                    SensorMessage(
                        timestamp="2000/01/01 00:00:00",
                        message="loss01",
                    )
                ],
                maxlen=_MAX_MESSAGE_COUNT,
            ),
        )

        data_manager = DataManager.get_instance()
        data_manager._data_sets[logger_id] = DataSet(sensors=[sensor])
        data_manager._handle_sensor_loss(topic=topic, message_dict=message_dict)

        result = data_manager._data_sets.get(logger_id)

        assert result is None

    ### 素子異常発生情報関連のテスト
    ### Tests related to Sensor Abnormality Information
    def test_handle_sensor_abnormal(self, reset_data_manager):
        logger_id = "0"
        model = "A352"
        serial = "0001"
        expect_message = "abnormal01"

        topic = f"logger/{logger_id}/sensor/{model}/{serial}/abnormal"
        message_dict = {
            "timestamp": "yyyy/mm/dd hh:mm:ss",
            "message": expect_message,
        }
        data_manager = DataManager.get_instance()
        data_manager._handle_sensor_abnormal(topic=topic, message_dict=message_dict)

        data_set = data_manager._data_sets[logger_id]

        assert len(data_set.sensors) == 1
        assert expect_message == data_set.sensors[0].abnormal[0].message  # type: ignore

    def test_handle_sensor_abnormal_none(self, reset_data_manager):
        logger_id = "0"
        model = "A352"
        serial = "0001"

        topic = f"logger/{logger_id}/sensor/{model}/{serial}/abnormal"
        message_dict: dict = {}

        sensor = Sensor(
            logger_id=logger_id,
            model=model,
            serial=serial,
            status=SensorStatus(timestamp="2000/01/01 00:00:00", status="OK"),
            abnormal=deque[SensorMessage](
                [
                    SensorMessage(
                        timestamp="2000/01/01 00:00:00",
                        message="abnormal01",
                    )
                ],
                maxlen=_MAX_MESSAGE_COUNT,
            ),
        )

        data_manager = DataManager.get_instance()
        data_manager._data_sets[logger_id] = DataSet(sensors=[sensor])
        data_manager._handle_sensor_abnormal(topic=topic, message_dict=message_dict)

        data_set = data_manager._data_sets[logger_id]

        assert len(data_set.sensors[0].abnormal) == 0

    def test_handle_sensor_abnormal_none_remove_data_set_frame(
        self, reset_data_manager
    ):
        logger_id = "0"
        model = "A352"
        serial = "0001"

        topic = f"logger/{logger_id}/sensor/{model}/{serial}/abnormal"
        message_dict: dict = {}

        sensor = Sensor(
            logger_id=logger_id,
            model=model,
            serial=serial,
            abnormal=deque[SensorMessage](
                [
                    SensorMessage(
                        timestamp="2000/01/01 00:00:00",
                        message="loss01",
                    )
                ],
                maxlen=_MAX_MESSAGE_COUNT,
            ),
        )

        data_manager = DataManager.get_instance()
        data_manager._data_sets[logger_id] = DataSet(sensors=[sensor])
        data_manager._handle_sensor_abnormal(topic=topic, message_dict=message_dict)

        result = data_manager._data_sets.get(logger_id)

        assert result is None

    ## ハードウェアモニター関連のテスト
    ## Tests related to Hardware Monitor
    def test__preprocess_hwmonitor_handler(self, reset_data_manager):
        logger_id = "0"
        topic = f"hwmonitor/{logger_id}"
        data_manager = DataManager.get_instance()
        result_logger_id, result_data_set, result_hwmonitor = (
            data_manager._preprocess_hwmonitor_handler(hwmonitor_topic=topic)
        )

        assert logger_id == result_logger_id
        assert result_data_set is not None
        assert result_hwmonitor is not None

    ### ステータス関連のテスト
    ### Tests related to Status
    def test__handle_hwmonitor_status(self, reset_data_manager):
        logger_id = "0"
        expect_status = "START"
        topic = f"hwmonitor/{logger_id}"
        message_dict = {"timestamp": "yyyy/mm/dd hh:mm:ss", "status": expect_status}
        data_manager = DataManager.get_instance()
        data_manager._handle_hwmonitor_status(topic=topic, message_dict=message_dict)

        data_set = data_manager._data_sets[logger_id]

        assert expect_status == data_set.hwmonitor.status.status  # type: ignore

    def test__handle_hwmonitor_status_none(self, reset_data_manager):
        logger_id = "0"
        topic = f"hwmonitor/{logger_id}"
        message_dict: dict = {}

        data_manager = DataManager.get_instance()

        hwmonitor = HardwareMonitor(
            logger_id=logger_id,
            status=HardwareStatus(timestamp="2000/01/01 01:01:01", status="START"),
            error=deque[HardwareError](
                [
                    HardwareError(
                        timestamp="2024/11/21 16:30:00",
                        level="CRITICAL",
                        message="critical error",
                    )
                ],
                maxlen=_MAX_MESSAGE_COUNT,
            ),
        )
        data_manager._data_sets[logger_id] = DataSet(hwmonitor=hwmonitor)
        data_manager._handle_hwmonitor_status(topic=topic, message_dict=message_dict)

        data_set = data_manager._data_sets[logger_id]

        assert data_set.hwmonitor is not None
        assert data_set.hwmonitor.status is None

    def test__handle_hwmonitor_status_none_remove_data_set_frame(
        self, reset_data_manager
    ):
        logger_id = "0"
        topic = f"hwmonitor/{logger_id}"
        message_dict: dict = {}

        data_manager = DataManager.get_instance()

        hwmonitor = HardwareMonitor(
            logger_id=logger_id,
            status=HardwareStatus(timestamp="2000/01/01 01:01:01", status="START"),
        )
        data_manager._data_sets[logger_id] = DataSet(hwmonitor=hwmonitor)
        data_manager._handle_hwmonitor_status(topic=topic, message_dict=message_dict)

        result = data_manager._data_sets.get(logger_id)

        assert result is None

    ### データ関連のテスト
    ### Tests related to Data
    def test__handle_hwmonitor_data(self, reset_data_manager):
        logger_id = "0"
        expect_cpu_temperature = 99.99
        expect_cpu_usage = 99.9
        expect_memory_usage = 99.9
        expect_disk_usage = 99.9
        topic = f"hwmonitor/{logger_id}/data"
        message_dict = {
            "timestamp": "2000/01/01 00:00:00",
            "cpu_temperature": expect_cpu_temperature,
            "cpu_usage": expect_cpu_usage,
            "memory_usage": expect_memory_usage,
            "disk_usage": expect_disk_usage,
        }
        data_manager = DataManager.get_instance()
        data_manager._handle_hwmonitor_data(topic=topic, message_dict=message_dict)

        data_set = data_manager._data_sets[logger_id]

        assert expect_cpu_temperature == data_set.hwmonitor.data.cpu_temperature  # type: ignore
        assert expect_cpu_usage == data_set.hwmonitor.data.cpu_usage  # type: ignore
        assert expect_memory_usage == data_set.hwmonitor.data.memory_usage  # type: ignore
        assert expect_disk_usage == data_set.hwmonitor.data.disk_usage  # type: ignore

    def test__handle_hwmonitor_data_none(self, reset_data_manager):
        logger_id = "0"
        topic = f"hwmonitor/{logger_id}/data"
        message_dict: dict = {}

        data_manager = DataManager.get_instance()

        hwmonitor = HardwareMonitor(
            logger_id=logger_id,
            status=HardwareStatus(timestamp="2000/01/01 01:01:01", status="START"),
            data=HardwareData(
                timestamp="2000/01/01 00:00:00",
                cpu_temperature=99.99,
                cpu_usage=99.99,
                memory_usage=99.9,
                disk_usage=99.9,
            ),
        )
        data_manager._data_sets[logger_id] = DataSet(hwmonitor=hwmonitor)
        data_manager._handle_hwmonitor_data(topic=topic, message_dict=message_dict)

        data_set = data_manager._data_sets[logger_id]

        assert data_set.hwmonitor is not None
        assert data_set.hwmonitor.data is None

    def test__handle_hwmonitor_data_none_remove_data_set_frame(
        self, reset_data_manager
    ):
        logger_id = "0"
        topic = f"hwmonitor/{logger_id}/data"
        message_dict: dict = {}

        data_manager = DataManager.get_instance()

        hwmonitor = HardwareMonitor(
            logger_id=logger_id,
            data=HardwareData(
                timestamp="2000/01/01 00:00:00",
                cpu_temperature=99.99,
                cpu_usage=99.99,
                memory_usage=99.9,
                disk_usage=99.9,
            ),
        )
        data_manager._data_sets[logger_id] = DataSet(hwmonitor=hwmonitor)
        data_manager._handle_hwmonitor_data(topic=topic, message_dict=message_dict)

        result = data_manager._data_sets.get(logger_id)

        assert result is None

    ### エラー関連のテスト
    ### Tests related to Errors
    def test__handle_hwmonitor_error(self, reset_data_manager):
        logger_id = "0"
        expect_level = "CRITICAL"
        expect_message = "error01"
        topic = f"hwmonitor/{logger_id}/error"
        message_dict = {
            "timestamp": "yyyy/mm/dd hh:mm:ss",
            "level": expect_level,
            "message": expect_message,
        }
        data_manager = DataManager.get_instance()
        data_manager._handle_hwmonitor_error(topic=topic, message_dict=message_dict)

        data_set = data_manager._data_sets[logger_id]

        assert expect_level == data_set.hwmonitor.error[0].level  # type: ignore
        assert expect_message == data_set.hwmonitor.error[0].message  # type: ignore

    def test__handle_hwmonitor_error_none(self, reset_data_manager):
        logger_id = "0"
        topic = f"hwmonitor/{logger_id}/error"
        message_dict: dict = {}

        data_manager = DataManager.get_instance()

        hwmonitor = HardwareMonitor(
            logger_id=logger_id,
            status=HardwareStatus(timestamp="2000/01/01 01:01:01", status="START"),
            error=deque[HardwareError](
                [
                    HardwareError(
                        timestamp="2024/11/21 16:30:00",
                        level="CRITICAL",
                        message="critical error",
                    )
                ],
                maxlen=_MAX_MESSAGE_COUNT,
            ),
        )
        data_manager._data_sets[logger_id] = DataSet(hwmonitor=hwmonitor)
        data_manager._handle_hwmonitor_error(topic=topic, message_dict=message_dict)

        data_set = data_manager._data_sets[logger_id]

        assert data_set.hwmonitor is not None
        assert len(data_set.hwmonitor.error) == 0

    def test__handle_hwmonitor_error_none_remove_data_set_frame(
        self, reset_data_manager
    ):
        logger_id = "0"
        topic = f"hwmonitor/{logger_id}/error"
        message_dict: dict = {}

        data_manager = DataManager.get_instance()

        hwmonitor = HardwareMonitor(
            logger_id=logger_id,
            error=deque[HardwareError](
                [
                    HardwareError(
                        timestamp="2024/11/21 16:30:00",
                        level="CRITICAL",
                        message="critical error",
                    )
                ],
                maxlen=_MAX_MESSAGE_COUNT,
            ),
        )
        data_manager._data_sets[logger_id] = DataSet(hwmonitor=hwmonitor)
        data_manager._handle_hwmonitor_error(topic=topic, message_dict=message_dict)

        result = data_manager._data_sets.get(logger_id)

        assert result is None

    ## VC判定データ関連のテスト
    ## Tests related to VC Data
    def test__preprocess_vc_data_handler(self, reset_data_manager):
        logger_id = "0"
        model = "A352"
        serial = "0001"
        data_manager = DataManager.get_instance()
        vc_data_key = data_manager._create_vc_data_key(
            logger_id=logger_id, model=model, serial=serial
        )

        topic = f"logger/{logger_id}/sensor/{model}/{serial}/vc"

        result_key, result_vc_data = data_manager._preprocess_vc_data_handler(
            vc_data_topic=topic
        )

        assert vc_data_key == result_key
        assert result_vc_data is not None

    ### レベル関連のテスト
    ### Tests related to Level
    def test__handle_vc_data_level(self, reset_data_manager):
        logger_id = "0"
        model = "A352"
        serial = "0001"
        expect_vc_level = "OA"

        topic = f"logger/{logger_id}/sensor/{model}/{serial}/vc"
        message_dict = {"timestamp": "2020/01/01 00:00:00", "level": expect_vc_level}

        data_manager = DataManager.get_instance()
        data_manager._handle_vc_data_level(topic=topic, message_dict=message_dict)

        vc_data_key = data_manager._create_vc_data_key(
            logger_id=logger_id, model=model, serial=serial
        )
        vc_data = data_manager._vc_data[vc_data_key]

        assert expect_vc_level == vc_data.vc_level.level  # type: ignore

    def test__handle_vc_data_level_none(self, reset_data_manager):
        logger_id = "0"
        model = "A352"
        serial = "0001"

        topic = f"logger/{logger_id}/sensor/{model}/{serial}/vc"
        message_dict: dict = {}

        vc_level = VCLevel(timestamp="2000/01/01 00:00:00", level="A")
        fft_data = FFTData(timestamp="2000/01/01 00:00:00", value=get_fft_data_sample())

        data_manager = DataManager.get_instance()
        vc_data_key = data_manager._create_vc_data_key(
            logger_id=logger_id, model=model, serial=serial
        )
        data_manager._vc_data[vc_data_key] = VCData(
            vc_level=vc_level, fft_data=fft_data
        )
        data_manager._handle_vc_data_level(topic=topic, message_dict=message_dict)

        vc_data = data_manager._vc_data[vc_data_key]

        assert vc_data.vc_level is None

    def test__handle_vc_data_level_none_remove_vc_data_frame(self, reset_data_manager):
        logger_id = "0"
        model = "A352"
        serial = "0001"

        topic = f"logger/{logger_id}/sensor/{model}/{serial}/vc"
        message_dict: dict = {}

        vc_level = VCLevel(timestamp="2000/01/01 00:00:00", level="A")

        data_manager = DataManager.get_instance()
        vc_data_key = data_manager._create_vc_data_key(
            logger_id=logger_id, model=model, serial=serial
        )
        data_manager._vc_data[vc_data_key] = VCData(vc_level=vc_level)
        data_manager._handle_vc_data_level(topic=topic, message_dict=message_dict)

        result = data_manager._vc_data.get(vc_data_key)

        assert result is None

    ### fft関連のテスト
    ### Tests related to FFT
    def test__handle_vc_data_fft(self, reset_data_manager):
        logger_id = "0"
        model = "A352"
        serial = "0001"
        expect_fft = get_fft_data_sample()

        topic = f"logger/{logger_id}/sensor/{model}/{serial}/fft"
        message_dict = {"timestamp": "2020/01/01 00:00:00", "value": expect_fft}

        data_manager = DataManager.get_instance()
        data_manager._handle_vc_data_fft(topic=topic, message_dict=message_dict)

        vc_data_key = data_manager._create_vc_data_key(
            logger_id=logger_id, model=model, serial=serial
        )
        vc_data = data_manager._vc_data[vc_data_key]

        assert expect_fft == vc_data.fft_data.value  # type: ignore

    def test__handle_vc_data_fft_none(self, reset_data_manager):
        logger_id = "0"
        model = "A352"
        serial = "0001"

        topic = f"logger/{logger_id}/sensor/{model}/{serial}/fft"
        message_dict: dict = {}

        vc_level = VCLevel(timestamp="2000/01/01 00:00:00", level="A")
        fft_data = FFTData(timestamp="2000/01/01 00:00:00", value=get_fft_data_sample())

        data_manager = DataManager.get_instance()
        vc_data_key = data_manager._create_vc_data_key(
            logger_id=logger_id, model=model, serial=serial
        )
        data_manager._vc_data[vc_data_key] = VCData(
            vc_level=vc_level, fft_data=fft_data
        )
        data_manager._handle_vc_data_fft(topic=topic, message_dict=message_dict)

        vc_data = data_manager._vc_data[vc_data_key]

        assert vc_data.fft_data is None

    def test__handle_vc_data_fft_none_remove_vc_data_frame(self, reset_data_manager):
        logger_id = "0"
        model = "A352"
        serial = "0001"

        topic = f"logger/{logger_id}/sensor/{model}/{serial}/fft"
        message_dict: dict = {}

        fft_data = FFTData(timestamp="2000/01/01 00:00:00", value=get_fft_data_sample())

        data_manager = DataManager.get_instance()
        vc_data_key = data_manager._create_vc_data_key(
            logger_id=logger_id, model=model, serial=serial
        )
        data_manager._vc_data[vc_data_key] = VCData(fft_data=fft_data)
        data_manager._handle_vc_data_fft(topic=topic, message_dict=message_dict)

        result = data_manager._vc_data.get(vc_data_key)

        assert result is None

    # クエリ系
    # Queries
    def test_get_data_sets(self, reset_data_manager):

        logger0_id = "0"
        logger1_id = "1"

        expect_logger0 = Logger(
            logger_id=logger0_id,
            status=LoggerStatus(timestamp="2024/12/12 12:12:12", status="START"),
            error=deque(
                [
                    LoggerError(
                        timestamp="2024/12/12 12:12:12",
                        level="CRITICAL",
                        message="hoge",
                    )
                ],
                maxlen=_MAX_MESSAGE_COUNT,
            ),
        )

        expect_sensor0 = Sensor(
            logger_id=logger0_id,
            model="A352",
            serial="0000",
            status=SensorStatus(timestamp="2024/12/01 00:00:00", status="OK"),
        )
        expect_sensor1 = Sensor(
            logger_id=logger1_id,
            model="A352",
            serial="0001",
            status=SensorStatus(timestamp="2024/12/01 10:00:00", status="NG"),
        )
        data_manager = DataManager.get_instance()
        data_manager._data_sets[logger0_id] = DataSet(
            logger=expect_logger0, sensors=[expect_sensor0]
        )
        data_manager._data_sets[logger1_id] = DataSet(sensors=[expect_sensor1])

        actual_data_set = data_manager.get_data_sets()

        assert 2 == len(actual_data_set)

    def test_get_vc_data(self, reset_data_manager):
        logger_id = "0"
        model = "A352"
        serial = "0001"
        expect_level: VCLevelType = "A"
        expect_fft = get_fft_data_sample()

        data_manager = DataManager.get_instance()
        vc_data_key = data_manager._create_vc_data_key(
            logger_id=logger_id, model=model, serial=serial
        )
        data_manager._vc_data[vc_data_key] = VCData(
            vc_level=VCLevel(timestamp="2000/01/01 00:00:00", level=expect_level),
            fft_data=FFTData(timestamp="2000/01/01 00:00:00", value=expect_fft),
        )
        actual = data_manager.get_vc_data(
            logger_id=logger_id, model=model, serial=serial
        )

        assert expect_level == actual.vc_level.level  # type: ignore
        assert expect_fft == actual.fft_data.value  # type: ignore

    def test_get_vc_data_none(self, reset_data_manager):
        logger_id = "0"
        model = "A352"
        serial = "0001"

        data_manager = DataManager.get_instance()

        actual = data_manager.get_vc_data(
            logger_id=logger_id, model=model, serial=serial
        )

        assert actual is None


class TestMultiThread:
    """
    マルチスレッドの動作確認

    Check the operation of multithreading
    """
    _thread_lock = threading.Lock()
    data: list[int] = []
    lock = threading.Lock()

    def _reset_data(self):
        self.data = []

    def _lock_decorator(func: Callable):  # type: ignore
        def _wrapper(self: Self, *args, **kwargs):
            with self.lock:
                result = func(self, *args, **kwargs)
            return result

        return _wrapper

    @_lock_decorator
    def get_data(self, sleep_time: int, get_result: list[int]):
        time.sleep(sleep_time)
        # threadからデータを取り出せないため引数に渡したlistにつめる
        # Because data cannot be retrieved from the thread, it is put into the list passed as an argument
        get_result.extend(
            copy.deepcopy(self.data)
        )
        return copy.deepcopy(self.data)

    @_lock_decorator
    def register_data(self, sleep_time: int, data: int):
        time.sleep(sleep_time)
        self.data.append(data)

    def test_simulate_get_loggers_in_register_logger(self):
        """
        ロガーの登録処理中にロガー取得処理が実行される場合をシミュレーション

        前提: dataは空

        インプット: 登録スレッド、取得スレッドを作成、実行し、登録処理にスリープを入れることで登録処理実行中に取得処理を行う

        アウトプット: スレッドがロックされているため、取得処理の結果が登録処理後のものになる

        Simulate the case where the logger retrieval process is executed during the logger registration process

        Prerequisite: data is empty

        Input: Create and execute registration and retrieval threads, and perform the retrieval process during the registration process by inserting a sleep in the registration process

        Output: Since the thread is locked, the result of the retrieval process will be after the registration process
        """
        self._reset_data()

        # thread内での戻り値を入れる
        # Store the return value in the thread
        get_result: list[int] = []

        data_for_register = 1

        # 登録処理に3秒のスリープ
        # 3 seconds sleep in the registration process
        register_thread = threading.Thread(
            target=self.register_data,
            name="register_data_thread",
            args=[3, data_for_register],
            daemon=True,
        )
        get_thread = threading.Thread(
            target=self.get_data,
            name="get_data_thread",
            args=[0, get_result],
            daemon=True,
        )

        register_thread.start()
        # 必ず登録処理が先に始まるようにする
        # Ensure that the registration process starts first
        time.sleep(1)
        get_thread.start()

        register_thread.join()
        get_thread.join()

        assert data_for_register == get_result[0]

    def test_simulate_register_logger_in_get_loggers(self):
        """
        ロガーの取得処理中にロガー登録処理が実行される場合をシミュレーション

        前提: dataは空

        インプット:
            - 登録スレッド、取得スレッドを作成、実行し、取得処理にスリープを入れることで取得処理実行中に登録処理を行う

        アウトプット:
            - スレッドがロックされているため、取得処理の結果が登録処理前のものになる
            - 処理後にdataを見ると登録処理後の値が入っている

        Simulate the case where the logger registration process is executed during the logger retrieval process

        Prerequisite: data is empty

        Input:
            - Create and execute registration and retrieval threads, and perform the registration process during the retrieval process by inserting a sleep in the retrieval process

        Output:
            - Since the thread is locked, the result of the retrieval process will be before the registration process
            - After the process, the data contains the value after the registration process
        """
        self._reset_data()

        # thread内での戻り値を入れる
        # Store the return value in the thread
        get_result: list[int] = []

        data_for_register = 1

        # 取得処理に3秒のスリープ
        # 3 seconds sleep in the retrieval process
        get_thread = threading.Thread(
            target=self.get_data,
            name="get_data_thread",
            args=[3, get_result],
            daemon=True,
        )
        register_thread = threading.Thread(
            target=self.register_data,
            name="register_data_thread",
            args=[0, data_for_register],
            daemon=True,
        )

        get_thread.start()
        # 必ず取得処理が先に始まるようにする
        # Ensure that the retrieval process starts first
        time.sleep(1)
        register_thread.start()

        get_thread.join()
        register_thread.join()

        assert 0 == len(get_result)
        assert data_for_register == self.data[0]
