# Copyright 2025, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

import json
import time
from collections import deque

import paho.mqtt.publish as publish
import pytest

from raspi_web.common import config
from raspi_web.data.hwmonitor import (
    HardwareData,
    HardwareError,
    HardwareMonitor,
    HardwareStatus,
)
from raspi_web.data.logger import Logger, LoggerError, LoggerStatus
from raspi_web.data.sensor import Sensor, SensorError, SensorMessage, SensorStatus
from raspi_web.data.vc_data import FFTData, VCData, VCLevel
from raspi_web.infra.data_manager import DataManager, DataSet
from raspi_web.infra.subscriber import MQTTSubscriber
from raspi_web.utils.constants import _MAX_MESSAGE_COUNT
from tests.testutils.data_sample import get_fft_data_sample


@pytest.fixture()
def reset_data_manager():
    # setup
    data_manager = DataManager.get_instance()
    data_manager._data_sets = {}
    data_manager._vc_data = {}

    yield
    # teardown
    data_manager._data_sets = {}
    data_manager._vc_data = {}


class TestMQTTSubscriber:
    HOST = config.get_str("MQTT_BROKER_HOST")
    PORT = config.get_int("MQTT_BROKER_PORT")

    def _publish_message(self, topic, payload):

        publish.single(
            topic=topic,
            payload=payload,
            hostname=self.HOST,
            port=self.PORT,
            retain=True,
        )

    def _delete_topic_message(self, topic):

        publish.single(
            topic=topic, payload=None, hostname=self.HOST, port=self.PORT, retain=True
        )

    # TODO: DataManagerをMock化してテストした方が良い
    # ロガー関連のテスト
    # TODO: It is better to mock DataManager for testing
    # Tests related to Logger
    def test_subscribe_logger_status(self, reset_data_manager):

        expect_logger_id = "0"
        expect_logger_status = "START"

        topic = f"logger/{expect_logger_id}"
        payload = {"timestamp": "2024/11/21 16:30:00", "status": expect_logger_status}

        self._publish_message(topic=topic, payload=None)

        mqtt_subscriber = MQTTSubscriber()
        mqtt_subscriber.start()

        self._publish_message(topic=topic, payload=json.dumps(payload))

        time.sleep(1)

        mqtt_subscriber.end()

        self._publish_message(topic=topic, payload=None)

        data_manager = DataManager.get_instance()
        data_set = data_manager._data_sets.get(expect_logger_id)

        assert expect_logger_id == data_set.logger.logger_id  # type: ignore
        assert expect_logger_status == data_set.logger.status.status  # type: ignore

    def test_subscribe_logger_status_none_payload(self, reset_data_manager):
        expect_logger_id = "0"

        data_manager = DataManager.get_instance()
        data_manager._data_sets[expect_logger_id] = DataSet(
            logger=Logger(
                logger_id=expect_logger_id,
                status=LoggerStatus(timestamp="2024/11/21 16:30:00", status="START"),
                error=deque[LoggerError](
                    [
                        LoggerError(
                            timestamp="2024/11/21 16:30:00",
                            level="CRITICAL",
                            message="critical error",
                        )
                    ],
                    maxlen=_MAX_MESSAGE_COUNT,
                ),
            )
        )

        topic = f"logger/{expect_logger_id}"

        self._publish_message(topic=topic, payload=None)

        mqtt_subscriber = MQTTSubscriber()
        mqtt_subscriber.start()

        self._publish_message(topic=topic, payload=None)

        time.sleep(1)

        mqtt_subscriber.end()

        self._publish_message(topic=topic, payload=None)

        data_set = data_manager._data_sets[expect_logger_id]

        # 該当のロガーには他のロガーエラー等が入っているため、loggerはNoneではない
        # Since the corresponding logger contains other logger errors, the logger is not None
        assert expect_logger_id == data_set.logger.logger_id  # type: ignore
        assert data_set.logger.status is None  # type: ignore

    def test_subscribe_logger_error(self, reset_data_manager):

        expect_logger_id = "0"
        expect_logger_error: LoggerError = LoggerError(
            timestamp="2024/11/21 16:30:00",
            level="CRITICAL",
            message="critical error",
        )

        topic = f"logger/{expect_logger_id}/error"
        payload = {
            "timestamp": expect_logger_error.timestamp,
            "level": expect_logger_error.level,
            "message": expect_logger_error.message,
        }

        self._publish_message(topic=topic, payload=None)

        mqtt_subscriber = MQTTSubscriber()
        mqtt_subscriber.start()

        self._publish_message(topic=topic, payload=json.dumps(payload))

        time.sleep(1)

        mqtt_subscriber.end()

        self._publish_message(topic=topic, payload=None)

        data_manager = DataManager.get_instance()
        data_set = data_manager._data_sets[expect_logger_id]

        assert expect_logger_id == data_set.logger.logger_id  # type: ignore
        assert expect_logger_error == data_set.logger.error[0]  # type: ignore

    def test_subscribe_logger_error_none_payload(self, reset_data_manager):
        expect_logger_id = "0"

        data_manager = DataManager.get_instance()
        data_manager._data_sets[expect_logger_id] = DataSet(
            logger=Logger(
                logger_id=expect_logger_id,
                status=LoggerStatus(timestamp="2024/11/21 16:30:00", status="START"),
                error=deque[LoggerError](
                    [
                        LoggerError(
                            timestamp="2024/11/21 16:30:00",
                            level="CRITICAL",
                            message="critical error",
                        )
                    ],
                    maxlen=_MAX_MESSAGE_COUNT,
                ),
            )
        )

        topic = f"logger/{expect_logger_id}/error"

        self._publish_message(topic=topic, payload=None)

        mqtt_subscriber = MQTTSubscriber()
        mqtt_subscriber.start()

        self._publish_message(topic=topic, payload=None)

        time.sleep(1)

        mqtt_subscriber.end()

        self._publish_message(topic=topic, payload=None)

        data_set = data_manager._data_sets[expect_logger_id]

        # 該当のロガーには他のロガーステータス等が入っているため、loggerはNoneではない
        # Since the corresponding logger contains other logger statuses, the logger is not None
        assert expect_logger_id == data_set.logger.logger_id  # type: ignore
        assert 0 == len(data_set.logger.error)  # type: ignore

    # センサー関連のテスト
    # Tests related to Sensor
    def test_subscribe_sensor_status(self, reset_data_manager):

        expect_logger_id = "0"
        expect_model = "A352"
        expect_serial_number = "0001"
        expect_sensor_status = "OK"

        topic = (
            f"logger/{expect_logger_id}/sensor/{expect_model}/{expect_serial_number}"
        )
        payload = {"timestamp": "2024/11/21 16:30:00", "status": expect_sensor_status}

        self._publish_message(topic=topic, payload=None)

        mqtt_subscriber = MQTTSubscriber()
        mqtt_subscriber.start()

        self._publish_message(topic=topic, payload=json.dumps(payload))

        time.sleep(1)

        mqtt_subscriber.end()

        self._publish_message(topic=topic, payload=None)

        data_manager = DataManager.get_instance()
        result_sensor = data_manager._data_sets[expect_logger_id].sensors[0]

        assert expect_logger_id == result_sensor.logger_id
        assert expect_sensor_status == result_sensor.status.status  # type: ignore

    def test_subscribe_sensor_status_none_payload(self, reset_data_manager):
        expect_logger_id = "0"
        model = "A352"
        serial = "0000"

        data_manager = DataManager.get_instance()
        data_manager._data_sets[expect_logger_id] = DataSet(
            sensors=[
                Sensor(
                    logger_id=expect_logger_id,
                    model=model,
                    serial=serial,
                    status=SensorStatus(timestamp="2024/11/21 16:30:00", status="OK"),
                    error=deque[SensorError](
                        [
                            SensorError(
                                timestamp="2024/11/21 17:00:00",
                                level="CRITICAL",
                                message="critical error",
                            )
                        ],
                        maxlen=_MAX_MESSAGE_COUNT,
                    ),
                )
            ]
        )

        topic = f"logger/{expect_logger_id}/sensor/{model}/{serial}"

        self._publish_message(topic=topic, payload=None)

        mqtt_subscriber = MQTTSubscriber()
        mqtt_subscriber.start()

        self._publish_message(topic=topic, payload=None)

        time.sleep(1)

        mqtt_subscriber.end()

        self._publish_message(topic=topic, payload=None)

        result_sensor = data_manager._data_sets[expect_logger_id].sensors[0]

        # 該当のセンサーには他のセンサーのエラー等が入っているため、loggerはNoneではない
        # Since the corresponding sensor contains other sensor errors, the logger is not None
        assert expect_logger_id == result_sensor.logger_id
        assert result_sensor.status is None

    def test_subscribe_sensor_error(self, reset_data_manager):

        expect_logger_id = "0"
        expect_model = "A352"
        expect_serial_number = "0000"
        expect_sensor_error: SensorError = SensorError(
            timestamp="2024/11/21 16:30:00",
            level="CRITICAL",
            message="critical error",
        )

        topic = f"logger/{expect_logger_id}/sensor/{expect_model}/{expect_serial_number}/error"
        payload = {
            "timestamp": expect_sensor_error.timestamp,
            "level": expect_sensor_error.level,
            "message": expect_sensor_error.message,
        }

        self._publish_message(topic=topic, payload=None)

        mqtt_subscriber = MQTTSubscriber()
        mqtt_subscriber.start()

        self._publish_message(topic=topic, payload=json.dumps(payload))

        time.sleep(1)

        mqtt_subscriber.end()

        self._publish_message(topic=topic, payload=None)

        data_manager = DataManager.get_instance()
        result_sensor = data_manager._data_sets[expect_logger_id].sensors[0]

        assert expect_logger_id == result_sensor.logger_id
        assert expect_sensor_error == result_sensor.error[0]  # type: ignore

    def test_subscribe_sensor_error_none_payload(self, reset_data_manager):
        expect_logger_id = "0"
        expect_model = "A352"
        expect_serial_number = "0000"

        data_manager = DataManager.get_instance()
        data_manager._data_sets[expect_logger_id] = DataSet(
            sensors=[
                Sensor(
                    logger_id=expect_logger_id,
                    model=expect_model,
                    serial=expect_serial_number,
                    status=SensorStatus(timestamp="2024/11/21 16:30:00", status="OK"),
                    error=deque[SensorError](
                        [
                            SensorError(
                                timestamp="2024/11/21 16:30:00",
                                level="CRITICAL",
                                message="critical error",
                            )
                        ],
                        maxlen=_MAX_MESSAGE_COUNT,
                    ),
                )
            ]
        )

        topic = f"logger/{expect_logger_id}/sensor/{expect_model}/{expect_serial_number}/error"

        self._publish_message(topic=topic, payload=None)

        mqtt_subscriber = MQTTSubscriber()
        mqtt_subscriber.start()

        self._publish_message(topic=topic, payload=None)

        time.sleep(1)

        mqtt_subscriber.end()

        self._publish_message(topic=topic, payload=None)

        result_sensor = data_manager._data_sets[expect_logger_id].sensors[0]

        assert expect_logger_id == result_sensor.logger_id
        assert 0 == len(result_sensor.error)

    def test_subscribe_sensor_loss(self, reset_data_manager):

        expect_logger_id = "0"
        expect_model = "A352"
        expect_serial_number = "0000"
        expect_sensor_loss: SensorMessage = SensorMessage(
            timestamp="2024/11/21 16:30:00",
            message="critical loss",
        )

        topic = f"logger/{expect_logger_id}/sensor/{expect_model}/{expect_serial_number}/loss"
        payload = {
            "timestamp": expect_sensor_loss.timestamp,
            "message": expect_sensor_loss.message,
        }

        self._publish_message(topic=topic, payload=None)

        mqtt_subscriber = MQTTSubscriber()
        mqtt_subscriber.start()

        self._publish_message(topic=topic, payload=json.dumps(payload))

        time.sleep(1)

        mqtt_subscriber.end()

        self._publish_message(topic=topic, payload=None)

        data_manager = DataManager.get_instance()
        result_sensor = data_manager._data_sets[expect_logger_id].sensors[0]

        assert expect_logger_id == result_sensor.logger_id
        assert expect_sensor_loss == result_sensor.loss[0]  # type: ignore

    def test_subscribe_sensor_loss_none_payload(self, reset_data_manager):
        expect_logger_id = "0"
        expect_model = "A352"
        expect_serial_number = "0000"

        data_manager = DataManager.get_instance()
        data_manager._data_sets[expect_logger_id] = DataSet(
            sensors=[
                Sensor(
                    logger_id=expect_logger_id,
                    model=expect_model,
                    serial=expect_serial_number,
                    status=SensorStatus(timestamp="2024/11/21 16:30:00", status="OK"),
                    loss=deque[SensorMessage](
                        [
                            SensorMessage(
                                timestamp="2024/11/21 16:30:00",
                                message="loss",
                            )
                        ],
                        maxlen=_MAX_MESSAGE_COUNT,
                    ),
                )
            ]
        )

        topic = f"logger/{expect_logger_id}/sensor/{expect_model}/{expect_serial_number}/loss"

        self._publish_message(topic=topic, payload=None)

        mqtt_subscriber = MQTTSubscriber()
        mqtt_subscriber.start()

        self._publish_message(topic=topic, payload=None)

        time.sleep(1)

        mqtt_subscriber.end()

        self._publish_message(topic=topic, payload=None)

        result_sensor = data_manager._data_sets[expect_logger_id].sensors[0]

        assert expect_logger_id == result_sensor.logger_id
        assert 0 == len(result_sensor.loss)

    def test_subscribe_sensor_abnormal(self, reset_data_manager):

        expect_logger_id = "0"
        expect_model = "A352"
        expect_serial_number = "0000"
        expect_sensor_abnormal: SensorMessage = SensorMessage(
            timestamp="2024/11/21 16:30:00",
            message="critical abnormal",
        )

        topic = f"logger/{expect_logger_id}/sensor/{expect_model}/{expect_serial_number}/abnormal"
        payload = {
            "timestamp": expect_sensor_abnormal.timestamp,
            "message": expect_sensor_abnormal.message,
        }

        self._publish_message(topic=topic, payload=None)

        mqtt_subscriber = MQTTSubscriber()
        mqtt_subscriber.start()

        self._publish_message(topic=topic, payload=json.dumps(payload))

        time.sleep(1)

        mqtt_subscriber.end()

        self._publish_message(topic=topic, payload=None)

        data_manager = DataManager.get_instance()
        result_sensor = data_manager._data_sets[expect_logger_id].sensors[0]

        assert expect_logger_id == result_sensor.logger_id
        assert expect_sensor_abnormal == result_sensor.abnormal[0]  # type: ignore

    def test_subscribe_sensor_abnormal_none_payload(self, reset_data_manager):
        expect_logger_id = "0"
        expect_model = "A352"
        expect_serial_number = "0000"

        data_manager = DataManager.get_instance()
        data_manager._data_sets[expect_logger_id] = DataSet(
            sensors=[
                Sensor(
                    logger_id=expect_logger_id,
                    model=expect_model,
                    serial=expect_serial_number,
                    status=SensorStatus(timestamp="2024/11/21 16:30:00", status="OK"),
                    abnormal=deque[SensorMessage](
                        [
                            SensorMessage(
                                timestamp="2024/11/21 16:30:00",
                                message="critical abnormal",
                            )
                        ],
                        maxlen=_MAX_MESSAGE_COUNT,
                    ),
                )
            ]
        )

        topic = f"logger/{expect_logger_id}/sensor/{expect_model}/{expect_serial_number}/abnormal"

        self._publish_message(topic=topic, payload=None)

        mqtt_subscriber = MQTTSubscriber()
        mqtt_subscriber.start()

        self._publish_message(topic=topic, payload=None)

        time.sleep(1)

        mqtt_subscriber.end()

        self._publish_message(topic=topic, payload=None)

        result_sensor = data_manager._data_sets[expect_logger_id].sensors[0]

        assert expect_logger_id == result_sensor.logger_id
        assert 0 == len(result_sensor.abnormal)

    # ハードウェアモニターに関するテスト
    # Tests related to Hardware Monitor
    def test_subscribe_hwmonitor_status(self, reset_data_manager):

        expect_logger_id = "0"
        expect_hwmonitor_status = "START"

        topic = f"hwmonitor/{expect_logger_id}"
        payload = {
            "timestamp": "2024/11/21 16:30:00",
            "status": expect_hwmonitor_status,
        }

        self._publish_message(topic=topic, payload=None)

        mqtt_subscriber = MQTTSubscriber()
        mqtt_subscriber.start()

        self._publish_message(topic=topic, payload=json.dumps(payload))

        time.sleep(1)

        mqtt_subscriber.end()

        self._publish_message(topic=topic, payload=None)

        data_manager = DataManager.get_instance()
        data_set = data_manager._data_sets.get(expect_logger_id)

        assert expect_logger_id == data_set.hwmonitor.logger_id  # type: ignore
        assert expect_hwmonitor_status == data_set.hwmonitor.status.status  # type: ignore

    def test_subscribe_hwmonitor_status_none_payload(self, reset_data_manager):
        expect_logger_id = "0"

        data_manager = DataManager.get_instance()
        data_manager._data_sets[expect_logger_id] = DataSet(
            hwmonitor=HardwareMonitor(
                logger_id=expect_logger_id,
                status=HardwareStatus(timestamp="2024/11/21 16:30:00", status="START"),
                error=deque[HardwareError](
                    [
                        HardwareError(
                            timestamp="2024/11/21 16:30:00",
                            level="CRITICAL",
                            message="critical error",
                        )
                    ],
                    maxlen=_MAX_MESSAGE_COUNT,
                ),
            ),
        )

        topic = f"hwmonitor/{expect_logger_id}"

        self._publish_message(topic=topic, payload=None)

        mqtt_subscriber = MQTTSubscriber()
        mqtt_subscriber.start()

        self._publish_message(topic=topic, payload=None)

        time.sleep(1)

        mqtt_subscriber.end()

        self._publish_message(topic=topic, payload=None)

        data_set = data_manager._data_sets[expect_logger_id]

        # 該当のハードウェアモニターには他のハードウェアモニターエラー等が入っているため、hwmonitorはNoneではない
        # Since the corresponding hardware monitor contains other hardware monitor errors, the hwmonitor is not None
        assert expect_logger_id == data_set.hwmonitor.logger_id  # type: ignore
        assert data_set.hwmonitor.status is None  # type: ignore

    def test_subscribe_hwmonitor_data(self, reset_data_manager):

        expect_logger_id = "0"
        expect_cpu_temperature = 10.1
        expect_cpu_usage = 10.2
        expect_memory_usage = 10.3
        expect_disk_usage = 10.4

        topic = f"hwmonitor/{expect_logger_id}/data"
        payload = {
            "timestamp": "2024/12/12 12:12:12",
            "cpu_temperature": expect_cpu_temperature,
            "cpu_usage": expect_cpu_usage,
            "memory_usage": expect_memory_usage,
            "disk_usage": expect_disk_usage,
        }

        self._publish_message(topic=topic, payload=None)

        mqtt_subscriber = MQTTSubscriber()
        mqtt_subscriber.start()

        self._publish_message(topic=topic, payload=json.dumps(payload))

        time.sleep(1)

        mqtt_subscriber.end()

        self._publish_message(topic=topic, payload=None)

        data_manager = DataManager.get_instance()
        data_set = data_manager._data_sets.get(expect_logger_id)

        assert expect_logger_id == data_set.hwmonitor.logger_id  # type: ignore
        assert expect_cpu_temperature == data_set.hwmonitor.data.cpu_temperature  # type: ignore
        assert expect_cpu_usage == data_set.hwmonitor.data.cpu_usage  # type: ignore
        assert expect_memory_usage == data_set.hwmonitor.data.memory_usage  # type: ignore
        assert expect_disk_usage == data_set.hwmonitor.data.disk_usage  # type: ignore

    def test_subscribe_hwmonitor_data_none_payload(self, reset_data_manager):
        expect_logger_id = "0"

        data_manager = DataManager.get_instance()
        data_manager._data_sets[expect_logger_id] = DataSet(
            hwmonitor=HardwareMonitor(
                logger_id=expect_logger_id,
                status=HardwareStatus(timestamp="2024/11/21 16:30:00", status="START"),
                data=HardwareData(
                    timestamp="2024/12/12 12:12:12",
                    cpu_temperature=10,
                    cpu_usage=10,
                    memory_usage=10,
                    disk_usage=10,
                ),
            ),
        )

        topic = f"hwmonitor/{expect_logger_id}/data"

        self._publish_message(topic=topic, payload=None)

        mqtt_subscriber = MQTTSubscriber()
        mqtt_subscriber.start()

        self._publish_message(topic=topic, payload=None)

        time.sleep(1)

        mqtt_subscriber.end()

        self._publish_message(topic=topic, payload=None)

        data_set = data_manager._data_sets[expect_logger_id]

        # 該当のハードウェアモニターには他のハードウェアモニターステータス等が入っているため、hwmonitorはNoneではない
        # Since the corresponding hardware monitor contains other hardware monitor statuses, the hwmonitor is not None
        assert expect_logger_id == data_set.hwmonitor.logger_id  # type: ignore
        assert data_set.hwmonitor.data is None  # type: ignore

    def test_subscribe_hwmonitor_error(self, reset_data_manager):

        expect_logger_id = "0"
        expect_hwmonitor_error: HardwareError = HardwareError(
            timestamp="2024/11/21 16:30:00",
            level="CRITICAL",
            message="critical error",
        )

        topic = f"hwmonitor/{expect_logger_id}/error"
        payload = {
            "timestamp": expect_hwmonitor_error.timestamp,
            "level": expect_hwmonitor_error.level,
            "message": expect_hwmonitor_error.message,
        }

        self._publish_message(topic=topic, payload=None)

        mqtt_subscriber = MQTTSubscriber()
        mqtt_subscriber.start()

        self._publish_message(topic=topic, payload=json.dumps(payload))

        time.sleep(1)

        mqtt_subscriber.end()

        self._publish_message(topic=topic, payload=None)

        data_manager = DataManager.get_instance()
        data_set = data_manager._data_sets[expect_logger_id]

        assert expect_logger_id == data_set.hwmonitor.logger_id  # type: ignore
        assert expect_hwmonitor_error == data_set.hwmonitor.error[0]  # type: ignore

    def test_subscribe_hwmonitor_error_none_payload(self, reset_data_manager):
        expect_logger_id = "0"

        data_manager = DataManager.get_instance()
        data_manager._data_sets[expect_logger_id] = DataSet(
            hwmonitor=HardwareMonitor(
                logger_id=expect_logger_id,
                status=HardwareStatus(timestamp="2024/11/21 16:30:00", status="START"),
                error=deque[HardwareError](
                    [
                        HardwareError(
                            timestamp="2024/11/21 16:30:00",
                            level="CRITICAL",
                            message="critical error",
                        )
                    ],
                    maxlen=_MAX_MESSAGE_COUNT,
                ),
            )
        )

        topic = f"hwmonitor/{expect_logger_id}/error"

        self._publish_message(topic=topic, payload=None)

        mqtt_subscriber = MQTTSubscriber()
        mqtt_subscriber.start()

        self._publish_message(topic=topic, payload=None)

        time.sleep(1)

        mqtt_subscriber.end()

        self._publish_message(topic=topic, payload=None)

        data_set = data_manager._data_sets[expect_logger_id]

        # 該当のハードウェアモニターには他のステータス等が入っているため、hwmonitorはNoneではない
        # Since the corresponding hardware monitor contains other statuses, the hwmonitor is not None
        assert expect_logger_id == data_set.hwmonitor.logger_id  # type: ignore
        assert 0 == len(data_set.hwmonitor.error)  # type: ignore

    # VCデータ関連のテスト
    ## Tests related to VC Data
    def test_subscribe_vc_data_level(self, reset_data_manager):

        logger_id = "0"
        model = "A352"
        serial_number = "0001"
        expect_vc_level = "G"

        topic = f"logger/{logger_id}/sensor/{model}/{serial_number}/vc"
        payload = {"timestamp": "2024/11/21 16:30:00", "level": expect_vc_level}

        self._publish_message(topic=topic, payload=None)

        mqtt_subscriber = MQTTSubscriber()
        mqtt_subscriber.start()

        self._publish_message(topic=topic, payload=json.dumps(payload))

        time.sleep(1)

        mqtt_subscriber.end()

        self._publish_message(topic=topic, payload=None)

        data_manager = DataManager.get_instance()
        expect_vc_data_key = data_manager._create_vc_data_key(
            logger_id=logger_id, model=model, serial=serial_number
        )
        result_vc_data = data_manager._vc_data[expect_vc_data_key]

        assert expect_vc_level == result_vc_data.vc_level.level  # type: ignore

    def test_subscribe_vc_data_level_none_payload(self, reset_data_manager):
        logger_id = "0"
        model = "A352"
        serial = "0000"

        data_manager = DataManager.get_instance()
        vc_data_key = data_manager._create_vc_data_key(
            logger_id=logger_id, model=model, serial=serial
        )

        data_manager._vc_data[vc_data_key] = VCData(
            vc_level=VCLevel(timestamp="2024/11/21 16:30:00", level="OA"),
            fft_data=FFTData(
                timestamp="2024/11/21 16:30:00",
                value=get_fft_data_sample(),
            ),
        )

        topic = f"logger/{logger_id}/sensor/{model}/{serial}/vc"

        self._publish_message(topic=topic, payload=None)

        mqtt_subscriber = MQTTSubscriber()
        mqtt_subscriber.start()

        self._publish_message(topic=topic, payload=None)

        time.sleep(1)

        mqtt_subscriber.end()

        self._publish_message(topic=topic, payload=None)

        result_vc_data = data_manager._vc_data[vc_data_key]

        # 該当のvcデータには他のfft等が入っているため、vc_dataはNoneではない
        # Since the corresponding VC data contains other FFT data, the vc_data is not None
        assert result_vc_data.vc_level is None

    def test_subscribe_vc_data_fft_data(self, reset_data_manager):

        logger_id = "0"
        model = "A352"
        serial_number = "0001"
        expect_fft_data = get_fft_data_sample()

        topic = f"logger/{logger_id}/sensor/{model}/{serial_number}/fft"
        payload = {"timestamp": "2024/11/21 16:30:00", "value": expect_fft_data}

        self._publish_message(topic=topic, payload=None)

        mqtt_subscriber = MQTTSubscriber()
        mqtt_subscriber.start()

        self._publish_message(topic=topic, payload=json.dumps(payload))

        time.sleep(1)

        mqtt_subscriber.end()

        self._publish_message(topic=topic, payload=None)

        data_manager = DataManager.get_instance()
        expect_vc_data_key = data_manager._create_vc_data_key(
            logger_id=logger_id, model=model, serial=serial_number
        )
        result_vc_data = data_manager._vc_data[expect_vc_data_key]

        assert expect_fft_data == result_vc_data.fft_data.value  # type: ignore

    def test_subscribe_vc_data_fft_data_none_payload(self, reset_data_manager):
        logger_id = "0"
        model = "A352"
        serial = "0000"

        data_manager = DataManager.get_instance()
        vc_data_key = data_manager._create_vc_data_key(
            logger_id=logger_id, model=model, serial=serial
        )
        data_manager._vc_data[vc_data_key] = VCData(
            vc_level=VCLevel(timestamp="2024/11/21 16:30:00", level="OA"),
            fft_data=FFTData(
                timestamp="2024/11/21 16:30:00",
                value=get_fft_data_sample(),
            ),
        )

        topic = f"logger/{logger_id}/sensor/{model}/{serial}/fft"

        self._publish_message(topic=topic, payload=None)

        mqtt_subscriber = MQTTSubscriber()
        mqtt_subscriber.start()

        self._publish_message(topic=topic, payload=None)

        time.sleep(1)

        mqtt_subscriber.end()

        self._publish_message(topic=topic, payload=None)

        result_vc_data = data_manager._vc_data[vc_data_key]

        # 該当のvcデータには他のlevel等が入っているため、vc_dataはNoneではない
        # Since the corresponding VC data contains other levels, the vc_data is not None
        assert result_vc_data.fft_data is None
