import datetime as dt
import json
import random
import sys
import time
from typing import Literal

import paho.mqtt.publish as publish
from topic_type import TopicType

from raspi_web.data.error_type import ErrorType
from raspi_web.data.vc_data import VCLevelType
from raspi_web.util import config

HOST = config.get_str("MQTT_BROKER_HOST")
PORT = config.get_int("MQTT_BROKER_PORT")


def _publish_message(topic, payload):

    publish.single(
        topic=topic,
        payload=payload,
        hostname=HOST,
        port=PORT,
        retain=True,
    )


# ロガー
def publish_logger_status(logger_id: str, status: Literal["START", "STOP"]):
    topic = f"logger/{logger_id}"

    timestamp = time.strftime("%Y/%m/%d %H:%M:%S", time.localtime())

    payload = {"timestamp": timestamp, "status": status}

    _publish_message(topic=topic, payload=json.dumps(payload))


def publish_logger_error(logger_id: str, level: ErrorType, message: str):
    topic = f"logger/{logger_id}/error"

    timestamp = time.strftime("%Y/%m/%d %H:%M:%S", time.localtime())

    payload = {"timestamp": timestamp, "level": level, "message": message}

    _publish_message(topic=topic, payload=json.dumps(payload))


# センサー
def publish_sensor_status(
    logger_id: str, model: str, serial: str, status: Literal["OK", "NG"]
):
    topic = f"logger/{logger_id}/sensor/{model}/{serial}"

    timestamp = time.strftime("%Y/%m/%d %H:%M:%S", time.localtime())

    payload = {"timestamp": timestamp, "status": status}

    _publish_message(topic=topic, payload=json.dumps(payload))


def publish_sensor_error(
    logger_id: str, model: str, serial: str, level: ErrorType, message: str
):
    topic = f"logger/{logger_id}/sensor/{model}/{serial}/error"

    timestamp = time.strftime("%Y/%m/%d %H:%M:%S", time.localtime())

    payload = {"timestamp": timestamp, "level": level, "message": message}

    _publish_message(topic=topic, payload=json.dumps(payload))


def publish_sensor_loss(logger_id: str, model: str, serial: str, message: str):
    topic = f"logger/{logger_id}/sensor/{model}/{serial}/loss"

    timestamp = time.strftime("%Y/%m/%d %H:%M:%S", time.localtime())

    payload = {"timestamp": timestamp, "message": message}

    _publish_message(topic=topic, payload=json.dumps(payload))


def publish_sensor_abnormal(logger_id: str, model: str, serial: str, message: str):
    topic = f"logger/{logger_id}/sensor/{model}/{serial}/abnormal"

    timestamp = time.strftime("%Y/%m/%d %H:%M:%S", time.localtime())

    payload = {"timestamp": timestamp, "message": message}

    _publish_message(topic=topic, payload=json.dumps(payload))


# ハードウェアモニター
def publish_hwmonitor_status(logger_id: str, status: Literal["START", "STOP"]):
    topic = f"hwmonitor/{logger_id}"

    timestamp = time.strftime("%Y/%m/%d %H:%M:%S", time.localtime())

    payload = {"timestamp": timestamp, "status": status}

    _publish_message(topic=topic, payload=json.dumps(payload))


def publish_hwmonitor_data(
    logger_id: str,
    cpu_temperature: float,
    cpu_usage: float,
    memory_usage: float,
    disk_usage: float,
):
    topic = f"hwmonitor/{logger_id}/data"

    timestamp = time.strftime("%Y/%m/%d %H:%M:%S", time.localtime())

    payload = {
        "timestamp": timestamp,
        "cpu_temperature": cpu_temperature,
        "cpu_usage": cpu_usage,
        "memory_usage": memory_usage,
        "disk_usage": disk_usage,
    }

    _publish_message(topic=topic, payload=json.dumps(payload))


def publish_hwmonitor_error(logger_id: str, level: ErrorType, message: str):
    topic = f"hwmonitor/{logger_id}/error"

    timestamp = time.strftime("%Y/%m/%d %H:%M:%S", time.localtime())

    payload = {"timestamp": timestamp, "level": level, "message": message}

    _publish_message(topic=topic, payload=json.dumps(payload))


def publish_vc_data_level(logger_id: str, model: str, serial: str, level: VCLevelType):
    topic = f"logger/{logger_id}/sensor/{model}/{serial}/vc"

    timestamp = time.strftime("%Y/%m/%d %H:%M:%S", time.localtime())

    payload = {"timestamp": timestamp, "level": level}

    _publish_message(topic=topic, payload=json.dumps(payload))


def publish_vc_data_fft(logger_id: str, model: str, serial: str, fft: list[float]):
    topic = f"logger/{logger_id}/sensor/{model}/{serial}/fft"

    timestamp = time.strftime("%Y/%m/%d %H:%M:%S", time.localtime())

    payload = {"timestamp": timestamp, "value": fft}

    _publish_message(topic=topic, payload=json.dumps(payload))


def publish_summary_data(logger_id: str, model: str, serial: str):
    topic = f"summary/{logger_id}/{model}/{serial}/data"
    num = 10

    def rnd() -> float:
        return random.random()

    def uni() -> float:
        return random.uniform(1e-6, 1e-1)

    # datetime
    today = dt.datetime.today()
    datetime: list[str] = [
        (today - dt.timedelta(weeks=(num - 1 - i))).strftime("%Y%m%d_%H%M%S")
        for i in range(num)
    ]

    # physical
    v = rnd()
    if v > 0.7:
        physical = "Velocity"
    elif v > 0.3:
        physical = "Acceleration"
    else:
        physical = "Displacement"

    # trend
    _AXIS = ["X", "Y", "Z", "C"]
    trend: dict[str, dict] = {}
    da: dict[str, list[float] | float | None] = {}
    for axis in _AXIS:
        da = {}
        da["value"] = [uni() for _ in range(num)]
        da["baseline"] = uni() if rnd() > 0.5 else None
        da["limit_alarm"] = uni() * 5 if rnd() > 0.5 else None
        da["limit_trip"] = uni() * 10 if rnd() > 0.5 else None
        trend[axis] = da

    # change
    def trend_to_change(tvalue: list[float]) -> list[float]:
        cl: list[float] = []
        pre: float | None = None
        for val in tvalue:
            cl.append(0.0) if pre is None else cl.append(abs(val - pre))
            pre = val
        return cl

    change: dict[str, dict] = {}
    for axis in _AXIS:
        da = {}
        da["value"] = trend_to_change(trend[axis]["value"])
        da["limit_alarm"] = uni() * 5 if rnd() > 0.5 else None
        da["limit_trip"] = uni() * 10 if rnd() > 0.5 else None
        change[axis] = da

    alert_rnd = rnd()

    # compose
    mesg = {
        "timestamp": dt.datetime.now().isoformat(" "),
        "datetime": datetime,
        "physical": physical,
        "trend": trend,
        "change": change,
        "alert": "Trip" if alert_rnd > 0.66 else "Alarm" if alert_rnd > 0.33 else "",
    }

    # publish
    _publish_message(topic=topic, payload=json.dumps(mesg))


def publish_summary_alert(logger_id: str, model: str, serial: str):
    # Web 機能では参照しないメッセージのため、最小限で
    topic = f"summary/{logger_id}/{model}/{serial}/alert"
    now = dt.datetime.now()
    data = {
        "timestamp": now.isoformat(" "),
        "datetime": now.strftime("%Y%m%d_%H%M%S"),
        "physical": "Velocity",
        "level": "",
        "trend": [],
        "change": [],
    }

    # publish
    _publish_message(topic=topic, payload=json.dumps(data))


def main(  # noqa: C901
    logger_id: str, model: str, serial: str, topic: TopicType, repeat: int
) -> int:
    """
    指定したトピックにサンプルメッセージを1秒おきに10回送信する

    Args:
        logger_id (str): サンプルトピックに用いるロガーID
        model (str): サンプルトピックに用いるセンサーモデル
        serial (str): サンプルトピックに用いるセンサーシリアル
        topic (TopicType): トピックの種類
        repeat (int): 繰り返し回数

    Returns:
        int: 終了コード
    """
    for i in range(repeat):
        time.sleep(1)
        match topic:
            # ロガー
            case "logger_status":
                status = "START" if i % 3 == 0 else "STOP"
                publish_logger_status(logger_id=logger_id, status=status)  # type: ignore

            case "logger_error":
                level = "CRITICAL" if i % 3 == 0 else "ERROR"
                message = "ErrorA" if i % 2 == 0 else "ErrorB"
                publish_logger_error(
                    logger_id=logger_id, level=level, message=message  # type: ignore
                )

            # センサー
            case "sensor_status":
                status = "OK" if i % 3 == 0 else "NG"
                publish_sensor_status(
                    logger_id=logger_id, model=model, serial=serial, status=status  # type: ignore
                )

            case "sensor_error":
                level = "CRITICAL" if i % 3 == 0 else "ERROR"
                message = "ErrorA" if i % 2 == 0 else "ErrorB"
                publish_sensor_error(
                    logger_id=logger_id,
                    model=model,
                    serial=serial,
                    level=level,  # type: ignore
                    message=message,
                )

            case "sensor_loss":
                message = "LossA" if i % 2 == 0 else "LossB"
                publish_sensor_loss(
                    logger_id=logger_id, model=model, serial=serial, message=message
                )

            case "sensor_abnormal":
                message = "AbnormalA" if i % 2 == 0 else "AbnormalB"
                publish_sensor_abnormal(
                    logger_id=logger_id, model=model, serial=serial, message=message
                )

            # ハードウェアモニター
            case "hwmonitor_status":
                status = "START" if i % 3 == 0 else "STOP"
                publish_hwmonitor_status(logger_id=logger_id, status=status)  # type: ignore

            case "hwmonitor_data":
                cpu_temperature = 10 if i % 2 == 0 else 20
                cpu_usage = 1 if i % 3 == 0 else 3
                memory_usage = 100 if i % 2 == 0 else 50
                disk_usage = 10 if i % 3 == 0 else 30

                publish_hwmonitor_data(
                    logger_id=logger_id,
                    cpu_temperature=cpu_temperature,
                    cpu_usage=cpu_usage,
                    memory_usage=memory_usage,
                    disk_usage=disk_usage,
                )

            case "hwmonitor_error":
                level = "CRITICAL" if i % 3 == 0 else "ERROR"
                message = "ErrorA" if i % 2 == 0 else "ErrorB"
                publish_hwmonitor_error(
                    logger_id=logger_id, level=level, message=message  # type: ignore
                )

            # VCデータ
            case "vc_data_level":
                level = "OA" if i % 3 == 0 else "G"
                publish_vc_data_level(
                    logger_id=logger_id, model=model, serial=serial, level=level  # type: ignore
                )

            case "vc_data_fft":
                fft = [random.uniform(1e-9, 1e2) for _ in range(27)]
                publish_vc_data_fft(
                    logger_id=logger_id, model=model, serial=serial, fft=fft
                )

            # サマリー
            case "summary_data":
                # パラメータは内部で生成している
                publish_summary_data(logger_id=logger_id, model=model, serial=serial)
                # 頻繁に変化するものではないため、1回で終了
                break

            case "summary_alert":
                publish_summary_alert(logger_id=logger_id, model=model, serial=serial)
                # 頻繁に変化するものではないため、1回で終了
                break

            # それ以外
            case _:
                print("Unknown topic type:", topic, file=sys.stderr)
                show_help()
                return -1

    return 0


def show_help():
    print("Available topic types:", str(TopicType)[len("typing.Literal[") : -1])


if __name__ == "__main__":
    logger_id = "0"
    model = "A352"
    serial = "0001"

    topic_type: TopicType = "logger_status"
    repeat: int = 10

    if len(sys.argv) > 1:
        if sys.argv[1] == "-h":
            show_help()
            sys.exit(0)
        else:
            topic_type = sys.argv[1]  # type: ignore
    if len(sys.argv) > 2:
        repeat = int(sys.argv[2])

    rc = main(
        logger_id=logger_id, model=model, serial=serial, topic=topic_type, repeat=repeat
    )
    sys.exit(rc)
