# Copyright 2025, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

from collections import deque
from dataclasses import dataclass, field
from typing import Literal

from raspi_web.const import _MAX_MESSAGE_COUNT

from .error_type import ErrorType


@dataclass
class LoggerStatus:
    """
    ロガーの状態

    Logger status

    Attributes:
        timestamp (str): ロガーの状態更新日時. "yyyy/mm/dd hh:mm:ss.mmmmmm".
                Timestamp of the logger status update. "yyyy/mm/dd hh:mm:ss.mmmmmm".
        status (Literal["START", "STOP"]): ロガーのステータス
                The status of the logger
    """
    timestamp: str
    status: Literal["START", "STOP"]


@dataclass
class LoggerError:
    """
    ロガーのエラー

    Logger error

    Attributes:
        timestamp (str): エラー発生日時. "yyyy/mm/dd hh:mm:ss.mmmmmm".
                Timestamp of the error occurrence. "yyyy/mm/dd hh:mm:ss.mmmmmm".
        level (ErrorType): エラーの深刻度
                Error severity
        message (str): エラーのメッセージ
                Error message
    """
    timestamp: str
    level: ErrorType
    message: str


@dataclass
class Logger:
    """
    ロガーの情報

    Logger information

    Attributes:
        logger_id (str): ロガーID
                Logger ID
        status (LoggerStatus): ロガーの状態. デフォルトはNone.
                Logger status. Defaults to None.
        error (deque[LoggerError]): ロガーのエラー. 最大3件まで蓄積. 古いものから削除. デフォルトは[].
                Logger errors. Stores up to 3 errors. Removes oldest first. Defaults to [].
    """
    logger_id: str
    status: LoggerStatus | None = None
    error: deque[LoggerError] = field(
        default_factory=lambda: deque(
            maxlen=_MAX_MESSAGE_COUNT,
        )
    )

    def update_status(self, status: LoggerStatus | None):
        """
        ロガーのステータスを更新する

        Update the logger status

        Args:
            status (LoggerStatus | None): 更新したい値
                    Value to update
        """
        self.status = status

    def update_error(self, error: LoggerError | None):
        """
        ロガーのエラーを更新する

        Update the logger errors

        Args:
            error (LoggerError | None): 更新したい値
                    Value to update
        """
        if error is not None:
            self.error.append(error)
        else:
            self.error = deque[LoggerError](maxlen=_MAX_MESSAGE_COUNT)

    def is_empty(self) -> bool:
        """
        ロガーの各要素が空かどうかを判断する

        Check if all elements of the logger are empty

        Returns:
            bool: 空ならTrue, 1箇所でも値があればFalse
                    True if empty, False if any element has a value
        """
        return not (self.status or self.error)
