# Copyright 2025, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

from dataclasses import dataclass, field
from typing import Literal, TypeAlias

# VCLevelTypeは、特定の文字列リテラルの型エイリアスを定義します
# VCLevelType defines a type alias for specific string literals
VCLevelType: TypeAlias = Literal["OA", "A", "B", "C", "D", "E", "F", "G"]


@dataclass
class VCLevel:
    """
    VC判定値

    VC judgment value
    """
    timestamp: str
    level: VCLevelType


@dataclass
class FFTData:
    """
    FFTデータ

    FFT data
    """
    timestamp: str
    value: list[float] = field(default_factory=list)


@dataclass
class VCData:
    """
    VC判定値、FFTデータ

    VC judgment value, FFT data

    Attributes:
        vc_level (VCLevel): VC判定値
                VC judgment value
        fft_data (FFTData): FFTデータ
                FFT data
    """
    vc_level: VCLevel | None = None
    fft_data: FFTData | None = None

    def is_empty(self) -> bool:
        """
        vc_dataが空かどうかを判断する

        Check if vc_data is empty

        Returns:
            bool: 空ならTrue, 1箇所でも値があればFalse
                    True if empty, False if any element has a value
        """
        if not (self.vc_level or self.fft_data):
            return True
        return False

    def update_vc_level(self, vc_level: VCLevel | None):
        """
        VCDataのレベルを更新する

        Update the level of VCData

        Args:
            vc_level (VCLevel | None): 更新したい値
                    Value to update
        """
        self.vc_level = vc_level

    def update_fft_data(self, fft_data: FFTData | None):
        """
        VCDataのfft_dataを更新する

        Update the fft_data of VCData

        Args:
            fft_data (FFTData | None): 更新したい値
                    Value to update
        """
        self.fft_data = fft_data
