# Copyright 2025, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

from dash import dash_table, html

# 表示する情報テーブルの列を定義
# Define columns for the information table to be displayed
_TARGET_INFO_TABLE_COLUMNS = [
    {"id": "logger_id", "name": "Logger ID"},
    {"id": "model", "name": "Model"},
    {"id": "serial", "name": "Serial number"},
]


def target_sensor(logger_id: str, model: str, serial: str):
    """
    グラフ画面上の Target Sensor ブロックを表示するコンポーネントを返す

    Returns a component that displays the Target Sensor block on the graph screen

    Args:
        logger_id: ロガーID
                Logger ID
        model: モデル
                Model
        serial: シリアルNO
                Serial number

    Returns:
        html.Div: Target Sensor を表示する Div コンポーネント
                Div component displaying the target sensor
    """
    return html.Div(
        [
            html.H2("Target Sensor"),
            dash_table.DataTable(
                id="target_info_table",
                columns=_TARGET_INFO_TABLE_COLUMNS,
                data=[{"logger_id": logger_id, "model": model, "serial": serial}],
                css=[{"selector": "table", "rule": "table-layout: fixed"}],
                style_table={"width": "30%"},
                style_cell={
                    "textAlign": "left",
                },
                style_cell_conditional=[
                    {"if": {"column_id": "logger_id"}, "width": "20%"},
                    {"if": {"column_id": "model"}, "width": "20%"},
                    {"if": {"column_id": "serial"}, "width": "40%"},
                ],
            ),
        ]
    )
