# Copyright 2025, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

from datetime import datetime as dt
from urllib.parse import parse_qs, urlparse

import dash
import plotly.graph_objects as go
from dash import dcc, html

from raspi_web.const import TREND_COLOR_ALARM, TREND_COLOR_BASE, TREND_COLOR_TRIP
from raspi_web.data.summary import SensorSummary
from raspi_web.infra.data_manager import DataManager
from raspi_web.ui.component.target_sensor import target_sensor

# グラフ軸
# Graph axis
_GRAPH_AXIS = ["X", "Y", "Z", "C"]

# ラベル軸
# Label axis
_LABEL_AXIS = [v if v != "C" else "Composite" for v in _GRAPH_AXIS]

# Dashアプリケーションのインスタンスを取得する
# Get the instance of the Dash application
app = dash.get_app()

# グラフページを登録する
# Register the graph page
dash.register_page(__name__, path_template="/trend")


def _xyzc_graph(summary: SensorSummary, key: str, axlist: list[str]) -> list[dcc.Graph]:
    """
    XYZC の４連のグラフを作成する

    Creates a set of four graphs for XYZC data

    Args:
        summary (SensorSummary): SensorSummary オブジェクト
            SensorSummary object
        key (str): trend|change キー
            trend|change key
        axlist (list[str]): 表示する軸のリスト
            List of axes to display

    Returns:
        list[dcc.Graph]: XYZC 4連グラフを格納したリスト
            List containing four XYZC graphs
    """
    # 共通項目
    # Common elements
    # - datetime: 日付軸はミリ秒を要求するため変換する
    # - datetime: Convert to milliseconds for the date axis
    datetime = [
        dt.strptime(s, "%Y%m%d_%H%M%S").timestamp() * 1000 for s in summary["datetime"]
    ]
    # - physical: ラベル表示用に装飾
    # - physical: Format for label display
    physical = summary["physical"]
    match physical:
        case "Velocity":
            physical = "Velocity (mm/s)"
        case "Acceleration":
            physical = "Acceleration (G)"
        case "Displacement":
            physical = "Displacement (mm)"

    graphdat = summary[key]

    # 軸別グラフ生成
    # Generate graphs for each axis
    x_axis = go.layout.XAxis(
        title="Datetime",
        linecolor="black",
        linewidth=1,
        gridcolor="#C8C8C8",
        type="date",
    )
    y_axis = go.layout.YAxis(
        title=physical,
        linecolor="black",
        linewidth=1,
        gridcolor="#C8C8C8",
        zerolinecolor="#C8C8C8",
        zerolinewidth=1,
    )

    lg = []
    for axis in axlist:
        name = f"{key}_{axis}"
        axisdat = graphdat[axis]

        # - レイアウト
        # - Layout
        layout = go.Layout(
            title={"text": _LABEL_AXIS[_GRAPH_AXIS.index(axis)]},
            plot_bgcolor="white",
            showlegend=True,
            xaxis=x_axis,
            yaxis=y_axis,
        )

        # - 折れ線グラフ追加
        # - Add line graph
        fig = go.Figure(layout=layout)
        fig.add_trace(
            go.Scatter(
                name=f"{axis} rms", x=datetime, y=axisdat["value"], mode="lines+markers"
            )
        )

        # - 閾値の水平線追加
        # - Add horizontal threshold lines
        if "baseline" in axisdat and axisdat["baseline"] is not None:
            fig.add_hline(
                name="baseline",
                y=axisdat["baseline"],
                line={"dash": "dot", "color": TREND_COLOR_BASE},
                showlegend=True,
            )
        if axisdat["limit_alarm"] is not None:
            fig.add_hline(
                name="alarm",
                y=axisdat["limit_alarm"],
                line={"dash": "dot", "color": TREND_COLOR_ALARM},
                showlegend=True,
            )
        if axisdat["limit_trip"] is not None:
            fig.add_hline(
                name="trip",
                y=axisdat["limit_trip"],
                line={"dash": "dot", "color": TREND_COLOR_TRIP},
                showlegend=True,
            )

        # - グラフ要素
        # - Graph component
        graph = dcc.Graph(
            id=name,
            figure=fig,
            style={"width": f"{100//len(axlist)}%", "display": "inline flow-root"},
        )
        lg.append(graph)

    return lg


def layout(logger_id=None, model=None, serial=None, **others):
    """
    Trend 画面のレイアウトを定義して返す関数

    Function that defines and returns the layout of the Trend screen

    Args:
        logger_id: ロガーID
            Logger ID
        model: モデル
            Model
        serial: シリアルNO
            Serial number

    Returns:
        html.Div|None: Trend グラフ画面を表示するDIVコンポーネント
            DIV component displaying the trend graph screen
    """
    if not (logger_id and model and serial):
        return None

    return html.Div(
        [
            html.H1("Vibration Trend and Change Graph"),
            target_sensor(logger_id, model, serial),
            html.H2("Trend of vibration magnitude"),
            html.Div(id="trend_graph"),
            dcc.Checklist(
                id="axis_check",
                options=[
                    {"label": lb, "value": vl}
                    for lb, vl in zip(_LABEL_AXIS, _GRAPH_AXIS)
                ],
                value=_GRAPH_AXIS,
                inline=True,
                inputStyle={"margin-inline": "1em 0.25em"},
            ),
            html.H2("Change in vibration magnitude"),
            html.Div(id="change_graph"),
            dcc.Location(id="url"),
            # 更新頻度は高くないが、反映させるために1分周期で更新
            # Update is not frequent, but set to 1 minute to reflect changes
            dcc.Interval(id="interval", interval=60 * 1000),
        ]
    )


@dash.callback(
    dash.Output(component_id="trend_graph", component_property="children"),
    dash.Output(component_id="change_graph", component_property="children"),
    dash.Input(component_id="url", component_property="search"),
    dash.Input(component_id="axis_check", component_property="value"),
    dash.Input(component_id="interval", component_property="n_intervals"),
)
def update_graph(search: str, selected: list[str], _):
    # クエリパラメータ解析
    # Parse query parameters
    qd = parse_qs(urlparse(search).query)
    logger_id = qd["logger_id"][0]
    model = qd["model"][0]
    serial = qd["serial"][0]

    # チェックボックス選択のソート: 選択順によって意図しない順序になるため
    # Sort checkbox selections: avoid unintended order due to selection sequence
    axis = [v for v in _GRAPH_AXIS if v in selected]

    # SensorSummary 取得
    # Retrieve SensorSummary
    dm = DataManager.get_instance()
    summary = dm.get_summary(logger_id=logger_id, model=model, serial=serial)
    if summary is None:
        return html.Div("No data available")

    return _xyzc_graph(summary, "trend", axis), _xyzc_graph(summary, "change", axis)
