# Copyright 2025, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

import threading
from typing import Callable, ParamSpec, TypeVar

# デコレータの引数にとる関数の引数の型のパラメータ仕様変数
# Parameter specification variable for the types of arguments of the function taken by the decorator
_P = ParamSpec("_P")

# デコレータの引数にとる関数の戻り値の型の変数
# Type variable for the return type of the function taken by the decorator
_R = TypeVar("_R")


class ThreadLock:
    """
    スレッドロックを提供するクラス

    関数単位でロックするデコレータ機能や、コンテキストとしてロックを提供する機能を有する

    Provides thread locking functionality.

    Includes a decorator for function-level locking and context manager support.
    """

    def __init__(self) -> None:
        # 再入可能スレッドロック
        # Reentrant thread lock
        self._lock = threading.RLock()

    def decorator(self, func: Callable[_P, _R]) -> Callable[_P, _R]:
        """
        関数に対してスレッドロックを提供するデコレーター

        A decorator that provides thread locking for a function.

        Args:
            func (Callable[_P, _R]): ロック対象の関数
                The function to be locked.

        Returns:
            Callable[_P, _R]: ロック対象の関数にロック機能を付加した関数
                The function wrapped with locking functionality.
        """

        def _wrapper(*args: _P.args, **kwargs: _P.kwargs):
            """
            対象の関数にスレッドロックを付加するラッパー関数
            A wrapper function that adds thread locking to the target function.
            """
            with self._lock:
                result = func(*args, **kwargs)
            return result

        return _wrapper

    def __enter__(self) -> None:
        """
        with を使用したコンテキストで使用可能にする enter メソッド

        ロックを獲得する

        Enter method to enable usage in a `with` context.

        Acquires the lock.
        """
        self._lock.acquire()

    def __exit__(self, *_) -> None:
        """
        with を使用したコンテキストで使用可能にする exit メソッド

        ロックを解放する

        Exit method to enable usage in a `with` context.

        Releases the lock.

        Args:
            _: 引数に応じた処理をしないため抑止
                    Suppresses handling of arguments as they are not used.
        """
        self._lock.release()
