# Copyright 2025, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

import json
import sys

import raspi_summary.config as config
import raspi_summary.util.logging as logging
from raspi_summary.const import Const, ExitCode
from raspi_summary.domain.logger import Measure, MeasureData, MeasureInfo
from raspi_summary.domain.summary import DataSet, SummaryCalc, SummaryData
from raspi_summary.util.message import MessageService


def main() -> int:  # noqa: C901
    """
    メイン関数

    Main function
    """
    # 初期化処理
    # Initialization process
    # - logging 初期化
    # - Initialize logging
    logging.init_logging()
    logger = logging.get_logger(__name__)

    # - 設定項目取得
    # - Retrieve configuration items
    logger_id = config.logger_id()
    output_path = config.output_path()
    max_measures = config.max_measurements()
    sensor_model = config.sensor_model()
    summary_config = config.summary_config()
    message_config = config.message_config()

    if not (
        logger_id
        and output_path
        and sensor_model
        and summary_config
        and max_measures is not None
    ):
        # 必須のものがすべて設定されていること
        # Ensure all required items are configured
        return ExitCode.EX_NG

    # - 収集データセット生成
    # - Create dataset for collected data
    dataset = DataSet()

    # - SummaryCalc オブジェクト生成
    # - Create SummaryCalc object
    summary_calc = SummaryCalc(summary_config)

    # メイン処理（データ集計）
    # Main process (data aggregation)
    # - 計測フォルダサーチ
    # - Search for measurement folders
    logger.debug(f"visit {output_path}")
    for measure in Measure.iterate(output_path, max_recent=max_measures):
        logger.debug(f"-visit {measure}")

        # - info ファイルサーチ
        # - Search for info files
        for info in MeasureInfo.iterate(measure, model_ptn=sensor_model):
            logger.debug(f"--visit {info}")
            sensor = f"Sensor {info.measure}/{info.key}"

            # - Raw モードの計測のみ対象
            # - Raw mode measurements only
            if info.output_type != info.OUTPUT_TYPE_RAW:
                logger.warning(
                    f"{sensor} is skipped "
                    + "because only raw mode measurements are summarized"
                )
                continue

            # - summary ファイル読み込み
            # - Load summary file
            summary_data = SummaryData.load(info)
            if summary_data is not None:
                # あればそれを使う
                # Use it if available
                logger.debug(f"---read  {summary_data}")

                # - summary 登録
                # - Register summary
                dataset.add(info.key, summary_data)
                continue

            # なければ計測データから計算
            # If not available, calculate from measurement data
            # - 計測データ取得
            # - Retrieve measurement data
            value: SummaryData.Value | None = None
            found = False
            for data in MeasureData.iterate(measure, info.key):
                logger.debug(f"---visit {data}")
                found = True

                # - サマリー計算
                # - Perform summary calculation
                value = summary_calc.summarize_measure(info, data)
                break

            # - サマリー計算結果確認
            # - Check summary calculation result
            if value is None:
                if found:
                    logger.warning(f"{sensor} does not have enought data")
                else:
                    logger.warning(f"{sensor} has no data")
                continue

            # - summary ファイル保存
            # - Save summary file
            summary_data = SummaryData.save(info, value)
            logger.debug(f"---saved {summary_data}")

            # - サマリー登録
            # - Register summary
            dataset.add(info.key, summary_data)

    # メイン処理（センサー別集計 & メッセージ送信）
    # Main process (sensor-wise aggregation & message sending)
    # - メッセージサービス生成
    # - Create message service
    message = MessageService(message_config)
    try:
        # - メッセージサービス初期化
        # - Initialize message service
        message.start()

        # - 見つかったセンサーごとに処理する
        # - Process each detected sensor
        for key, datalist in dataset.iterate():
            # - センサー監視指標取得
            # - Get monitoring parameters for the sensor
            sensor_config = config.sensor_config(key.model, key.serial)

            # センサーサマリー
            # Sensor summary
            summary, alert = summary_calc.summarize_sensor(datalist, sensor_config)

            # - ログダンプ
            # - Dump summary to log
            summary_dict = summary.to_dict()
            logger.info(f"Sensor {key} summary: {json.dumps(summary_dict, indent=2)}")

            # - メッセージ送信
            # - Send message
            topic = Const.TOPIC_DATA.format(
                logger_id=logger_id, model=key.model, serial=key.serial
            )
            message.send(topic, summary_dict)

            # サマリー警告 - 警告がある場合のみ
            # Summary alert - only if alert exists
            if not alert.level:
                continue

            # - ログダンプ
            # - Dump alert to log
            alert_dict = alert.to_dict()
            logger.warning(f"Sensor {key} alert: {json.dumps(alert_dict, indent=2)}")

            # - メッセージ送信
            # - Send alert message
            alert_topic = Const.TOPIC_ALERT.format(
                logger_id=logger_id, model=key.model, serial=key.serial
            )
            message.send(alert_topic, alert_dict)

    except Exception:
        logger.exception("Unknown error occurred")
        return ExitCode.EX_NG

    finally:
        message.close()

    return ExitCode.EX_OK


if __name__ == "__main__":
    rc = main()
    sys.exit(rc)
