# Copyright 2025, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

import csv
import re
from collections.abc import Generator
from datetime import datetime
from pathlib import Path
from typing import NamedTuple

import raspi_summary.util.logging as logging


class _PathElement:
    """
    Path 要素を持つ、ファイル・ディレクトリ要素の共通クラス

    Common class for file and directory elements with a Path component
    """

    def __init__(self, path: Path) -> None:
        self.path = path

    def __str__(self) -> str:
        return self.path.name


class Measure(_PathElement):
    """
    Logger が生成する計測フォルダを表すクラス

    Class representing a measurement folder generated by the logger
    """

    _PT_MEASURE = "*_*"
    """
    計測フォルダ名のパターン

    Pattern for measurement folder names
    """

    def __init__(self, path: Path) -> None:
        """
        Measure のコンストラクタ

        Constructor for Measure

        Args:
            path (Path): 計測フォルダを示す Path オブジェクト
                Path object representing the measurement folder

        Raises:
            ValueError: 指定された Path から年月日時分秒を解析できない場合
                If the datetime cannot be parsed from the given Path
        """
        super().__init__(path)

        # 年月日時分秒を解析 -> 書式が合わない場合は例外が発生する
        # Parse year, month, day, hour, minute, second
        # -> raises exception if format is incorrect
        self.datetime = datetime.strptime(path.name, "%Y%m%d_%H%M%S")

    @classmethod
    def iterate(
        cls, output_path: str, max_recent: int = 0, reverse: bool = False
    ) -> Generator["Measure"]:
        """
        計測フォルダをサーチする Generator を返す

        - デフォルトではフォルダ名の昇順で出力される
        - max_recent と reverse を両方指定した場合、先に max_recent により最新からの件数が絞り込まれ、
          その結果から reverse により返却順が決められる
        - 計測フォルダ名が所定の形式に合わない場合は、WARNING 出力してスキップされる
            - このフォルダは max_recent の対象からも除外される

        Returns a generator that searches measurement folders.

        - By default, folders are returned in ascending name order.
        - If both max_recent and reverse are specified, the latest folders are first filtered by max_recent,
            then the return order is determined by reverse.
        - If a folder name does not match the expected format, a WARNING is logged and the folder is skipped.
            - Such folders are also excluded from max_recent filtering.

        Args:
            output_path (str): ロガーの保存先フォルダパスを示す文字列
                String representing the logger's output folder path
            max_recent (int): サーチする件数を最近 N 件に限定する、デフォルト=0（全件）
                Limits the number of folders to the most recent N; default is 0 (all)
            reverse (bool): True 指定でフォルダ名の降順で出力する、デフォルト=False
                If True, folders are returned in descending name order;
                default is False

        Returns:
            Generator[Measure]: Measure を順次返す Generator
                Generator that yields Measure objects
        """
        logger = logging.get_logger(__name__)

        # 計測フォルダサーチ - max_recent に対応するため、2パスで処理する
        # Search measurement folders — processed in two passes to support max_recent

        # - 全件サーチ
        # - Search all folders
        measures = []
        for path in sorted(Path(output_path).glob(cls._PT_MEASURE)):
            # - ディレクトリであること
            # - Must be a directory
            if not path.is_dir():
                continue

            # - Measure オブジェクト化
            # - Convert to Measure object
            try:
                measures.append(Measure(path))
            except ValueError:
                # フォルダ名が想定に合わない場合は次へ
                # Skip if folder name does not match expected format
                logger.warning("The measurement folder name is incorrect: " + path.name)

        # - max_recent
        if max_recent:
            # 降順にして絞り込み、昇順に戻す
            # Sort descending, filter, then restore ascending order
            measures = measures[::-1][:max_recent][::-1]

        # - reverse
        if reverse:
            # 降順に並び替え
            # Sort in descending order
            measures = measures[::-1]

        # - iterate & yield
        for meas in measures:
            yield meas


class SensorKey(NamedTuple):
    """
    センサーのキー情報を表すクラス

    Class representing key information of a sensor
    """

    model: str
    serial: str

    def __str__(self) -> str:
        return f"{self.model}_{self.serial}"


class MeasureInfo(_PathElement):
    """
    センサーが出力する計測情報ファイルを表すクラス

    Class representing a measurement info file output by a sensor
    """

    _PT_INFO = "{model}_{serial}_info.csv"
    """
    計測情報ファイル名のパターン; `model` と `serial` を置き換えて使う

    Pattern for measurement info file names; replace `model` and `serial`
    """

    _RE_INFO = re.compile(_PT_INFO.format(model="([^_]*)", serial="([^_]*)"))
    """
    計測情報ファイル名から `model` と `serial` を抜き出す正規表現オブジェクト

    Regex object to extract `model` and `serial` from measurement info file name
    """

    _KEY_SPS = "SPS"
    _KEY_PHYSICAL = "PHYSICAL"
    _KEY_OUTPUT_TYPE = "OUTPUT_TYPE"
    _KEYS = [_KEY_SPS, _KEY_PHYSICAL, _KEY_OUTPUT_TYPE]

    OUTPUT_TYPE_RAW = "Raw"
    """Raw mode"""

    def __init__(self, path: Path, measure: Measure | str) -> None:
        """
        MeasureInfo のコンストラクタ

        Constructor for MeasureInfo

        Args:
            path (Path): INFO ファイルを示す Path オブジェクト
                Path object representing the INFO file
            measure (Measure|str): Measureオブジェクト or measure フォルダ名
                Measure object or name of the measure folder

        Raises:
            Exception: ファイル名から SensorKey を抽出できない場合
                If SensorKey cannot be extracted from the file name
                - 基本的には発生しない
                    Normally should not occur
        """
        super().__init__(path)

        # SensorKey を抽出
        # Extract SensorKey
        match = self._RE_INFO.fullmatch(path.name)
        assert match
        self.key = SensorKey(model=match.group(1), serial=match.group(2))

        # Measure パス要素
        # Measure path element
        self.measure: str = str(measure)

        # 属性抽出用辞書
        # Dictionary for extracting attributes
        self._prop: dict[str, str] = {}

    @property
    def sps(self) -> int:
        self._load()
        return int(self._prop[self._KEY_SPS])

    @property
    def physical(self) -> str:
        self._load()
        return self._prop[self._KEY_PHYSICAL]

    @property
    def output_type(self) -> str:
        self._load()
        return self._prop[self._KEY_OUTPUT_TYPE]

    def _load(self) -> None:
        """
        計測情報ファイルから必要な情報をロードする

        Load necessary information from the measurement info file
        """
        # 読み込み済みスキップ
        # Skip if already loaded
        if self._prop:
            return

        # _KEYS に指定された項目を保持する
        # Store items specified in _KEYS
        with open(self.path, newline="") as f:
            for row in csv.reader(f):
                if row[0] in self._KEYS:
                    self._prop[row[0]] = row[1]

    @classmethod
    def iterate(
        cls, measure: Measure, model_ptn: str = "*"
    ) -> Generator["MeasureInfo"]:
        """
        INFO ファイルをサーチする Generator を返す

        出力は INFO ファイル名の昇順で行われる

        Returns a generator that searches for INFO files

        Output is in ascending order of INFO file names

        Args:
            measure (Measure): 計測フォルダを表す Measure オブジェクト
                Measure object representing the measurement folder
            model_ptn (str): 対象モデルを指定するパターン文字列
                Pattern string to specify target model
                - "*" (default) : 全てのモデルを対象にする
                    Targets all models
                - else: その文字列に含まれるモデルを対象にする
                    Targets models included in the string

        Returns:
            Generator[MeasureInfo]: MeasureInfo を順次返す Generator
                Generator that yields MeasureInfo objects
        """
        logger = logging.get_logger(__name__)

        # INFO ファイルサーチ
        # Search for INFO files
        ptn = cls._PT_INFO.format(model="*", serial="*")
        for path in sorted(measure.path.glob(ptn)):
            # - ファイルであること
            # - Must be a file
            if not path.is_file():
                continue

            # - MeasureInfo 化
            # - Convert to MeasureInfo
            try:
                info = MeasureInfo(path, measure)

                # - 対象モデルにマッチすること
                # - Must match target model
                if model_ptn == "*" or info.key.model in model_ptn:
                    yield info

            except Exception:
                logger.exception(f"Failed to iterate measure info: {path.absolute()}")


class MeasureData(_PathElement):
    """
    センサーが出力した計測データファイルを表すクラス

    Class representing measurement data files output by a sensor
    """

    _PT_MEASURE = "{model}_{serial}_*_*_*_*.csv"
    """
    計測データファイル名のパターン; `model` と `serial` を置き換えて使う

    Pattern for measurement data file names; replace `model` and `serial`
    """

    def __init__(self, path: Path) -> None:
        """
        MeasureData のコンストラクタ

        Constructor for MeasureData

        Args:
            path (Path): 計測データファイルを示す Path オブジェクト
                Path object representing the measurement data file
        """
        super().__init__(path)

    @classmethod
    def iterate(
        cls, measure: Measure, key: SensorKey, reverse: bool = False
    ) -> Generator["MeasureData"]:
        """
        そのセンサーの計測データファイルをサーチする Generator を返す

        デフォルトでは計測データファイル名の昇順で出力する

        Returns a generator that searches for measurement data files for the sensor

        By default, files are returned in ascending name order

        Args:
            measure (Measure): 計測フォルダを表す Measure オブジェクト
                Measure object representing the measurement folder
            key (SensorKey): 対象センサーのキー情報を表す SensorKey オブジェクト
                SensorKey object representing the target sensor
            reverse (bool): True 指定でファイル名の降順で出力する、デフォルト=False
                If True, files are returned in descending name order; default is False

        Returns:
            Generator[MeasureData]: MeasureData を順次返す Generator
                Generator that yields MeasureData objects
        """
        # 計測データファイルサーチ
        # Search for measurement data files
        ptn = cls._PT_MEASURE.format(model=key.model, serial=key.serial)
        for data in sorted(measure.path.glob(ptn), reverse=reverse):
            # - ファイルであること
            # - Must be a file
            if not data.is_file():
                continue

            # - MeasureData 化
            # - Convert to MeasureData
            yield MeasureData(data)
