# Copyright 2025, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

import os

import dotenv

# モジュールロードで自動読み込み
# Automatically load on module load
dotenv.load_dotenv(override=True)


def init(env_path: str) -> None:
    """
    明示的 Dotenv 初期化を行う関数

    通常はモジュールロードと同時にデフォルトの .env がロードされる想定。
    テストなどの特殊な用途の初期化用。

    Function to explicitly initialize Dotenv

    Normally, the default .env is expected to be loaded at the same time as the module load.
    For special purposes such as testing.

    Args:
        env_path (str): .env ファイルへのパス
                Path to the .env file
    """
    dotenv.load_dotenv(env_path, override=True)


def parse_int(value: str) -> int | None:
    """
    文字列から整数値に変換する関数

    Function to convert a string to an integer

    Args:
        value (str): 変換対象の文字列
                String to be converted

    Returns:
        int | None : 変換した整数値、変換に失敗した場合はNoneを返す
                Converted integer, returns None if conversion fails
    """
    try:
        return int(value)
    except ValueError:
        return None


def parse_float(value: str) -> float | None:
    """
    文字列から浮動小数点値に変換する関数

    Function to convert a string to a floating-point value

    Args:
        value (str): 変換対象の文字列
                String to be converted

    Returns:
        float | None : 変換した浮動小数値、変換に失敗した場合はNoneを返す
                Converted floating-point value, returns None if conversion fails
    """
    try:
        return float(value)
    except ValueError:
        return None


def parse_bool(value: str) -> bool | None:
    """
    文字列から真偽値に変換する関数

    Function to convert a string to a boolean value

    Args:
        value (str): 変換対象の文字列
                String to be converted

    Returns:
        bool | None : 変換した真偽値、変換に失敗した場合はNoneを返す
                Converted boolean value, returns None if conversion fails
    """
    try:
        lvalue = value.lower()
        return True if lvalue == "true" else False if lvalue == "false" else None
    except ValueError:
        return None


def get_str(key: str) -> str | None:
    """
    環境変数の名前を指定してその値を取得する関数

    Function to get the value of an environment variable by specifying its name

    Args:
        key (str): 取得したい環境変数（.envファイル）のキー名
                Key name of the environment variable to be retrieved (.env file)

    Returns:
        str | None : 環境変数から取得した値
                Value retrieved from the environment variable
    """
    return os.getenv(key)


def get_int(key: str) -> int | None:
    """
    環境変数の名前を指定してその値を整数で取得する関数

    Function to get the value of an environment variable as an integer by specifying its name

    Args:
        key (str): 取得したい環境変数（.envファイル）のキー名
                Key name of the environment variable to be retrieved (.env file)

    Returns:
        int | None : 環境変数から取得した値、整数に変換に失敗した場合はNoneを返す
                Value retrieved from the environment variable, returns None if conversion to integer fails
    """
    s = get_str(key)
    if s is None:
        return None
    return parse_int(s)


def get_float(key: str) -> float | None:
    """
    環境変数の名前を指定してその値を浮動小数で取得する関数

    Function to get the value of an environment variable as a floating-point number by specifying its name

    Args:
        key (str): 取得したい環境変数（.envファイル）のキー名
                Key name of the environment variable to be retrieved (.env file)

    Returns:
        float | None : 環境変数から取得した値、浮動小数に変換に失敗した場合はNoneを返す
                Value retrieved from the environment variable, returns None if conversion to floating-point fails
    """
    s = get_str(key)
    if s is None:
        return None
    return parse_float(s)


def get_bool(key: str) -> bool | None:
    """
    環境変数の名前を指定してその値を真偽値で取得する関数

    Function to get the value of an environment variable as a boolean by specifying its name

    Args:
        key (str): 取得したい環境変数（.envファイル）のキー名
                Key name of the environment variable to be retrieved (.env file)

    Returns:
        bool | None : 環境変数から取得した値、真偽値に変換に失敗した場合はNoneを返す
                Value retrieved from the environment variable, returns None if conversion to boolean fails
    """
    s = get_str(key)
    if s is None:
        return None
    return parse_bool(s)
