# Copyright 2025, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

import json
import logging
import time
from datetime import datetime
from traceback import format_exception_only
from typing import NamedTuple, cast

import paho.mqtt.client as mqtt
from paho.mqtt.enums import CallbackAPIVersion


class MessageConfig(NamedTuple):
    """
    メッセージ送信に関する設定値を保持するクラス

    Class that holds configuration values related to message sending
    """

    host: str | None
    port: int | None
    clid: str | None


class MessageService:
    """
    MQTT メッセージ送信機能を提供するクラス

    Class that provides MQTT message sending functionality
    """

    _logger: logging.Logger = logging.getLogger(__name__)

    _QOS: int = 1
    """
    MQTT メッセージの QOS: 配信保証させるために 1

    MQTT message QOS: set to 1 to ensure delivery guarantee
    """

    def __init__(self, config: MessageConfig) -> None:
        self._config = config
        self._mqttc: mqtt.Client | None = None

    def send(self, topic: str, data: dict) -> None:
        """
        指定トピックにメッセージを送信するメソッド

        Method to send a message to the specified topic

        Args:
            topic (str): メッセージ送信先トピック
                Topic to which the message will be sent
            data (dict): メッセージとして送信するデータ dict
                Data to be sent as a message in dict format
                - JSON 形式に変換して送信される
                  Converted to JSON format before sending
        """
        if not self._mqttc:
            return

        # 送信データ構築
        # Build message data
        # - timestamp はすべてに付加する
        # - Add timestamp to all messages
        msgd = {"timestamp": datetime.now().isoformat(" ")}

        # - 受け取ったデータを追加
        # - Add received data
        msgd.update(data)

        # メッセージ送信
        # Send message
        self._logger.debug(f"Publish message: topic={topic}, data={msgd}")
        inf = self._mqttc.publish(
            topic=topic, payload=json.dumps(msgd), retain=True, qos=self._QOS
        )

        if inf.rc != mqtt.MQTT_ERR_SUCCESS:
            self._logger.warning(
                f"Failed to publish message: topic={topic}, data={msgd}, rc={inf.rc}"
            )

    def start(self) -> None:
        """
        サービスを開始するメソッド

        メッセージブローカーへの接続を行う

        Method to start the service

        Establish connection to the message broker
        """
        logger = MessageService._logger
        # * この logger のレベルを DEBUG に設定すると paho のログも出力される
        # * Setting this logger level to DEBUG will also output paho logs
        # logger.setLevel(logging.DEBUG)

        # 設定で有効な場合のみ処理する
        # Process only if configuration is valid
        if not (self._config.host and self._config.port):
            logger.debug("Skipped to connect message broker")
            return

        # MQTT クライアント構築
        # Build MQTT client
        self._mqttc = mqtt.Client(
            CallbackAPIVersion.VERSION2, client_id=self._config.clid
        )
        self._mqttc.on_connect = MessageService._on_connect

        # - 接続失敗時にサービス無効化が必要なため、userdata 登録する
        # - Register userdata to disable service on connection failure
        self._mqttc.user_data_set({"service": self})

        # - DEBUG モードの場合は MQTT のログも出力する
        # - Output MQTT logs if in DEBUG mode
        if logger.getEffectiveLevel() == logging.DEBUG:
            self._mqttc.enable_logger(logger)

        # 接続
        # Connect
        try:
            self._mqttc.connect(self._config.host, self._config.port)

        except Exception as e:
            # - 接続失敗： 無効化
            # - Connection failed: disable service
            logger.warning(
                f"Failed to connect message broker: {format_exception_only(e)}"
            )
            self._invalidate()

        else:
            # - ループ開始
            # - Start loop
            self._mqttc.loop_start()
            # - 少し待つ
            # - Wait a bit
            time.sleep(1)

    def close(self) -> None:
        """
        サービスを終了するメソッド

        メッセージブローカーへの接続を閉じる

        Method to stop the service

        Close the connection to the message broker
        """
        if self._mqttc:
            self._mqttc.loop_stop()
            self._invalidate()

    def _invalidate(self) -> None:
        """
        MQTT クライアントを破棄する

        Destroy the MQTT client
        """
        self._mqttc = None

    @staticmethod
    def _on_connect(client: mqtt.Client, userdata: dict, flags, rc, props):
        """
        MQTT 接続コールバック

        MQTT connection callback
        """
        logger = MessageService._logger
        logger.debug(f"At on_connect: flags={flags}, rc={rc}")

        if rc == mqtt.MQTT_ERR_SUCCESS:
            logger.info("Connected to message broker")
        else:
            # ** on_connect は失敗時にも呼ばれうるが、このサービスは失敗時は継続しないため破棄する
            # ** on_connect may be called even on failure, but this service does not continue
            #    on failure, so it will be destroyed
            logger.warning(
                f"Failed to connect message broker: rc={rc}, "
                + "messaging is disabled and measurement will continue."
            )

            # - 接続に失敗した場合は再試行させずに即切断する
            # - Disconnect immediately without retry if connection fails
            client.disconnect()

            # - サービスを無効化する
            # - Disable the service
            service = cast(MessageService, userdata.get("service"))
            service._invalidate()
