# Copyright 2025, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

import logging
from pathlib import Path

from raspi_summary.domain.logger import Measure, MeasureData, MeasureInfo, SensorKey
from tests.asset.path import MEASURE_PATH


class TestLogger:
    class TestMeasure:
        def test_iterate(self, caplog):
            # WARNING 以上のログを収集
            # Capture logs with level WARNING or higher
            caplog.set_level(logging.WARNING)

            # 出力フォルダーをサーチ
            # Search output folder
            ls = list(Measure.iterate(MEASURE_PATH))

            assert len(ls) == 2

            # measure 出力は昇順
            # Measure output should be in ascending order
            assert ls[0].path.name == "20250801_091011"
            assert ls[1].path.name == "20250801_101112"
            # 20250801_invalid は含まれない
            # "20250801_invalid" should not be included

            assert ls[0].datetime.year == 2025
            assert ls[0].datetime.month == 8
            assert ls[0].datetime.day == 1

            assert ls[1].datetime.hour == 10
            assert ls[1].datetime.minute == 11
            assert ls[1].datetime.second == 12

            # ログが出力されていること
            # Ensure a warning log was emitted
            assert len(caplog.records) == 1
            assert caplog.records[0].levelname == "WARNING"
            assert (
                caplog.records[0].message
                == "The measurement folder name is incorrect: 20250801_invalid"
            )

        def test_iterate_reverse(self, caplog):
            # 降順でフォルダを取得
            # Retrieve folders in descending order
            ls = list(Measure.iterate(MEASURE_PATH, reverse=True))

            assert len(ls) == 2

            # measure 出力は降順
            # Measure output should be in descending order
            assert ls[1].path.name == "20250801_091011"
            assert ls[0].path.name == "20250801_101112"
            # 20250801_invalid は含まれない
            # "20250801_invalid" should not be included

        def _create_dummy_measures(self, tmp_path: Path, names: list[str]) -> None:
            for name in names:
                Path(tmp_path, name).mkdir()

        _dummy_list = [
            "20250801_100000",
            "20250801_110000",
            "20250801_120000",
            "20250801_130000",
            "20250801_140000",
        ]

        def test_max_recent(self, tmp_path: Path):
            # ダミーフォルダ作成
            # Create dummy folders
            self._create_dummy_measures(tmp_path, self._dummy_list)

            # 件数を指定してフォルダを取得
            # Retrieve folders with specified count
            ls = list(Measure.iterate(str(tmp_path), max_recent=3))

            # 3件返ってくること
            # Should return 3 folders
            assert len(ls) == 3
            # 結果は昇順
            # Result should be in ascending order
            assert ls[0].path.name == "20250801_120000"
            assert ls[1].path.name == "20250801_130000"
            assert ls[2].path.name == "20250801_140000"

        def test_max_recent_reversed(self, tmp_path: Path):
            # ダミーフォルダ作成
            # Create dummy folders
            self._create_dummy_measures(tmp_path, self._dummy_list)

            # 件数を指定してフォルダを取得
            # Retrieve folders with specified count
            ls = list(Measure.iterate(str(tmp_path), max_recent=3, reverse=True))

            # 3件返ってくること
            # Should return 3 folders
            assert len(ls) == 3
            # 結果は絞り込んだものの降順
            # Result should be in descending order after filtering
            assert ls[0].path.name == "20250801_140000"
            assert ls[1].path.name == "20250801_130000"
            assert ls[2].path.name == "20250801_120000"

        def test_max_recent_without_invalid(self, tmp_path: Path):
            # 不正なものを含めてダミーフォルダ作成
            # Create dummy folders including one with invalid name
            self._create_dummy_measures(tmp_path, self._dummy_list + ["20250801_13NG"])

            # 件数を指定してフォルダを取得
            # Retrieve folders with specified count
            ls = list(Measure.iterate(str(tmp_path), max_recent=3))

            # 3件返ってくること
            # Should return 3 folders
            assert len(ls) == 3
            # NG を含んでいないこと
            # Should not include invalid folder
            assert "20250801_13NG" not in ls
            # 結果は昇順
            # Result should be in ascending order
            assert ls[0].path.name == "20250801_120000"
            assert ls[1].path.name == "20250801_130000"
            assert ls[2].path.name == "20250801_140000"

    class TestSensorKey:
        def test_construct(self):
            key = SensorKey("A342", "00000100")
            assert key.model == "A342"
            assert key.serial == "00000100"

        def test_to_str(self):
            key = SensorKey("A342", "00000100")
            assert str(key) == "A342_00000100"

        def test_equals(self):
            key1 = SensorKey("A342", "00000100")
            key2 = SensorKey("A342", "00000100")
            assert key1 == key2

    class TestMeasureInfo:
        def test_iterate(self):
            me = list(Measure.iterate(MEASURE_PATH))
            ls = list(MeasureInfo.iterate(me[0]))

            assert len(ls) == 3

            # info 出力は昇順
            # Info output should be in ascending order
            assert ls[0].path.name == "A342_00000100_info.csv"
            assert ls[1].path.name == "A342_00000200_info.csv"
            assert ls[2].path.name == "A352_E0000271_info.csv"

        def test_iterate_not_contain_invalid(self):
            me = list(Measure.iterate(MEASURE_PATH))
            ls = list(MeasureInfo.iterate(me[1]))

            assert len(ls) == 2

            # info 出力は昇順
            # Info output should be in ascending order
            assert ls[0].path.name == "A342_00000100_info.csv"
            assert ls[1].path.name == "A342_00000200_info.csv"
            # "A342_00009999_info_invalid.csv" は含まない
            # "A342_00009999_info_invalid.csv" should not be included

        def test_iterate_with_single_model_pattern(self):
            me = list(Measure.iterate(MEASURE_PATH))
            ls = list(MeasureInfo.iterate(me[0], model_ptn="A342"))

            assert len(ls) == 2

            assert ls[0].path.name == "A342_00000100_info.csv"
            assert ls[1].path.name == "A342_00000200_info.csv"

        def test_iterate_with_dual_model_pattern(self):
            me = list(Measure.iterate(MEASURE_PATH))
            ls = list(MeasureInfo.iterate(me[0], model_ptn="A342, A352"))

            assert len(ls) == 3

            assert ls[0].path.name == "A342_00000100_info.csv"
            assert ls[1].path.name == "A342_00000200_info.csv"
            assert ls[2].path.name == "A352_E0000271_info.csv"

        def test_construct_directly_for_test(self):
            path = Path(MEASURE_PATH, "20250801_091011", "A342_00000100_info.csv")
            assert path.exists()

            info = MeasureInfo(path, "dummy")
            assert info.key.model == "A342"
            assert info.key.serial == "00000100"

        def test_load_and_prop_A342(self):
            path = Path(MEASURE_PATH, "20250801_091011", "A342_00000100_info.csv")
            info = MeasureInfo(path, "dummy")

            assert info.sps == 3000
            assert type(info.sps).__name__ == "int"
            assert info.physical == "Velocity"
            assert info.output_type == "Raw"

        def test_load_and_prop_A352(self):
            path = Path(MEASURE_PATH, "20250801_091011", "A352_E0000271_info.csv")
            assert path.exists()
            info = MeasureInfo(path, "dummy")

            assert info.sps == 1000
            assert info.physical == "Acceleration"
            assert info.output_type == "Raw"

    class TestMeasureData:
        def test_iterate(self):
            me = list(Measure.iterate(MEASURE_PATH))
            cnt = 0
            order_tested = False
            for info in MeasureInfo.iterate(me[0]):
                ls = []
                for data in MeasureData.iterate(me[0], info.key):
                    ls.append(data.path.name)
                    cnt += 1
                    assert data.path.name.startswith(
                        f"{info.key.model}_{info.key.serial}"
                    )
                # 昇順であること
                # Should be in ascending order
                if len(ls) > 1:
                    assert ls == sorted(ls)
                    order_tested = True

            assert cnt == 4
            assert order_tested

        def test_iterate_reverse(self):
            me = list(Measure.iterate(MEASURE_PATH))
            cnt = 0
            order_tested = False
            for info in MeasureInfo.iterate(me[0]):
                ls = []
                for data in MeasureData.iterate(me[0], info.key, reverse=True):
                    ls.append(data.path.name)
                    cnt += 1
                    assert data.path.name.startswith(
                        f"{info.key.model}_{info.key.serial}"
                    )
                # 降順であること
                # Should be in descending order
                if len(ls) > 1:
                    assert ls == sorted(ls, reverse=True)
                    order_tested = True

            assert cnt == 4
            assert order_tested

        def test_iterate_should_not_contain_invalid(self):
            me = list(Measure.iterate(MEASURE_PATH))
            ls = []
            for info in MeasureInfo.iterate(me[1]):
                for data in MeasureData.iterate(me[1], info.key):
                    ls.append(data.path.name)
                    assert data.path.name.startswith(
                        f"{info.key.model}_{info.key.serial}"
                    )

            # 結果は3件ある
            # Should contain 3 valid entries
            assert len(ls) == 3

            # こちらのファイル名は含まない
            # This file name should not be included
            assert "A342_00000100_R4_1_invalid.csv" not in ls

            # FIXME: こちらのファイル名はパターンマッチの都合上含んでしまう
            # This file name is included due to pattern matching
            assert "A342_00000200_R4_1_250801_101113_invalid.csv" in ls
