# Copyright 2024, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.


class Constants:
    """
    アプリケーションで使用する定数を定義したクラス

    Class defining constants used in the application
    """

    LOGGER_VERSION = "1.3.0"

    PHYSICAL_VELOCITY = "Velocity"
    PHYSICAL_DISPLACEMENT = "Displacement"
    PHYSICAL_ACCELERATION = "Acceleration"
    START_TYPE_AUTO = "auto"
    START_TYPE_MANUAL = "manual"
    MODE_RAW = "Raw"
    MODE_RMS = "RMS"
    MODE_PP = "P-P"
    SPS_3000 = 3000
    SPS_1000 = 1000
    SPS_500 = 500
    SPS_300 = 300
    SPS_200 = 200
    SPS_100 = 100
    SPS_50 = 50
    BAUD_RATE_230400 = 230_400
    BAUD_RATE_460800 = 460_800
    BAUD_RATE_921600 = 921_600
    LOGGER_ID_PATTERN = r"^[0-9a-zA-Z]{1,63}$"
    INITIAL_WAIT_RANGE = (0, 3600)
    MEASURE_TIME_SEC_RANGE = (0, 2**63 - 1)
    FILE_ROTATE_MIN_RANGE = (1, 60)
    BAUD_RATES = [BAUD_RATE_230400, BAUD_RATE_460800, BAUD_RATE_921600]
    A342_PHYSICALS = [PHYSICAL_VELOCITY, PHYSICAL_DISPLACEMENT]
    A342_MODES = [MODE_RAW, MODE_RMS, MODE_PP]
    A342_RMS_PP_OUTPUT_INTERVAL_RANGE = (1, 255)
    A352_SPSS = [SPS_50, SPS_100, SPS_200, SPS_500, SPS_1000]
    A370_SPSS = [SPS_50, SPS_100, SPS_200, SPS_500, SPS_1000]
    MESSAGE_PORT_RANGE = (0, 65535)
    LOG_LEVELS = ["DEBUG", "INFO", "WARN", "ERROR", "CRITICAL"]
    LOG_LEVEL_DEFAULT = LOG_LEVELS[1]

    SEC_IN_MIN = 60
    TEST_OK = "OK"
    TEST_NG = "NG"
    TEST_NOT_AVAILABLE = "Not Available"


class ConfigKeys:
    """
    設定ファイルで使用するキーを定義したクラス

    Class defining keys used in the configuration file
    """

    LOGGER_ID = "LOGGER_ID"
    OUTPUT_PATH = "OUTPUT_PATH"
    INITIAL_WAIT = "INITIAL_WAIT"
    MEASURE_TIME_SEC = "MEASURE_TIME_SEC"
    FILE_ROTATE_MIN = "FILE_ROTATE_MIN"
    BAUD_RATE = "BAUD_RATE"
    A342_PHYSICAL = "A342_PHYSICAL"
    A342_MODE = "A342_MODE"
    A342_RMSPP_OUTPUT_INTERVAL = "A342_RMSPP_OUTPUT_INTERVAL"
    A352_SPS = "A352_SPS"
    A370_SPS = "A370_SPS"
    SENSOR_DATA_DIAG = "SENSOR_DATA_DIAG"
    SENSOR_DATA_DIAG_SEC = "SENSOR_DATA_DIAG_SEC"
    MESSAGE_SEND = "MESSAGE_SEND"
    MESSAGE_HOST = "MESSAGE_HOST"
    MESSAGE_PORT = "MESSAGE_PORT"
    LOG_LEVEL = "LOG_LEVEL"


class InfoKeys:
    """
    計測情報ファイルのキーを定義したクラス

    Class defining keys used in the measurement information file
    """

    OS = "OS"
    LOGGER_VERSION = "LOGGER_VERSION"
    LOGGER_ID = "LOGGER_ID"
    START_TYPE = "START_TYPE"
    TIME_ZONE = "TIME_ZONE"
    DATE = "DATE"
    MEASURE_TIME_SEC = ConfigKeys.MEASURE_TIME_SEC
    INITIAL_WAIT = ConfigKeys.INITIAL_WAIT
    FILE_ROTATE_MIN = ConfigKeys.FILE_ROTATE_MIN
    BAUD_RATE = ConfigKeys.BAUD_RATE
    PORT = "PORT"
    SENSOR = "SENSOR"
    PRODUCT_ID = "PRODUCT_ID"
    SERIAL_NO = "SERIAL_NO"
    FIRM_VER = "FIRM_VER"
    PHYSICAL = "PHYSICAL"
    OUTPUT_TYPE = "OUTPUT_TYPE"
    SPS = "SPS"
    FILTER = "FILTER"
    RMS_PP_INTERVAL = "RMS_PP_INTERVAL"
    SENSOR_DATA_DIAG = ConfigKeys.SENSOR_DATA_DIAG
    SENSOR_DATA_DIAG_SEC = ConfigKeys.SENSOR_DATA_DIAG_SEC
    MESSAGE_SEND = ConfigKeys.MESSAGE_SEND
    MESSAGE_HOST = ConfigKeys.MESSAGE_HOST
    MESSAGE_PORT = ConfigKeys.MESSAGE_PORT


class ExitCode:
    """
    システムの終了コードを定義したクラス

    Class defining system exit codes
    """

    EX_OK: int = 0
    EX_NG: int = 1
