# Copyright 2024, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

import datetime
import logging
import logging.handlers
import os
import shutil
import sys
import traceback
from multiprocessing import Queue
from typing import Any, Type, cast

from logger import ConfigKeys, Constants
from logger.utils.env import get_str

from .service import ServiceBase

# ログレベル： Config を経由せず設定値から直接取得
# Log level: Directly obtain from configuration values without using Config
_LOG_LEVEL: str = str(
    get_str(ConfigKeys.LOG_LEVEL)
    if get_str(ConfigKeys.LOG_LEVEL) in Constants.LOG_LEVELS
    else Constants.LOG_LEVEL_DEFAULT
)


class MeasureLoggingService(ServiceBase):
    """
    センサーデータ計測におけるロギングを提供するサービス

    マルチプロセス環境でのロギングに対応し、サーバーによるロギング構築と出力、
    クライアントによるデータ転送を行う。
    シングルプロセス環境での動作は現状対応していない。

    Service providing logging for sensor data measurement

    Supports logging in a multi-process environment, constructing and outputting logs by the server,
    and transferring data by the client.
    Operation in a single-process environment is not currently supported.
    """

    class Server(ServiceBase.Server):
        """
        MeasureLoggingService のサーバークラス

        Server class for MeasureLoggingService
        """

        def __init__(self) -> None:
            self._log_file: str | None = None

        def start(self) -> None:
            """
            サービスを開始するメソッド

            一時ログファイルの解決、ロギングの構築を行う

            Method to start the service

            Resolves temporary log files and sets up logging
            """
            # 出力ファイル： 計測終了までは一時ファイルに出力する
            # Output file: Output to a temporary file until the measurement is complete
            self._log_file = "{}/_logger_tmp_{}.log".format(
                os.getcwd(), format(datetime.datetime.now(), "%Y%m%d_%H%M%S")
            )

            # ロギング構築
            # Set up logging
            root = logging.getLogger()
            root.setLevel(_LOG_LEVEL)

            # - StreamHandler 構築
            # - Set up StreamHandler
            stream_handler = logging.StreamHandler(sys.stdout)
            stream_handler.setFormatter(_Formatter())
            root.addHandler(stream_handler)

            # - FileHandler 構築
            # - Set up FileHandler
            try:
                file_handler = logging.FileHandler(self._log_file)
                file_handler.setFormatter(_Formatter())
                root.addHandler(file_handler)
            except Exception:
                # 出力先により例外が発生する可能性がある - Stream のみで継続
                # Exceptions may occur depending on the output destination - continue with Stream only
                root.error("An error occured while initializing logging")
                # - logging.exception() でトレースが出力されないケースがあるため明示的に出力
                # - Explicitly output as traceback may not be output with logging.exception()
                traceback.print_exc(file=sys.stderr)

        def declare_type(self) -> list[Type]:
            """
            このサーバーが処理できるデータ型を宣言するメソッド

            Method to declare the data types this server can process
            """
            return [logging.LogRecord]

        def handle_data(self, data: Any) -> None:
            """
            データに対する処理を実行するメソッド

            Method to execute processing on the data
            """
            record = cast(logging.LogRecord, data)
            logger = logging.getLogger(record.name)
            logger.handle(record)

        def close(self, **kwargs) -> None:
            """
            サービスを終了するメソッド

            終了時は、一時ファイルとして作成していたログファイルを、正規のパスに変更する。

            Method to terminate the service

            Upon termination, change the log file created as a temporary file to the proper path.

            Args:
                kwargs (dict): 名前付き可変引数を格納した dict
                        Dictionary containing named variable arguments
                    - "output_dir" の文字列があると、その配下にログファイルを移動する
                        If the string "output_dir" is present, move the log file under that directory
            """
            output_dir = kwargs.get("output_dir")
            if output_dir and self._log_file and os.path.isdir(output_dir):
                shutil.move(self._log_file, f"{output_dir}/measure.log")
                self._log_file = None

    class Client(ServiceBase.Client):
        """
        MeasureLoggingService のクライアントクラス

        Client class for MeasureLoggingService
        """

        def setup(self, **kwargs) -> None:
            """
            サービスを利用するクライアントプロセスを設定するメソッド

            引数より Queue を取得し、Queue を通じてサーバーに転送するロギングを構築する

            Method to set up the client process using the service

            Retrieves the Queue from the arguments and sets up logging to transfer to the server via the Queue

            Args:
                kwargs (dict): 名前付き可変引数を格納した dict
                    Dictionary containing named variable arguments
                    - "queue" としてサーバー転送用 Queue を必要とする
                      Requires a Queue for server transfer as "queue"
            """
            # Queue を取得する
            # Retrieve the Queue
            queue: Queue = cast(Queue, kwargs.get("queue"))
            assert queue

            # ロギング構築
            # Set up logging
            qh = logging.handlers.QueueHandler(queue)
            root = logging.getLogger()
            root.addHandler(qh)
            root.setLevel(_LOG_LEVEL)

    def create_server(self) -> ServiceBase.Server:
        """
        Server オブジェクトを生成するメソッド

        Method to create a Server object
        """
        return MeasureLoggingService.Server()

    def create_client(self) -> ServiceBase.Client:
        """
        Client オブジェクトを生成するメソッド

        Method to create a Client object
        """
        return MeasureLoggingService.Client()


class StdoutLoggingService(ServiceBase):
    """
    標準出力のみに出力するロギングサービス

    - 簡易なログを使いたいツール向け
    - シングルプロセス環境での利用を想定するが、マルチプロセス環境でも結果的に動作可能

    Logging service that outputs only to standard output

    - For tools that want to use simple logs
    - Intended for use in a single-process environment, but can also work in a multi-process environment
    """

    class Server(ServiceBase.Server):
        """
        StdoutLoggingService のサーバークラス: 実装はありません

        Server class for StdoutLoggingService: No implementation
        """

        def start(self) -> None:
            pass

        def declare_type(self) -> list[Type]:
            return []

        def handle_data(self, data: Any) -> None:
            pass

        def close(self, **kwargs) -> None:
            pass

    class Client(ServiceBase.Client):
        """
        StdoutLoggingService のクライアントクラス

        Client class for StdoutLoggingService
        """

        def setup(self, **kwargs) -> None:
            """
            ロギングをセットアップします

            Set up logging
            """
            handler = logging.StreamHandler(sys.stdout)
            handler.setFormatter(_Formatter())
            level: str = str(kwargs.get("level")) if "level" in kwargs else _LOG_LEVEL
            logging.basicConfig(handlers=[handler], level=level)

    def create_server(self) -> ServiceBase.Server:
        return StdoutLoggingService.Server()

    def create_client(self) -> ServiceBase.Client:
        return StdoutLoggingService.Client()


class _Formatter(logging.Formatter):
    """
    ログ出力のフォーマッター

    Formatter for log output
    """

    def format(self, rec: logging.LogRecord) -> str:
        return "%-8s|%s %s: %s" % (
            rec.levelname,
            self.formatTime(rec),
            rec.name,
            rec.getMessage(),
        )
