# Copyright 2024, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

from typing import Any, Type


class ServiceBase:
    """
    ロガー上で固有の機能を提供する「サービス」のベースクラス

    - マルチプロセス環境で Queue を用いてデータを転送し、処理を行う仕組みへの適用を想定している
    - シングルプロセス環境でも同じサービスを稼働させることを想定

    具象サービスクラスで、各クラスやメソッドをr実装すること。

    Base class for "services" providing specific functionalities on the logger

    - Intended for use in a multi-process environment to transfer and process data using a Queue
    - Also intended to run the same services in a single-process environment

    Implement each class and method in the concrete service class.
    """

    class Server:
        """
        サービスを提供するサーバークラス

        - マルチプロセス環境で、サーバープロセス上で Queue から取得したデータを処理する機能を提供する
        - シングルプロセス環境では、そのプロセス上で処理を提供する

        Server class providing the service

        - Provides functionality to process data retrieved from the Queue on the server process
          in a multi-process environment
        - Provides processing on the same process in a single-process environment
        """

        def start(self) -> None:
            """
            サービスを開始するメソッド

            マルチプロセス環境では、サーバープロセス上で start が呼び出されるため、
            サービスに必要な初期化処理を適宜実行する。

            Method to start the service

            In a multi-process environment, start is called on the server process,
            so perform any necessary initialization for the service.
            """
            raise NotImplementedError

        def declare_type(self) -> list[Type]:
            """
            このサーバーが処理できるデータ型を宣言するメソッド

            このサーバーが処理できる型（複数）を宣言することで、
            サーバープロセス上で Queue に転送されるデータの型に応じてサーバーがディスパッチされる。

            Method to declare the data types this server can handle

            By declaring the types this server can handle, the server is dispatched
            based on the data types transferred to the Queue on the server process.

            Returns:
                list[Type]: 処理できる型を格納したリスト
                        List of types that can be processed
            """
            raise NotImplementedError

        def handle_data(self, data: Any) -> None:
            """
            データに対する処理を実行するメソッド

            Method to process the data

            Args:
                data (Any): 処理対象のデータ
                    Data to be processed
                    - マルチプロセス環境では Queue から取得される
                      Retrieved from the Queue in a multi-process environment
            """
            raise NotImplementedError

        def close(self, **kwargs) -> None:
            """
            サービスを終了するメソッド

            マルチプロセス環境では、サーバープロセス上で close が呼び出されるため、
            サービスに必要な終了処理を適宜実行する。

            Method to stop the service

            In a multi-process environment, close is called on the server process,
            so perform any necessary termination procedures for the service.

            Args:
                kwargs (dict): 名前付き可変引数を格納する dict
                    Dict containing named variable arguments
                    - ServiceFrame.close() に渡された名前付き引数が転送され呼び出される
                      Named arguments passed to ServiceFrame.close() are transferred and called
                    - 引数名を合わせることで、外部からパラメータを受け渡すことが可能
                      By matching argument names, parameters can be passed from external sources
            """
            raise NotImplementedError

    class Client:
        """
        サービスを利用するクライアントクラス

        - マルチプロセス環境で、各クライアントプロセス上でサーバーにアクセスするための手段を設定する
        - シングルプロセス環境では、サーバーを参照し処理するための手段を設定する

        Client class to use the service

        - Sets up means to access the server on each client process in a multi-process environment
        - Sets up means to reference and process the server in a single-process environment
        """
        def setup(self, **kwargs) -> None:
            """
            サービスを利用するクライアントプロセスを設定するメソッド

            使用される環境に応じて引数が異なるため、それに応じて設定を行う

            Method to set up the client process using the service

            Configure according to the environment used, as arguments vary

            Args:
                kwargs (dict): 名前付き可変引数を格納する dict
                    Dict containing named variable arguments
                    - マルチプロセス環境では "queue" として、サーバープロセスに送信する Queue が渡される
                      In a multi-process environment, "queue" is passed as the Queue to send to the server process
                    - シングルプロセス環境では "server" として、サーバーそのものが渡される
                      In a single-process environment, "server" is passed as the server itself
            """
            raise NotImplementedError

    def create_server(self) -> Server:
        """
        Server オブジェクトを生成するメソッド

        Method to create a Server object
        """
        raise NotImplementedError

    def create_client(self) -> Client:
        """
        Client オブジェクトを生成するメソッド

        Method to create a Client object
        """
        raise NotImplementedError


class ServiceFrame:
    """
    「サービス」を動作させる基盤を提供する Singleton クラス

    - マルチプロセス環境では、サーバープロセスを起動し、登録されたサービスを開始・実行・終了する
    - シングルプロセス環境では、登録されたサービスを開始・実行・終了する

    このクラスではシングルプロセス環境でのサービス基盤を提供する

    Singleton class providing the foundation to run "services"

    - In a multi-process environment, starts, runs, and stops registered services by launching server processes
    - In a single-process environment, starts, runs, and stops registered services

    This class provides the service foundation in a single-process environment
    """

    # シングルトンインスタンス
    # Singleton instance
    _instance: "ServiceFrame | None" = None

    # シングルトンを成立させるため、通常のインスタンス生成を抑止
    # Prevent normal instance creation to enforce Singleton pattern
    def __new__(cls):
        raise NotImplementedError("Cannot instantiate through constructor")

    @classmethod
    def __internal_new__(cls):
        instance = super().__new__(cls)
        instance.__init__()  # type: ignore
        return instance

    @classmethod
    def get_instance(cls) -> "ServiceFrame":
        """
        シングルトンの ServiceFrame インスタンスを取得するメソッド

        Method to get the Singleton instance of ServiceFrame

        Returns:
            ServiceFrame
        """
        if not cls._instance:
            cls._instance = cls.__internal_new__()
        assert isinstance(cls._instance, ServiceFrame)
        return cls._instance

    def __init__(self):
        self._started: bool = False
        self._services: list[ServiceBase] = []
        self._servers: list[ServiceBase.Server] = []

    def add_service(self, service: ServiceBase) -> None:
        """
        サービスを登録するメソッド

        - サービス基盤が開始前の場合、開始時にサービスが起動される
        - サービス基盤が開始済みの場合、登録と同時にサービスが起動される

        Method to register a service

        - If the service foundation has not started, the service will start upon foundation start
        - If the service foundation has started, the service will start upon registration

        Args:
            service (ServiceBase): サービス基盤に登録するサービス
                    Service to be registered with the service foundation
        """
        # サービスを登録
        # Register the service
        self._services.append(service)
        server = service.create_server()
        self._servers.append(server)

        # 登録後の処理を実行 - 基盤の種類に応じて処理が異なる
        # Execute post-registration processing - varies based on foundation type
        self._post_add(service, server)

    def _post_add(self, service: ServiceBase, server: ServiceBase.Server) -> None:
        """
        サービス登録後の処理を実行するテンプレートメソッド
        （サービス基盤の実装種類に応じて定義する）

        Template method to execute post-service registration processing
        (Defined according to the type of service foundation implementation)

        Args:
            service (ServiceBase): 登録したサービス
                    Registered service
            server (ServiceBase.Server): 登録したサービスから生成したサーバー
                    Server generated from the registered service
        """
        if self._started:
            self._start_service(service, server)

    def start(self) -> None:
        """
        サービス基盤を開始するメソッド

        このクラスの実装では、現在のプロセス上でサービスを起動する。

        Method to start the service foundation

        In this class implementation, starts the services on the current process.
        """
        self._started = True
        for service, server in zip(self._services, self._servers, strict=True):
            self._start_service(service, server)

    @staticmethod
    def _start_service(service: ServiceBase, server: ServiceBase.Server) -> None:
        """
        サービスの起動を行う関数

        このクラスでのみ使われる実装の想定で、staticmethod としている。
        シングルプロセス構成では、サーバーの起動とクライアントのセットアップを同時に行う。

        Function to start the service

        Intended for use only in this class, hence defined as staticmethod.
        In a single-process configuration, starts the server and sets up the client simultaneously.
        """
        # サーバー起動
        # Start the server
        server.start()

        # クライアントセットアップ
        # - サーバーと直結させるため、引数で指定する
        # Set up the client
        # - Specify in arguments to directly connect with the server
        client = service.create_client()
        client.setup(server=server)

    def close(self, **kwargs) -> None:
        """
        サービス基盤を終了するメソッド

        このクラスの実装では、現在のプロセス上でサーバーを停止する。

        Method to stop the service foundation

        In this class implementation, stops the servers on the current process.

        Args:
            kwargs (dict): 名前付き可変引数を格納した dict
                Dict containing named variable arguments
                - ここで渡された名前付き引数は、そのままサーバーの停止に引き渡される
                  Named arguments passed here are directly passed to stop the servers
                - 名前を合わせておくことで、個別のサーバーに引数を渡すことができる
                  By matching names, arguments can be passed to individual servers
        """
        # 逆順でサーバーを停止する
        # Stop the servers in reverse order
        for server in self._servers[::-1]:
            server.close(**kwargs)
        self._started = False
