# Copyright 2024, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

import logging
import os
import signal
import sys
import traceback

from logger.core import (
    ConfigOverrideOption,
    MeasureLoggingService,
    MessageService,
    MultiProcessServiceFrame,
    TerminateException,
    Topic,
    raise_terminate_exception,
)
from logger.core.factory import LoggerFactoryImpl
from logger.measure import Controller
from logger.utils.env import parse_int
from logger.utils.lock import ExclusiveLock, LockError

from .constants import Constants, ExitCode
from .factory import LoggerFactory


def main(factory: LoggerFactory | None = None):  # noqa: C901
    """
    ロガープログラムのメイン関数。

    パッケージのメインモジュール（`__main__.py`）から呼び出す。
    ロガー内部の主要オブジェクトを生成するファクトリを引数に受け取ることができる。

    ロガープログラムを利用するアプリケーションにおいて、
    ファクトリを指定して `main()` を実行可能にすることを想定している。

    Main function of the logger program.

    Called from the main module (`__main__.py`) of the package.
    Can take a factory that generates major objects within the logger as an argument.

    In applications using the logger program,
    it is assumed that `main()` can be executed by specifying the factory.

    Args:
        factory (Optional[LoggerFactory]):
            ロガー内部の主要オブジェクトを生成するファクトリ
            Factory that generates major objects within the logger
            - 未指定時はデフォルトの LoggerFactory が使用される
              If not specified, the default LoggerFactory is used.
    """

    # プロセス初期化
    # - シグナルハンドラー登録 : SIGTERM, SIGINT 発生時は例外処理にて対処する
    # Process initialization
    # - Register signal handlers: handle exceptions on SIGTERM, SIGINT
    signal.signal(signal.SIGTERM, raise_terminate_exception)
    signal.signal(signal.SIGINT, raise_terminate_exception)

    # - マルチプロセス機構構築
    # - Build multi-process mechanism
    svc_frame = MultiProcessServiceFrame.get_instance()

    # - ロギングサービス登録
    # - Register logging service
    svc_frame.add_service(MeasureLoggingService())

    # - 排他ロック生成
    # - Create exclusive lock
    lock = ExclusiveLock("/tmp/measure.lock")

    # メイン処理開始
    # Start main process
    controller: Controller | None = None
    logger: logging.Logger | None = None
    try:
        # マルチプロセス機構開始
        # - ロギングも開始される
        # Start multi-process mechanism
        # - Logging also starts
        svc_frame.start()
        logger = logging.getLogger(__name__)

        # - プロセスIDを出力
        # - Output process ID
        logger.debug(f"process id: {os.getpid()}")

        # 多重起動防止
        # Prevent multiple instances
        try:
            lock.acquire()
        except LockError:
            logger.error("Another measurement is running, cannot start measurement")
            return ExitCode.EX_NG

        # 引数処理
        # Argument processing
        match len(sys.argv):
            case 1:
                start_type = Constants.START_TYPE_AUTO
            case 2:
                if sys.argv[1] == Constants.START_TYPE_AUTO:
                    start_type = Constants.START_TYPE_AUTO
                else:
                    start_type = Constants.START_TYPE_MANUAL
            case _:
                logger.error(
                    "Invalid argument length; "
                    + 'Argument must be "auto" or an integer.'
                )
                return ExitCode.EX_NG

        # - "auto"
        if start_type == Constants.START_TYPE_AUTO:
            meas_time_sec_arg = None
        # - integer
        else:
            meas_time_sec_arg = parse_int(sys.argv[1])
            if meas_time_sec_arg is None:
                logger.error("Measurement time argument must be an integer")
                return ExitCode.EX_NG

        # ファクトリ初期化
        # Initialize factory
        if factory is None:
            factory = LoggerFactoryImpl()

        # 設定オブジェクト取得
        # Get configuration object
        config = factory.create_configurator().get_config(
            ConfigOverrideOption(measure_time_sec=meas_time_sec_arg)
        )
        if config is None:
            logger.error("Config file has wrong value, cannot start measurement")
            return ExitCode.EX_NG

        # メッセージサービス登録
        # Register message service
        svc_frame.add_service(MessageService(config=config, root_elem="logger"))

        # - STOP メッセージ送信
        # - Send STOP message
        MessageService.send(Topic.logger(), "STOP")

        # 計測コントローラ生成
        # Create measurement controller
        controller = Controller(config=config, start_type=start_type, factory=factory)
        try:
            # 計測初期化
            # Initialize measurement
            controller.init()

        except TerminateException:
            # シグナルにより中断された場合、正常終了とする
            # If interrupted by a signal, exit normally
            return ExitCode.EX_OK

        except Exception as e:
            m = "Failed to initialize measurement"
            logger.exception(m)
            MessageService.send(Topic.logger_error(), m, exc=e)
            logger.error("Cannot start measurement")
            return ExitCode.EX_NG

        try:
            # 計測開始
            # - START メッセージ送信
            # Start measurement
            # - Send START message
            MessageService.send(Topic.logger(), "START")

            # - コントローラ開始指示
            # - Instruct controller to start
            controller.start()

        except TerminateException:
            # シグナルにより中断された場合、正常終了とする
            # - この後の finally ブロックで計測終了が処理される
            # - 各所でログが出力されるため、ここではログ出力しない
            # If interrupted by a signal, exit normally
            # - Measurement termination will be handled in the finally block
            # - Logs will be output in various places, so no log output here
            return ExitCode.EX_OK

        except Exception as e:
            m = "Failed to measure"
            logger.exception(m)
            MessageService.send(Topic.logger_error(), m, exc=e)
            logger.error("Cannot continue measurement")
            return ExitCode.EX_NG

        # 計測成功
        # Measurement successful
        return ExitCode.EX_OK

    except TerminateException:
        # 上の try ブロック外でのシグナル中断も正常終了とする
        # Signal interruption outside the above try block also exits normally
        return ExitCode.EX_OK

    except Exception as e:
        m = "Unknown error occurred"
        if logger:
            logger.exception(m)
        else:
            print(f"{m}:\n{traceback.format_exc()}")
        MessageService.send(Topic.logger_error(), m, exc=e)
        return ExitCode.EX_NG

    finally:
        # 計測終了〜プロセス終了
        # - 計測コントローラ停止
        # Measurement termination to process termination
        # - Stop measurement controller
        output_dir: str | None = None
        if controller:
            output_dir = controller.get_output_dir()
            # 計測停止指示: terminate=True として停止を指示する
            # - タイマー実行で計測終了している場合は、停止指示は無視される
            # Instruct to stop measurement: stop with terminate=True
            # - If measurement has ended with a timer, the stop instruction is ignored
            controller.stop(terminate=True)

        # - STOP メッセージ送信
        # - Send STOP message
        MessageService.send(Topic.logger(), "STOP")

        # - マルチプロセス機構停止 - ロギングも停止される
        # - Stop multi-process mechanism - logging also stops
        svc_frame.close(output_dir=output_dir)

        # - 排他ロック解放
        # - Release exclusive lock
        lock.release()
