# Copyright 2024, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

import math
import time
from dataclasses import dataclass
from typing import Any, TypedDict

from logger import Constants, InfoKeys
from logger.utils.convert import to_dec32, to_int32, to_uint16

from .comm import Comm, WaitCommand
from .data import MeasureData, SelfTestResult
from .job import ReaderArgs, WriterArgs
from .sensor import Sensor


@dataclass
class A352ReaderArgs(ReaderArgs):
    """
    A352 の Reader 情報を保持したオブジェクト

    Object holding the Reader information for A352
    """

    def format_packet(self, index: int, packet: list[int]) -> MeasureData:
        """
        1レコード分のデータを受け取って計測データに変換するメソッド

        パケット構成：
        Packet structure:
        - 0: 0x80
        - 1-2: FLAG
        - 3-6: TEMP
        - 7-10: XACCL, 11-14: YACCL, 15-18: ZACCL
        - 19-20: COUNT
        - 21: 0x0D

        Args:
            index (int): 何番目のデータか
                    Which data index
            packet (list[int]): 変換対象のデータ配列
                    Data array to be converted

        Returns:
            MeasureData: 変換後のデータオブジェクト
                    Converted data object
        """
        # flag は x, y, z, filter mismatch, hard errorの5桁
        # flag consists of 5 bits: x, y, z, filter mismatch, hard error
        flag = 0b00011111 & packet[2]
        temperature = to_int32(packet[3:7]) * -0.0037918 + 34.987
        x = to_dec32(packet[7:11])
        y = to_dec32(packet[11:15])
        z = to_dec32(packet[15:19])
        count = to_uint16(packet[19], packet[20])

        data = MeasureData(
            index=index, temperature=temperature, x=x, y=y, z=z, count=count, flag=flag
        )

        return data

    def _gen_invalid_data(self, index: int, count: int) -> MeasureData:
        """
        データ抜けがあった際に不正な値であることを示すデータクラスオブジェクトを生成するメソッド

        Method to generate a data class object indicating invalid values when data is missing

        Args:
            index (int): 不正な計測データのindex
                    Index of the invalid measurement data
            count (int): 不正な計測データのカウント値、complement_dataの中で前のデータから算出する
                    Count value of the invalid measurement data,
                    calculated from the previous data in complement_data

        Returns:
            MeasureData: 補完したindexとcountを持つ不正なことを表すデータオブジェクト
                    Data object indicating invalidity with the complemented index and count
        """
        invalid_value = to_dec32([0x7F, 0xFF, 0xFF, 0xFF])
        return MeasureData(
            index=index,
            count=count,
            temperature=to_int32([0x7F, 0xFF, 0xFF, 0xFF]) * -0.0037918 + 34.987,
            x=invalid_value,
            y=invalid_value,
            z=invalid_value,
            flag=0,
        )


class A352(Sensor):
    """
    A352センサークラス

    A352 sensor class
    """

    class _SPSConfig(TypedDict):
        sps_command: int
        filter_command: int
        filter_name: str
        count_diff: int

    _RECORD_SIZE = 22

    _SPS_CONFIGS: dict[int, _SPSConfig] = {
        Constants.SPS_1000: {
            "sps_command": 0b0010,
            "filter_command": 0b1010,
            "filter_name": "FIR Kaiser TAP512 fc=460",
            "count_diff": 4,
        },
        Constants.SPS_500: {
            "sps_command": 0b0011,
            "filter_command": 0b1001,
            "filter_name": "FIR Kaiser TAP512 fc=210",
            "count_diff": 8,
        },
        Constants.SPS_200: {
            "sps_command": 0b0100,
            "filter_command": 0b1000,
            "filter_name": "FIR Kaiser TAP512 fc=60",
            "count_diff": 20,
        },
        Constants.SPS_100: {
            "sps_command": 0b0101,
            "filter_command": 0b0111,
            "filter_name": "FIR Kaiser TAP512 fc=16",
            "count_diff": 40,
        },
        Constants.SPS_50: {
            "sps_command": 0b0110,
            "filter_command": 0b0110,
            "filter_name": "FIR Kaiser TAP512 fc=9",
            "count_diff": 80,
        },
    }

    def __init__(
        self,
        port: str,
        baud: int,
        product_id: str,
        sps: int,
        logger_id: str,
        file_rotate_min: int,
        sensor_data_diag: bool,
        sensor_data_diag_sec: float,
    ) -> None:
        super().__init__(
            model="A352",
            port=port,
            baud=baud,
            product_id=product_id,
            sps=sps,
            logger_id=logger_id,
            file_rotate_min=file_rotate_min,
            mode=Constants.MODE_RAW,
            physical=Constants.PHYSICAL_ACCELERATION,
            sensor_data_diag=sensor_data_diag,
        )
        self.filter = A352._SPS_CONFIGS[sps]["filter_name"]
        self.sps_command = A352._SPS_CONFIGS[self.sps]["sps_command"]
        self.filter_command = A352._SPS_CONFIGS[self.sps]["filter_command"]
        self.count_max = 2**16
        self.count_diff = A352._SPS_CONFIGS[self.sps]["count_diff"]
        self.diag_broken_count = math.ceil(sensor_data_diag_sec * self.sps)

    def init(self) -> None:
        """
        与えられた計測設定に基づきセンサーの計測情報を設定するメソッド

        Method to set the sensor measurement information based on the given measurement settings
        """
        self._logger.info("Set measurement configuration")
        comm = Comm()
        comm.open(self.port, self.baud)
        comm.send(
            [
                [0, 0xFF, 0xFF, 0x0D],  # おまじない 3 回 （センサー復帰） / Magic 3
                [0, 0xFF, 0xFF, 0x0D],
                [0, 0xFF, 0xFF, 0x0D],
                [0, 0xFE, 0x00, 0x0D],  # Window 0
                [0, 0x83, 0x02, 0x0D],  # MODE_CTRL: 1-0:10=ToConfig
                # W: 出力タイプ / Output type
                [0, 0xFE, 0x01, 0x0D],  # Window 1
                [0, 0x81, 0x8E, 0x0D],  # SIG_CTRL(ND): 15:TEMP 11-9:ACCL
                [0, 0x80, 0x00, 0x0D],  # SIG_CTRL: 7-5:XYZ=000:Accel
                WaitCommand(0.3),  # Wait (未指定のためA342に準じる) / Follow A342 as it is unspecified
                # W: SMPL_CTRL
                [0, 0xFE, 0x01, 0x0D],  # Window 1
                [0, 0x85, self.sps_command, 0x0D],  # SMPL_CTRL: 11-8:DOUT_RATE
                # W: FILTER_CTRL
                [0, 0xFE, 0x01, 0x0D],  # Window 1
                [0, 0x86, self.filter_command, 0x0D],  # FILTER_CTRL: 3-0:FILTER_SEL
                # W: UART auto sampling
                [0, 0xFE, 0x01, 0x0D],  # Window 1
                [0, 0x88, 0x01, 0x0D],  # UART_CTRL: 1:AUTO_START=0 0:UART_AUTO=1
                [0, 0x8D, 0xC7, 0x0D],  # BURST_CTRL: 15:FLAG=1 14:TEMP 10-8:XYZ
                [0, 0x8C, 0x02, 0x0D],  # BURST_CTRL: 1:COUNT=1 0:CHKSM=0
            ]
        )

    def add_info(self, info_d: dict[str, Any]) -> None:
        """
        計測情報ファイルへの出力用に設定項目を登録する

        このクラスでは以下の情報項目を登録する：
        - FILTER

        Register configuration items for output to the measurement information file

        This class registers the following information items:
        - FILTER

        Args:
            info_d (dict) 情報登録用辞書
                Dictionary for information registration
                - 登録する値は最終的に文字列化されるため、数値型のまま登録も可能
                    The values to be registered will eventually be converted to strings,
                    so it is possible to register them as numeric types
        """
        # 上位実装を実行
        # Execute the superclass implementation
        super().add_info(info_d)

        # 項目を追加
        # Add item
        info_d[InfoKeys.FILTER] = self.filter

    def to_reader_args(self) -> ReaderArgs:
        """
        Reader プロセスに渡す設定情報に変換するメソッド

        Method to convert to configuration information to be passed to the Reader process

        Returns:
            ReaderArgs: Reader プロセスに渡す設定情報
                    Configuration information to be passed to the Reader process
        """
        # 一回の取得データ数が小さいとデータとびが発生するため、最大4096byteとして1秒間分のデータをまとめて読み込む
        # To prevent data loss when the amount of data acquired at one time is small,
        # read up to 4096 bytes, which is equivalent to one second of data
        read_length = min(A352._RECORD_SIZE * self.sps, Comm.DEFAULT_READ_BYTES)
        return A352ReaderArgs(
            model=self.model,
            serial=self.serial,
            port=self.port,
            baud=self.baud,
            record_size=A352._RECORD_SIZE,
            record_per_sec=self.sps,
            start_command=Sensor.DEFAULT_START_COMMAND,
            end_command=Sensor.DEFAULT_END_COMMAND,
            record_begin=Sensor.DEFAULT_RECORD_BEGIN,
            record_end=Sensor.DEFAULT_RECORD_END,
            count_max=self.count_max,
            count_diff=self.count_diff,
            count_start=self.count_diff,
            read_length=read_length,
            timeout=Comm.DEFAULT_TIMEOUT,
            sensor_data_diag=self.sensor_data_diag,
            diag_broken_count=self.diag_broken_count,
        )

    def to_writer_args(self) -> WriterArgs:
        """Writer プロセスに渡す設定情報に変換するメソッド

        Method to convert to configuration information to be passed to the Writer process

        Returns:
            WriterArgs: Writer プロセスに渡す設定情報
                    Configuration information to be passed to the Writer process
        """
        return WriterArgs(
            model=self.model,
            serial=self.serial,
            logger_id=self.logger_id,
            port=self.port,
            record_per_file=self._get_record_per_file(),
        )

    def self_test(self) -> SelfTestResult:
        """
        センサーのセルフテストを実行する

        Execute self-test of the sensor

        Returns:
            SelfTestResult: セルフテスト結果
                    Result of the self-test
        """
        self._logger.info(f"Start self test for {self}")
        comm = Comm()
        comm.open(self.port, self.baud)

        # セルフテスト実行順序
        # - 加速度、温度、電源電圧
        # - 不揮発性メモリ
        # - X軸感度
        # - Y軸感度
        # - Z軸感度
        # Order of self-test execution
        # - Acceleration, temperature, power voltage
        # - Non-volatile memory
        # - X-axis sensitivity
        # - Y-axis sensitivity
        # - Z-axis sensitivity
        result: list[list[int]] = []
        for command in [0b00000111, 0b00001000, 0b00010000, 0b00100000, 0b01000000]:
            comm.send(
                [
                    [0, 0xFE, 0x01, 0x0D],  # Window 1
                    [0, 0x83, command, 0x0D],  # Execute self-test
                ]
            )
            while True:
                packet = comm.send([[4, 0x02, 0x00, 0x0D]])
                # セルフテスト完了判定
                # Self-test completion check
                if packet[1] & command == 0:
                    break
                else:
                    time.sleep(0.1)

            result.append(
                comm.send(
                    [
                        [0, 0xFE, 0x00, 0x0D],  # Window 0
                        [4, 0x04, 0x00, 0x0D],  # R: DIAG_STAT
                    ]
                )
            )

        # 加速度: bit1
        # 温度: bit9
        # 電源電圧: bit8
        # Acceleration: bit1
        # Temperature: bit9
        # Power voltage: bit8
        acc_result = self._determine_test_bit(result[0][2], 0b00000010)
        temp_result = self._determine_test_bit(result[0][1], 0b00000010)
        vdd_result = self._determine_test_bit(result[0][1], 0b00000001)

        # 不揮発性メモリ: bit2
        # Non-volatile memory: bit2
        mem_result = self._determine_test_bit(result[1][2], 0b00000100)

        # 感度: bit10,11
        # Sensitivity: bit10, 11
        sens_x_result = self._determine_test_bit(result[2][1], 0b00001100)
        sens_y_result = self._determine_test_bit(result[3][1], 0b00001100)
        sens_z_result = self._determine_test_bit(result[4][1], 0b00001100)

        test_result = SelfTestResult(
            model=self.model,
            serial=self.serial,
            acceleration=acc_result,
            sensitivity=f"X:{sens_x_result},Y:{sens_y_result},Z:{sens_z_result}",
            temperature=temp_result,
            power_voltage=vdd_result,
            flash_memory=mem_result,
            structural_resonance=Constants.TEST_NOT_AVAILABLE,
        )
        test_result.output()

        return test_result

    def _get_record_per_file(self) -> int:
        """
        ファイルごとのレコード数を返す

        Returns the number of records per file

        Returns:
            int: ファイルごとのレコード数
                    Number of records per file
        """
        return self.sps * self.file_rotate_min * Constants.SEC_IN_MIN
