# Copyright 2024, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

from dataclasses import dataclass


@dataclass
class MeasureData:
    """
    計測データを個々の属性に展開したデータクラス

    Data class that expands measurement data into individual attributes
    """

    index: int
    count: int
    temperature: float
    x: float
    y: float
    z: float
    flag: int

    def to_csv_line(self) -> str:
        """
        自身をcsvファイルの1行に変換するメソッド

        Method to convert itself into a line of a CSV file

        Returns:
            str: csvファイルの1行、floatの値は小数点以下9桁に四捨五入される
                    A line of a CSV file. Float values are rounded to 9 decimal places.
        """

        # flagの値は0bを頭につけた2進数5桁で出力する
        # The value of flag is output as a 5-digit binary number with 0b at the beginning
        return f"{self.index},{self.count},{self.temperature:.9f},{self.x:.9f},{self.y:.9f},{self.z:.9f},{self.flag:#010b}\n"


@dataclass
class SelfTestResult:
    """
    セルフテストの結果を保持するクラス

    Class to hold the results of a self-test
    """

    model: str
    serial: str
    acceleration: str
    sensitivity: str
    temperature: str
    power_voltage: str
    flash_memory: str
    structural_resonance: str

    def output(self) -> None:
        txt = (
            f"Model: {self.model}\n"
            + f"SerialNo.: {self.serial}\n"
            + f"Acceleration: {self.acceleration}\n"
            + f"Sensitivity: {self.sensitivity}\n"
            + f"Temperature: {self.temperature}\n"
            + f"Power Voltage: {self.power_voltage}\n"
            + f"Flash Memory: {self.flash_memory}\n"
            + f"Structural Resonance: {self.structural_resonance}\n"
        )
        print(txt)
