# Copyright 2024, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

import logging
import sys
import traceback

from logger import ExitCode
from logger.core import ServiceFrame, StdoutLoggingService
from logger.core.factory import LoggerFactoryImpl
from logger.measure import Controller


def main():
    """
    メイン関数 for selftest

    Main function for selftest
    """

    # ロギング開始
    # Start logging
    svc_frame = ServiceFrame.get_instance()
    svc_frame.add_service(StdoutLoggingService())
    svc_frame.start()

    logger = logging.getLogger(__name__)
    logger.info("Start self test")

    try:
        factory = LoggerFactoryImpl()
        config = factory.create_configurator().get_config()

        if config is None:
            logger.error("Config file has wrong value, cannot start self test")
            return ExitCode.EX_NG

        controller = Controller(config=config, start_type="", factory=factory)

        controller.self_test()

        return ExitCode.EX_OK

    except Exception:
        # ** exception() でトレースが含まれないことがあったため、明示的に実行
        # ** Explicitly execute because there were cases where exception() did not include the trace
        # logger.exception("Failed self test")
        logger.error(f"Failed self test\n{traceback.format_exc()}")
        return ExitCode.EX_NG

    finally:
        svc_frame.close()


if __name__ == "__main__":
    code = main()
    sys.exit(code)
