# Copyright 2024, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

import fcntl


class ExclusiveLock:
    """
    プロセスを跨いでも複数の処理が同時実行させないように排他制御をかけるクラス

    Class to apply exclusive control to prevent multiple processes from running simultaneously across processes
    """

    def __init__(self, lockfile: str) -> None:
        self._lockfile = open(lockfile, mode="w")

    def acquire(self) -> None:
        """
        排他制御を要求するメソッド

        Method to request exclusive control

        Raises:
            LockError: 排他制御が獲得できなかった場合のエラー
                    Error if exclusive control could not be acquired
        """
        self._reopen()
        try:
            fcntl.flock(self._lockfile.fileno(), fcntl.LOCK_EX | fcntl.LOCK_NB)
        except IOError:
            raise LockError from None

    def release(self) -> None:
        """
        排他制御を解除するメソッド

        Method to release exclusive control
        """
        self._reopen()
        try:
            fcntl.flock(self._lockfile.fileno(), fcntl.LOCK_UN)
            self._lockfile.close()
        except Exception:
            pass

    def _reopen(self) -> None:
        """
        内部的に排他制御に利用しているファイルがreleaseで閉じられた後もエラーにならないよう再度開くメソッド

        Method to reopen the file used for exclusive control internally to avoid errors
        after it is closed by release
        """
        if self._lockfile.closed:
            self._lockfile = open(self._lockfile.name, mode="w")


class LockError(Exception):
    """
    排他制御の獲得に失敗した際に上がるエラー

    Error raised when exclusive control acquisition fails
    """

    pass
