# Copyright 2024, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

import math

import pytest

from logger import Constants
from logger.measure import (
    A342,
    A342ReaderArgs,
    WriterArgs,
)
from logger.measure.comm import Comm, WaitCommand
from logger.measure.sensor import SelfTestResult
from tests.measure.test_A352 import TestA352


class TestA342:
    class TestNew:
        @pytest.mark.parametrize(
            "physical, mode, baud, expect",
            [
                (
                    Constants.PHYSICAL_VELOCITY,
                    Constants.MODE_RAW,
                    Constants.BAUD_RATE_460800,
                    0b0000000,
                ),
                (
                    Constants.PHYSICAL_VELOCITY,
                    Constants.MODE_RMS,
                    Constants.BAUD_RATE_460800,
                    0b0010000,
                ),
                (
                    Constants.PHYSICAL_VELOCITY,
                    Constants.MODE_PP,
                    Constants.BAUD_RATE_460800,
                    0b0100000,
                ),
                (
                    Constants.PHYSICAL_VELOCITY,
                    Constants.MODE_RAW,
                    Constants.BAUD_RATE_921600,
                    0b0000010,
                ),
                (
                    Constants.PHYSICAL_VELOCITY,
                    Constants.MODE_RMS,
                    Constants.BAUD_RATE_921600,
                    0b0010010,
                ),
                (
                    Constants.PHYSICAL_VELOCITY,
                    Constants.MODE_PP,
                    Constants.BAUD_RATE_921600,
                    0b0100010,
                ),
                (
                    Constants.PHYSICAL_DISPLACEMENT,
                    Constants.MODE_RAW,
                    Constants.BAUD_RATE_460800,
                    0b1000000,
                ),
                (
                    Constants.PHYSICAL_DISPLACEMENT,
                    Constants.MODE_RMS,
                    Constants.BAUD_RATE_460800,
                    0b1010000,
                ),
                (
                    Constants.PHYSICAL_DISPLACEMENT,
                    Constants.MODE_PP,
                    Constants.BAUD_RATE_460800,
                    0b1100000,
                ),
                (
                    Constants.PHYSICAL_DISPLACEMENT,
                    Constants.MODE_RAW,
                    Constants.BAUD_RATE_921600,
                    0b1000010,
                ),
                (
                    Constants.PHYSICAL_DISPLACEMENT,
                    Constants.MODE_RMS,
                    Constants.BAUD_RATE_921600,
                    0b1010010,
                ),
                (
                    Constants.PHYSICAL_DISPLACEMENT,
                    Constants.MODE_PP,
                    Constants.BAUD_RATE_921600,
                    0b1100010,
                ),
            ],
        )
        def test_new_should_define_valid_sig_ctl_command(
            self, physical, mode, baud, expect
        ):
            sensor = A342(
                port="/dev/AAA",
                baud=baud,
                product_id="A352XXX",
                logger_id="hoge",
                file_rotate_min=1,
                mode=mode,
                physical=physical,
                rms_pp_interval=1,
                sensor_data_diag=True,
                sensor_data_diag_sec=0.1,
            )
            assert sensor.sig_ctl_command == expect

        def test_new_should_define_default_smpl_ctl_command(self):
            sensor = A342(
                port="/dev/AAA",
                baud=Constants.BAUD_RATE_460800,
                product_id="A352XXX",
                logger_id="hoge",
                file_rotate_min=1,
                mode=Constants.MODE_RAW,
                physical=Constants.PHYSICAL_VELOCITY,
                rms_pp_interval=1,
                sensor_data_diag=True,
                sensor_data_diag_sec=0.1,
            )
            assert sensor.smpl_ctl_command == (0x0A, 0x07)

        @pytest.mark.parametrize(
            "rms_pp_interval, expect",
            [
                (1, (1, 4)),
                (2, (2, 5)),
                (3, (3, 5)),
                (4, (4, 6)),
                (6, (6, 6)),
                (7, (7, 7)),
                (13, (13, 7)),
                (14, (14, 8)),
                (27, (27, 8)),
                (28, (28, 9)),
                (54, (54, 9)),
                (55, (55, 10)),
                (109, (109, 10)),
                (110, (110, 11)),
                (218, (218, 11)),
                (219, (219, 12)),
                (255, (255, 12)),
            ],
        )
        def test_new_should_define_valid_smpl_ctl_command(
            self, rms_pp_interval, expect
        ):
            sensor = A342(
                port="/dev/AAA",
                baud=Constants.BAUD_RATE_460800,
                product_id="A352XXX",
                logger_id="hoge",
                file_rotate_min=1,
                mode=Constants.MODE_RMS,
                physical=Constants.PHYSICAL_VELOCITY,
                rms_pp_interval=rms_pp_interval,
                sensor_data_diag=True,
                sensor_data_diag_sec=0.1,
            )
            assert sensor.smpl_ctl_command == expect

    class RecordPer:
        @pytest.mark.parametrize(
            "mode, physical, rms_pp_interval, file_rotate_min",
            [
                (Constants.MODE_RAW, Constants.PHYSICAL_VELOCITY, 1, 1),
                (Constants.MODE_RAW, Constants.PHYSICAL_DISPLACEMENT, 1, 1),
                *[
                    (
                        Constants.MODE_RMS,
                        Constants.PHYSICAL_VELOCITY,
                        rms_pp_interval,
                        file_rotate_min,
                    )
                    for rms_pp_interval in range(1, 256)
                    for file_rotate_min in range(1, 11)
                ],
                *[
                    (
                        Constants.MODE_RMS,
                        Constants.PHYSICAL_DISPLACEMENT,
                        rms_pp_interval,
                        file_rotate_min,
                    )
                    for rms_pp_interval in range(1, 256)
                    for file_rotate_min in range(1, 11)
                ],
            ],
        )
        def test_record_per_file_should_be_divided_by_record_per_sec(
            self, mode, physical, rms_pp_interval, file_rotate_min
        ):
            sensor = A342(
                port="/dev/AAA",
                baud=Constants.BAUD_RATE_460800,
                product_id="A352XXX",
                logger_id="hoge",
                file_rotate_min=file_rotate_min,
                mode=mode,
                physical=physical,
                rms_pp_interval=rms_pp_interval,
                sensor_data_diag=True,
                sensor_data_diag_sec=0.1,
            )
            record_per_file = sensor._get_record_per_file()
            record_per_sec = sensor._get_record_per_sec()
            assert record_per_file % record_per_sec == 0

    class TestToReaderArgs:
        def test_to_reader_args(self):
            sensor = A342(
                port="/dev/AAA",
                baud=Constants.BAUD_RATE_460800,
                product_id="A352XXX",
                logger_id="hoge",
                file_rotate_min=1,
                mode=Constants.MODE_RAW,
                physical=Constants.PHYSICAL_VELOCITY,
                rms_pp_interval=1,
                sensor_data_diag=True,
                sensor_data_diag_sec=0.1,
            )
            sensor.serial = "test_serial"
            expect = A342ReaderArgs(
                model="A342",
                serial="test_serial",
                port="/dev/AAA",
                baud=Constants.BAUD_RATE_460800,
                record_size=13,
                record_per_sec=3000,
                start_command=[
                    [0, 0xFF, 0xFF, 0x0D],  # Recovery magic 3
                    [0, 0xFF, 0xFF, 0x0D],
                    [0, 0xFF, 0xFF, 0x0D],
                    [0, 0xFE, 0x00, 0x0D],  # WINDOW_ID(L) write command.(WINDOW=0)
                    [0, 0x83, 0x01, 0x0D],  # Automatic sampling mode
                    WaitCommand(0.18),
                ],
                end_command=[
                    [0, 0xFE, 0x00, 0x0D],  # Window 0
                    [0, 0x83, 0x02, 0x0D],  # MODE_CTRL: 1-0:10=ToConfig
                    WaitCommand(0.01),  # Wait (undefined, follow to A342)
                ],
                record_begin=0x80,
                record_end=0x0D,
                read_length=4096,
                timeout=Comm.DEFAULT_TIMEOUT,
                count_diff=1,
                count_max=4,
                count_start=1,
                sensor_data_diag=True,
                diag_broken_count=300,
            )
            assert sensor.to_reader_args() == expect

        def test_to_reader_args_pp_velo(self):
            rms_pp_interval = 1
            sensor_data_diag_sec = 0.1
            sensor = A342(
                port="/dev/AAA",
                baud=Constants.BAUD_RATE_460800,
                product_id="A352XXX",
                logger_id="hoge",
                file_rotate_min=1,
                mode=Constants.MODE_PP,
                physical=Constants.PHYSICAL_VELOCITY,
                rms_pp_interval=rms_pp_interval,
                sensor_data_diag=True,
                sensor_data_diag_sec=sensor_data_diag_sec,
            )
            args = sensor.to_reader_args()
            assert args.diag_broken_count == math.ceil(
                sensor_data_diag_sec * 10 / rms_pp_interval
            )

        def test_to_reader_args_pp_disp(self):
            rms_pp_interval = 1
            sensor_data_diag_sec = 0.1
            sensor = A342(
                port="/dev/AAA",
                baud=Constants.BAUD_RATE_460800,
                product_id="A352XXX",
                logger_id="hoge",
                file_rotate_min=1,
                mode=Constants.MODE_PP,
                physical=Constants.PHYSICAL_DISPLACEMENT,
                rms_pp_interval=rms_pp_interval,
                sensor_data_diag=True,
                sensor_data_diag_sec=sensor_data_diag_sec,
            )
            args = sensor.to_reader_args()
            assert args.diag_broken_count == math.ceil(
                sensor_data_diag_sec * 1 / rms_pp_interval
            )

    class TestToWriterArgs:
        def test_to_writer_args(self):
            sensor = A342(
                port="/dev/AAA",
                baud=Constants.BAUD_RATE_460800,
                product_id="A352XXX",
                logger_id="hoge",
                file_rotate_min=5,
                mode=Constants.MODE_RAW,
                physical=Constants.PHYSICAL_VELOCITY,
                rms_pp_interval=1,
                sensor_data_diag=True,
                sensor_data_diag_sec=0.1,
            )
            sensor.serial = "test_serial"
            expect = WriterArgs(
                model="A342",
                serial="test_serial",
                port="/dev/AAA",
                logger_id="hoge",
                record_per_file=3000 * 60 * 5,
            )
            assert sensor.to_writer_args() == expect

    class TestAddInfo:
        KEYS: list[str] = TestA352.TestAddInfo.KEYS[0:-1] + ["RMS_PP_INTERVAL"]

        def test_add_part_of_info_keys(self):
            sensor = A342(
                port="/dev/AAA",
                baud=Constants.BAUD_RATE_460800,
                product_id="A352XXX",
                logger_id="hoge",
                file_rotate_min=5,
                mode=Constants.MODE_RAW,
                physical=Constants.PHYSICAL_VELOCITY,
                rms_pp_interval=1,
                sensor_data_diag=True,
                sensor_data_diag_sec=0.1,
            )
            sensor.serial = "test_serial"

            # add_info
            info_d: dict = {}
            sensor.add_info(info_d)

            assert len(info_d) == len(self.KEYS)
            assert list(info_d.keys()) == self.KEYS
            # Value checks are performed in the Controller tests

    class TestSelfTest:
        _void = [0x00]
        _cont = [0x02, 0xFF, 0x0D]
        _done = [0x02, 0x00, 0x0D]
        _diag1_0 = [0x04, 0x00, 0x00, 0x0D]  # DIAG_STAT1 0
        _diag2_0 = [0x0C, 0x00, 0x00, 0x0D]  # DIAG_STAT2 0

        _a342 = A342(
            port="/dev/AAA",
            baud=Constants.BAUD_RATE_460800,
            product_id="A352XXX",
            logger_id="hoge",
            file_rotate_min=5,
            mode=Constants.MODE_RAW,
            physical=Constants.PHYSICAL_VELOCITY,
            rms_pp_interval=1,
            sensor_data_diag=True,
            sensor_data_diag_sec=0.1,
        )

        def _ok_result(self) -> SelfTestResult:
            return SelfTestResult(
                model=self._a342.model,
                serial=self._a342.serial,
                acceleration=Constants.TEST_OK,
                sensitivity=Constants.TEST_NOT_AVAILABLE,
                temperature=Constants.TEST_OK,
                power_voltage=Constants.TEST_OK,
                flash_memory=Constants.TEST_OK,
                structural_resonance="X:{},Y:{},Z:{}".format(
                    Constants.TEST_OK, Constants.TEST_OK, Constants.TEST_OK
                ),
            )

        def test_self_test_ok(self, mock_comm):
            send_sidefx = [
                self._void,  # 0b00000111 : acceleration, temperature, power voltage
                self._cont,
                self._done,
                self._diag1_0,
                self._void,  # 0b00001000 : non-volatile memory
                self._cont,
                self._done,
                self._diag1_0,
                self._void,  # 0b10000000 : structural resonance
                self._cont,
                self._done,
                self._diag1_0,
                self._diag2_0,
            ]
            mock_comm(send_sidefx=send_sidefx)

            # Execute
            result = self._a342.self_test()

            # Judge
            assert result == self._ok_result()

        def test_self_test_ng_ptn1(self, mock_comm):
            send_sidefx = [
                self._void,  # 0b00000111 : acceleration, temperature, power voltage
                self._done,
                [0x04, 0x00, 0b00000010, 0x0D],  # DIAG_STAT1 ACC_ERR_ALL=1
                self._void,  # 0b00001000 : non-volatile memory
                self._done,
                [0x04, 0x00, 0b00000100, 0x0D],  # DIAG_STAT1 FLASH_ERR=1
                self._void,  # 0b10000000 : structural resonance
                self._done,
                self._diag1_0,
                [0x0C, 0b00000001, 0x00, 0x0D],  # DIAG_STAT2 X_EXI_LEVEL=01
            ]
            mock_comm(send_sidefx=send_sidefx)

            # Execute
            result = self._a342.self_test()

            # Judge
            expect = self._ok_result()
            expect.acceleration = Constants.TEST_NG
            expect.flash_memory = Constants.TEST_NG
            expect.structural_resonance = "X:{},Y:{},Z:{}".format(
                Constants.TEST_NG,
                Constants.TEST_OK,
                Constants.TEST_OK,
            )

            assert result == expect

        def test_self_test_ng_ptn2(self, mock_comm):
            send_sidefx = [
                self._void,  # 0b00000111 : acceleration, temperature, power voltage
                self._done,
                [0x04, 0b00000010, 0x00, 0x0D],  # DIAG_STAT1 TEMP_ERR=1
                self._void,  # 0b00001000 : non-volatile memory
                self._done,
                self._diag1_0,
                self._void,  # 0b10000000 : structural resonance
                self._done,
                self._diag1_0,
                [0x0C, 0b00000100, 0x00, 0x0D],  # DIAG_STAT2 Y_EXI_LEVEL=01
            ]
            mock_comm(send_sidefx=send_sidefx)

            # Execute
            result = self._a342.self_test()

            # Judge
            expect = self._ok_result()
            expect.temperature = Constants.TEST_NG
            expect.structural_resonance = "X:{},Y:{},Z:{}".format(
                Constants.TEST_OK,
                Constants.TEST_NG,
                Constants.TEST_OK,
            )

            assert result == expect

        def test_self_test_ng_ptn3(self, mock_comm):
            send_sidefx = [
                self._void,  # 0b00000111 : acceleration, temperature, power voltage
                self._done,
                [0x04, 0b00000001, 0x00, 0x0D],  # DIAG_STAT1 VDD_ERR=1
                self._void,  # 0b00001000 : non-volatile memory
                self._done,
                self._diag1_0,
                self._void,  # 0b10000000 : structural resonance
                self._done,
                self._diag1_0,
                [0x0C, 0b00010000, 0x00, 0x0D],  # DIAG_STAT2 Z_EXI_LEVEL=01
            ]
            mock_comm(send_sidefx=send_sidefx)

            # Execute
            result = self._a342.self_test()

            # Judge
            expect = self._ok_result()
            expect.power_voltage = Constants.TEST_NG
            expect.structural_resonance = "X:{},Y:{},Z:{}".format(
                Constants.TEST_OK,
                Constants.TEST_OK,
                Constants.TEST_NG,
            )

            assert result == expect
