# Copyright 2024, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

import datetime
import os
import time

import pytest

from logger import Constants
from logger.core import Config, LoggerError
from logger.core.factory import LoggerFactoryImpl
from logger.measure import A342, A352, Controller


class TestController:
    config = Config(
        output_path="/tmp",
        logger_id="RP1",
        initial_wait=0,
        measure_time_sec=10,
        file_rotate_min=1,
        baud_rate=Constants.BAUD_RATE_460800,
        A342_physical=Constants.PHYSICAL_VELOCITY,
        A342_mode=Constants.MODE_RAW,
        A342_rmspp_output_interval=1,
        A352_sps=Constants.SPS_1000,
        A370_sps=Constants.SPS_1000,
        sensor_data_diag=True,
        sensor_data_diag_sec=0.1,
        message_send=True,
        message_host="localhost",
        message_port=1883
    )
    factory = LoggerFactoryImpl()

    @pytest.mark.usefixtures("mock_controller_list_comports")
    class TestInit:
        def test_success(self, mock_sensor, mock_A352):
            mock_sensor(load_product_id_return="A352xxx")
            mock_A352(ready_return=True)
            TestController.factory.set_config(TestController.config)
            controller = Controller(
                config=TestController.config,
                start_type=Constants.START_TYPE_AUTO,
                factory=TestController.factory,
            )
            try:
                controller.init()
            except Exception:
                pytest.fail("unexpected error")

        def test_failed_when_unknown_product_id(self, mock_sensor, mock_A352):
            mock_sensor(load_product_id_return="XXXXXXX")
            mock_A352(ready_return=True)
            TestController.factory.set_config(TestController.config)
            controller = Controller(
                config=TestController.config,
                start_type=Constants.START_TYPE_AUTO,
                factory=TestController.factory,
            )
            with pytest.raises(ValueError) as e:
                controller.init()
            assert str(e.value) == "Unknown PRODUCT_ID: XXXXXXX"

        def test_failed_when_sensor_is_not_ready(self, mock_sensor, mock_A352):
            mock_sensor(load_product_id_return="A352xxx")
            mock_A352(ready_return=False)
            TestController.factory.set_config(TestController.config)
            controller = Controller(
                config=TestController.config,
                start_type=Constants.START_TYPE_AUTO,
                factory=TestController.factory,
            )
            with pytest.raises(LoggerError) as e:
                controller.init()
            assert str(e.value) == "No Ready Sensor found"

    class TestOutputInfo:
        def test_success_with_start_time_and_end_time_defined(self, tmpdir, mocker):
            mocker.patch("platform.platform", return_value="test_platform")
            os.environ["TZ"] = "Asia/Tokyo"
            time.tzset()
            TestController.factory.set_config(TestController.config)

            # Set properties for testing
            controller = Controller(
                config=TestController.config,
                start_type=Constants.START_TYPE_AUTO,
                factory=TestController.factory,
            )
            controller.config.output_path = tmpdir
            controller._start_time = datetime.datetime(2024, 1, 1, 0, 0, 0)
            controller._end_time = datetime.datetime(2024, 1, 1, 0, 1, 0)

            out_dir: str = controller.get_output_dir()  # type: ignore
            os.makedirs(out_dir, exist_ok=True)
            sensor = A352(
                port="/dev/AAA",
                baud=Constants.BAUD_RATE_460800,
                product_id="A352XXXXX",
                sps=1000,
                logger_id="RP1",
                file_rotate_min=1,
                sensor_data_diag=True,
                sensor_data_diag_sec=0.1,
            )
            sensor.serial = "serial"
            sensor.firm_ver = 129
            controller._sensors = [sensor]

            # Output
            controller._output_info()
            expect = (
                "OS,test_platform\r\n"
                + "LOGGER_VERSION,1.3.0\r\n"
                + "LOGGER_ID,RP1\r\n"
                + "DATE,2024/01/01 00:00:00\r\n"
                + "TIME_ZONE,UTC+0900\r\n"
                + "START_TYPE,auto\r\n"
                + "MEASURE_TIME_SEC,60\r\n"
                + "INITIAL_WAIT,0\r\n"
                + "FILE_ROTATE_MIN,1\r\n"
                + "BAUD_RATE,460800\r\n"
                + "SENSOR_DATA_DIAG,True\r\n"
                + "SENSOR_DATA_DIAG_SEC,0.100000\r\n"
                + "MESSAGE_SEND,True\r\n"
                + "MESSAGE_HOST,localhost\r\n"
                + "MESSAGE_PORT,1883\r\n"
                + "PORT,/dev/AAA\r\n"
                + "SENSOR,A352\r\n"
                + "PRODUCT_ID,A352XXXXX\r\n"
                + "SERIAL_NO,serial\r\n"
                + "FIRM_VER,0x0081\r\n"
                + "PHYSICAL,Acceleration\r\n"
                + "OUTPUT_TYPE,Raw\r\n"
                + "SPS,1000\r\n"
                + "FILTER,FIR Kaiser TAP512 fc=460\r\n"
            )
            with open(
                f"{tmpdir}/20240101_000000/A352_serial_info.csv", newline="\r\n"
            ) as f:
                actual = f.read()
                assert actual == expect

        def test_success_with_end_time_not_defined(self, tmpdir, mocker):
            mocker.patch("platform.platform", return_value="test_platform")
            os.environ["TZ"] = "Asia/Tokyo"
            time.tzset()

            # Set properties for testing
            TestController.config.sensor_data_diag = False
            TestController.config.sensor_data_diag_sec = 3
            TestController.factory.set_config(TestController.config)
            controller = Controller(
                config=TestController.config,
                start_type=Constants.START_TYPE_AUTO,
                factory=TestController.factory,
            )
            controller.config.output_path = tmpdir
            controller._start_time = datetime.datetime(2024, 1, 1, 0, 0, 0)

            out_dir: str = controller.get_output_dir()  # type: ignore
            os.makedirs(out_dir, exist_ok=True)
            sensor = A342(
                port="/dev/AAA",
                baud=Constants.BAUD_RATE_460800,
                product_id="A342XXXXX",
                logger_id="RP1",
                file_rotate_min=1,
                mode=Constants.MODE_RMS,
                physical=Constants.PHYSICAL_DISPLACEMENT,
                rms_pp_interval=1,
                sensor_data_diag=False,
                sensor_data_diag_sec=3,
            )
            sensor.serial = "serial"
            sensor.firm_ver = 640
            controller._sensors = [sensor]

            # Output
            controller._output_info()
            expect = (
                "OS,test_platform\r\n"
                + "LOGGER_VERSION,1.3.0\r\n"
                + "LOGGER_ID,RP1\r\n"
                + "DATE,2024/01/01 00:00:00\r\n"
                + "TIME_ZONE,UTC+0900\r\n"
                + "START_TYPE,auto\r\n"
                + "MEASURE_TIME_SEC,\r\n"
                + "INITIAL_WAIT,0\r\n"
                + "FILE_ROTATE_MIN,1\r\n"
                + "BAUD_RATE,460800\r\n"
                + "SENSOR_DATA_DIAG,False\r\n"
                + "SENSOR_DATA_DIAG_SEC,3.000000\r\n"
                + "MESSAGE_SEND,True\r\n"
                + "MESSAGE_HOST,localhost\r\n"
                + "MESSAGE_PORT,1883\r\n"
                + "PORT,/dev/AAA\r\n"
                + "SENSOR,A342\r\n"
                + "PRODUCT_ID,A342XXXXX\r\n"
                + "SERIAL_NO,serial\r\n"
                + "FIRM_VER,0x0280\r\n"
                + "PHYSICAL,Displacement\r\n"
                + "OUTPUT_TYPE,RMS\r\n"
                + "SPS,300\r\n"
                + "RMS_PP_INTERVAL,1\r\n"
            )
            with open(
                f"{tmpdir}/20240101_000000/A342_serial_info.csv", newline="\r\n"
            ) as f:
                actual = f.read()
                assert actual == expect

    class TestOutputDir:
        def test_output_dir_without_existing_folder(self, tmpdir):
            TestController.factory.set_config(TestController.config)
            controller = Controller(
                config=TestController.config,
                start_type=Constants.START_TYPE_AUTO,
                factory=TestController.factory,
            )
            controller.config.output_path = tmpdir
            controller._start_time = datetime.datetime(2024, 1, 1, 0, 0, 0)

            outdir = f"{tmpdir}/20240101_000000"
            assert not os.path.exists(outdir)

            assert controller._output_dir is None
            assert controller.get_output_dir() == outdir
            assert controller._output_dir == outdir

        def test_output_dir_with_one_existing_folder(self, tmpdir):
            TestController.factory.set_config(TestController.config)
            controller = Controller(
                config=TestController.config,
                start_type=Constants.START_TYPE_AUTO,
                factory=TestController.factory,
            )
            controller.config.output_path = tmpdir
            controller._start_time = datetime.datetime(2024, 1, 1, 0, 0, 0)

            outdir = f"{tmpdir}/20240101_000000"
            assert not os.path.exists(outdir)
            os.makedirs(outdir)
            assert os.path.exists(outdir)

            assert controller._output_dir is None
            newdir = controller.get_output_dir()
            assert newdir != outdir
            assert newdir == f"{outdir}_1"
            assert controller._output_dir == newdir

        def test_output_dir_with_two_existing_folders(self, tmpdir):
            TestController.factory.set_config(TestController.config)
            controller = Controller(
                config=TestController.config,
                start_type=Constants.START_TYPE_AUTO,
                factory=TestController.factory,
            )
            controller.config.output_path = tmpdir
            controller._start_time = datetime.datetime(2024, 1, 1, 0, 0, 0)

            outdir = f"{tmpdir}/20240101_000000"
            assert not os.path.exists(outdir)
            os.makedirs(outdir)
            os.makedirs(outdir + "_1")

            assert controller._output_dir is None
            newdir = controller.get_output_dir()
            assert newdir != outdir
            assert newdir == f"{outdir}_2"
            assert controller._output_dir == newdir
